/**
 * JASMINe VMMapi: JASMINe Virtual Machine Management API
 * Copyright (C) 2009 France Telecom R&D
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id$
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.vmm.agent.driver.xen;

import java.util.UUID;

import org.apache.log4j.Logger;
import org.ow2.jasmine.vmm.agent.driver.util.RemoteExec;
import org.ow2.jasmine.vmm.api.VMConfigSpec;

/**
 * The XenSSH class gathers static methods wrapping SSH remote invocation to
 * shell scripts that must be installed on Xen backend hosts.
 */
public class XenSSH {
    static Logger logger = Logger.getLogger(XenSSH.class);

    /**
     * Creates a new VM
     * 
     * @param host target host
     * @param vmName VM label
     * @param vmSpec VM parameters
     * @param templateFileName absolute path of the template image
     * @return SSH result
     * @throws RemoteExec.SshException raised if the SSH connection cannot be
     *         established
     */
    public static RemoteExec.Result createVM(final XenHost host, final String vmName, final VMConfigSpec vmSpec,
        final String templateFileName) throws RemoteExec.SshException {
        String command = "createVM -d " + vmSpec.getDiskSizeMB() + " -t " + templateFileName + " -m "
            + vmSpec.getMemorySizeMB() + " -c " + vmSpec.getNumVCPU() + " -u " + UUID.randomUUID().toString() + " " + vmName;
        XenSSH.logger.debug("launching command " + command);
        return RemoteExec.commandAsRoot(host.getHostName(), host.getSshAuthInfo(), command);
    }

    /**
     * Suspends a VM
     * 
     * @param host host containing the VM
     * @param vmName label of the VM to suspend
     * @return SSH result
     * @throws RemoteExec.SshException raised if the SSH connection cannot be
     *         established
     */
    public static RemoteExec.Result suspendVM(final XenHost host, final String vmName) throws RemoteExec.SshException {
        String command = "suspendVM " + vmName;
        XenSSH.logger.debug("Launching command " + command);
        return RemoteExec.commandAsRoot(host.getHostName(), host.getSshAuthInfo(), command);
    }

    /**
     * Resumes a suspended VM
     * 
     * @param host host containing the VM
     * @param vmName label of the VM to resume
     * @return SSH result
     * @throws RemoteExec.SshException raised if the SSH connection cannot be
     *         established
     */
    public static RemoteExec.Result resumeVM(final XenHost host, final String vmName) throws RemoteExec.SshException {
        String command = "resumeVM " + vmName;
        XenSSH.logger.debug("Launching command " + command);
        return RemoteExec.commandAsRoot(host.getHostName(), host.getSshAuthInfo(), command);
    }

    /**
     * Starts a stopped VM
     * 
     * @param host host containing the VM
     * @param vmName label of the VM to start
     * @return SSH result
     * @throws RemoteExec.SshException raised if the SSH connection cannot be
     *         established
     */
    public static RemoteExec.Result startVM(final XenHost host, final String vmName) throws RemoteExec.SshException {
        String command = "startVM " + vmName;
        XenSSH.logger.debug("Launching command " + command);
        return RemoteExec.commandAsRoot(host.getHostName(), host.getSshAuthInfo(), command);
    }

    /**
     * Shutdowns a VM
     * 
     * @param host host containing the VM
     * @param vmName label of the VM to shutdown
     * @return SSH result
     * @throws RemoteExec.SshException raised if the SSH connection cannot be
     *         established
     */
    public static RemoteExec.Result shutdownVM(final XenHost host, final String vmName) throws RemoteExec.SshException {
        String command = "shutdownVM " + vmName;
        XenSSH.logger.debug("Launching command " + command);
        return RemoteExec.commandAsRoot(host.getHostName(), host.getSshAuthInfo(), command);
    }

    /**
     * Destroys a VM
     * 
     * @param host host containing the VM
     * @param vmName label of the VM to destroy
     * @return SSH result
     * @throws RemoteExec.SshException raised if the SSH connection cannot be
     *         established
     */
    public static RemoteExec.Result destroyVM(final XenHost host, final String vmName) throws RemoteExec.SshException {
        String command = "destroyVM " + vmName;
        XenSSH.logger.debug("Launching command " + command);
        return RemoteExec.commandAsRoot(host.getHostName(), host.getSshAuthInfo(), command);
    }

    /**
     * Migrates a VM
     * 
     * @param sourceHost host containing the VM to migrate
     * @param vmName label of the VM to migrate
     * @param targetHostName host to which the VM will migrate
     * @return SSH result
     * @throws RemoteExec.SshException raised if the SSH connection cannot be
     *         established
     */
    public static RemoteExec.Result migrateVM(final XenHost sourceHost, final String vmName, final String targetHostName)
        throws RemoteExec.SshException {
        String command = "migrateVM " + vmName + " " + targetHostName;
        XenSSH.logger.debug("Launching command " + command);
        return RemoteExec.commandAsRoot(sourceHost.getHostName(), sourceHost.getSshAuthInfo(), command);
    }

    /**
     * Performs management operations after a successful VM migration
     * 
     * @param sourceHost host containing the VM which has migrated
     * @param vmName label of the migrated VM
     * @param targetHostName host which has received the migrated VM
     * @return SSH result
     * @throws RemoteExec.SshException raised if the SSH connection cannot be
     *         established
     */
    public static RemoteExec.Result postMigrateVM(final XenHost sourceHost, final String vmName, final String targetHostName)
        throws RemoteExec.SshException {
        String command = "postMigrateVM " + vmName + " " + targetHostName;
        XenSSH.logger.debug("Launching command " + command);
        return RemoteExec.commandAsRoot(sourceHost.getHostName(), sourceHost.getSshAuthInfo(), command);
    }

}
