/**
 * JASMINe VMMapi: JASMINe Virtual Machine Management API
 * Copyright (C) 2009 France Telecom R&D
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: DummyVirtualMachine.java 3177 2009-03-20 13:30:34Z alitokmen $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.vmm.agent.driver.dummy;

import java.net.InetAddress;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;

import javax.management.MBeanNotificationInfo;
import javax.management.Notification;
import javax.management.ObjectName;

import org.apache.log4j.Logger;
import org.ow2.jasmine.vmm.agent.domain.ManagedResource;
import org.ow2.jasmine.vmm.api.BadVMPowerStateException;
import org.ow2.jasmine.vmm.api.HostMXBean;
import org.ow2.jasmine.vmm.api.IllegalOperationException;
import org.ow2.jasmine.vmm.api.InsufficientResourcesException;
import org.ow2.jasmine.vmm.api.NotificationTypes;
import org.ow2.jasmine.vmm.api.VMMException;
import org.ow2.jasmine.vmm.api.VirtualMachineMXBean;

/**
 * Dummy driver VirtualMachine MXBean implementation
 */
class DummyVirtualMachine extends ManagedResource implements VirtualMachineMXBean {
    static Logger logger = Logger.getLogger(DummyVirtualMachine.class);

    private String name;

    private DummyHost host;

    private String uuid;

    private String ipAddress;

    private int numVCPU;

    private int schedulingCap, schedulingWeight;

    private int diskSizeMB;

    private long memorySizeMB, memoryUsedMB;

    private PowerState status = PowerState.RUNNING;

    private Date startTime;

    public DummyVirtualMachine(final String name, final String uuid, final ObjectName objectName, final DummyHost host) {
        super(objectName);
        this.name = name;
        this.host = host;
        this.uuid = uuid;
        memorySizeMB = 1024;

        startTime = new Date(System.currentTimeMillis());

        try {
            ipAddress = InetAddress.getByName("198.7.9.66").getHostAddress();
        } catch (Exception ex) {
            logger.error("Failed to get IP address", ex);
        }
    }

    public Date getStartTime() {
        return startTime;
    }

    public long getUpTimeSeconds() {
        return (System.currentTimeMillis() - startTime.getTime()) / 1000;
    }

    public int getNumVCPUs() {
        return numVCPU;
    }

    public void setNumVCPUs(final int numVCPU) {
        this.numVCPU = numVCPU;
    }

    public int getDiskSizeMB() {
        return diskSizeMB;
    }

    public void setDiskSizeMB(final int diskSizeMB) {
        this.diskSizeMB = diskSizeMB;
    }

    public long getMemorySizeMB() {
        return memorySizeMB;
    }

    public long getMemoryUsedMB() {
        return memoryUsedMB;
    }

    public void setMemorySizeMB(final long memorySizeMB) {
        this.memorySizeMB = memorySizeMB;
        memoryUsedMB = memorySizeMB / 2;
    }

    Map<String, String> userData = new HashMap<String, String>();

    public void addUserData(final String key, final String value) {
        userData.put(key, value);
    }

    public String getUserData(final String key) {
        return userData.get(key);
    }

    public void onVMStatusChanged() {
        logger.info("VM " + getNameLabel() + " state=" + getState());
        emitNotification(NotificationTypes.VM_STATE_CHANGE, getState().toString(), null);
    }

    public boolean canLiveMigrateToHost(final HostMXBean targetHost) {
        return targetHost != host && getMemorySizeMB() <= targetHost.getFreeMemoryMB();
    }

    public void destroy() {
        host.onDestroyVM(this);
    }

    public float[] getLoadPerVCPU() {
        return new float[] {8, 10};
    }

    public float getCPULoad() {
        return 0;
    }

    public HostMXBean getHostMBean() {
        return host;
    }

    public String getGuestIpAddress() {
        return ipAddress;
    }

    public String getMacAddress() {
        return "8E:10:65:AB:CE:26";
    }

    public int getMemory() {
        return 0;
    }

    public String getNameLabel() {
        return name;
    }

    public String getPinnedVCPUs() {
        return null;
    }

    public int getSchedulingCap() {
        return schedulingCap;
    }

    public int getSchedulingWeight() {
        return schedulingWeight;
    }

    public PowerState getState() {
        return status;
    }

    public String getUuid() {
        return uuid;
    }

    public void reboot() {
    }

    public void resume() {
        if (status == PowerState.SUSPENDED) {
            status = PowerState.RUNNING;
            onVMStatusChanged();
            logger.info("VM " + name + " resumed");
        }
    }

    public void setMacAddress(final String mac) {
    }

    public void setPinnedVCPUs(final String pinnedVCPUs) {
    }

    public void setSchedulingCap(final int schedulingCap) {
        this.schedulingCap = schedulingCap;
    }

    public void setSchedulingWeight(final int schedulingWeight) {
        this.schedulingWeight = schedulingWeight;
    }

    public void shutdown() {
        if (status == PowerState.RUNNING) {
            status = PowerState.HALTED;
            onVMStatusChanged();
            logger.info("VM " + name + " shutdowned");
        }
    }

    public void start() {
        if (status == PowerState.HALTED) {
            status = PowerState.RUNNING;
            onVMStatusChanged();
            logger.info("VM " + name + " started");
        }
    }

    public void suspend() throws BadVMPowerStateException {
        if (status == PowerState.RUNNING) {
            status = PowerState.SUSPENDED;
            onVMStatusChanged();
            logger.info("VM " + name + " suspended");
        }
    }

    int count = 0;

    public void migrate(HostMXBean targetHost, final boolean live) throws IllegalOperationException, VMMException {
        String targetHostName = targetHost.getHostName();
        boolean foundHostInServerPool = false;
        // TODO need to found out how to check equality between MXBean whether
        // proxy or local
        for (HostMXBean h : host.getServerPool().getManagedHosts()) {
            if (h.getHostName().equals(targetHost.getHostName())) {
                foundHostInServerPool = true;
                targetHost = h;
                break;
            }
        }

        if (!foundHostInServerPool) {
            throw new IllegalOperationException("Source and target hosts belong to different server pools");
        }

        emitNotification(NotificationTypes.VM_MIGRATION_START, targetHostName, uuid);
        try {
            Thread.sleep(1000);
        } catch (Exception ex) {
        }

        DummyHost target = (DummyHost) targetHost;
        host.postMigrateVM(this, target);
        host = target;

        emitNotification(NotificationTypes.VM_MIGRATION, targetHostName, uuid);
    }

    @Override
    public MBeanNotificationInfo[] getNotificationInfo() {
        return new MBeanNotificationInfo[] {new MBeanNotificationInfo(new String[] {NotificationTypes.VM_STATE_CHANGE,
            NotificationTypes.VM_MIGRATION, NotificationTypes.LOG, NotificationTypes.ERROR}, Notification.class.getName(),
            "VM event")};
    }

    public void makeTemplate(final String vmImageID, final String name, final String description) throws InsufficientResourcesException,
        IllegalOperationException, VMMException {
        // TODO

    }
}
