/**
 * JASMINe VMMapi: JASMINe Virtual Machine Management API
 * Copyright (C) 2009 France Telecom R&D
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: VMwareServerPool.java 3417 2009-04-15 17:12:17Z dangtran $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.vmm.agent.driver.vmware;

import java.util.ArrayList;
import java.util.Calendar;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

import javax.management.ObjectName;

import org.apache.log4j.Logger;
import org.ow2.jasmine.vmm.agent.domain.ManagedResource;
import org.ow2.jasmine.vmm.agent.domain.ServerPool;
import org.ow2.jasmine.vmm.agent.jmx.MBeanObjectNamer;
import org.ow2.jasmine.vmm.agent.main.AgentCommon;
import org.ow2.jasmine.vmm.api.HostMXBean;
import org.ow2.jasmine.vmm.api.VMMException;
import org.ow2.jasmine.vmm.api.VirtualMachineImageMXBean;
import org.ow2.jasmine.vmm.api.VirtualMachineImageStoreMXBean;

import com.vmware.vim.ArrayOfManagedObjectReference;
import com.vmware.vim.DynamicProperty;
import com.vmware.vim.ManagedObjectReference;
import com.vmware.vim.ObjectContent;

/**
 * VMware driver ServerPool MXBean implementation
 */
class VMwareServerPool extends ServerPool {
    static protected Logger logger = Logger.getLogger(VMwareServerPool.class);

    static private int KEEP_ALIVE_PERIOD_MS = 2 * 60 * 1000;

    private static final int THREADPOOL_SIZE = 20;

    private static final int CONNECTION_DEFAULT_POOL_SIZE = 4;

    private static ExecutorService executorService = Executors.newFixedThreadPool(VMwareServerPool.THREADPOOL_SIZE);

    private ConnectionPool connectionPool;

    private VMwareServiceConnection defaultConnection;

    private HashMap<String, VMwareHost> hostRefs = new HashMap<String, VMwareHost>();

    private String datacenter, vmFolderName;

    private ManagedObjectReference datacenterRef, vmFolderRef, datastoreRef, vmTemplateFolderRef, hostFolderRef;

    private VMwareVMTemplateDataStore imageStore;

    public VMwareServerPool(final String name, final ObjectName objectName, final Properties props) throws VMMException {
        super(name, objectName);
        String hostName = props.getProperty("virtualCenterHostName");
        this.datacenter = props.getProperty("datacenter");
        this.vmFolderName = props.getProperty("vmFolderPath");
        String vmTemplateFolderName = props.getProperty("vmTemplateFolderPath");
        String userName = props.getProperty("user");
        String password = props.getProperty("password");
        String datastoreName = props.getProperty("datastore");

        try {
            this.connectionPool = new ConnectionPool(VMwareServerPool.CONNECTION_DEFAULT_POOL_SIZE, hostName, userName,
                password);
            this.defaultConnection = this.connectionPool.getConnection();

            VMwareServerPool.logger.debug("Finding datacenter " + this.datacenter);
            this.datacenterRef = this.defaultConnection.getService().findByInventoryPath(
                this.defaultConnection.getServiceContent().getSearchIndex(), this.datacenter);
            if (this.datacenterRef == null) {
                String msg = "VMware Driver initialization error: datacenter " + this.datacenter + " not found ";
                VMwareServerPool.logger.error(msg);
                throw new VMMException(msg);
            }

            // Find the virtual machine folder for this folder.
            if (this.vmFolderName != null) {
                this.vmFolderRef = this.defaultConnection.getService().findByInventoryPath(
                    this.defaultConnection.getServiceContent().getSearchIndex(), this.vmFolderName);
            } else {
                this.vmFolderRef = (ManagedObjectReference) this.getDynamicProarray(this.datacenterRef, "vmFolder")[0].getVal();
            }
            if (this.vmFolderRef == null) {
                String msg = "VMware Driver initialization error: virtual machine folder " + this.vmFolderName + " not found";
                VMwareServerPool.logger.error(msg);
                throw new VMMException(msg);
            }

            // Find the virtual machine folder for this folder.
            // if (this.vmFolderName != null) {
            // ArrayOfManagedObjectReference subVmFolderRefs =
            // (ArrayOfManagedObjectReference) this.getDynamicProarray(
            // this.vmFolderRef, "childEntity")[0].getVal();
            // if (subVmFolderRefs == null) {
            // String msg =
            // "VMware Driver initialization error: virtual machine sub folders is not found";
            // VMwareServerPool.logger.error(msg);
            // throw new VMMException(msg);
            // }
            // for (ManagedObjectReference mor :
            // subVmFolderRefs.getManagedObjectReference()) {
            // String folderName = (String) this.getDynamicProarray(mor,
            // "name")[0].getVal();
            // if (folderName.equals(this.vmFolderName)) {
            // this.vmFolderRef = mor;
            // break;
            // }
            // }
            // }

            // Find the virtual machine template folder
            if (vmTemplateFolderName != null) {
                this.vmTemplateFolderRef = this.defaultConnection.getService().findByInventoryPath(
                    this.defaultConnection.getServiceContent().getSearchIndex(), vmTemplateFolderName);
            } else {
                this.vmTemplateFolderRef = this.vmFolderRef;
            }
            if (this.vmTemplateFolderRef == null) {
                String msg = "VMware Driver initialization error: virtual machine template folder " + vmTemplateFolderName
                    + " not found";
                VMwareServerPool.logger.error(msg);
                throw new VMMException(msg);
            }

            // this.vmTemplateFolderRef = this.vmFolderRef;
            // if (vmTemplateFolderName != null) {
            // ArrayOfManagedObjectReference subVmFolderRefs =
            // (ArrayOfManagedObjectReference) this.getDynamicProarray(
            // this.vmFolderRef, "childEntity")[0].getVal();
            // if (subVmFolderRefs == null) {
            // String msg =
            // "VMware Driver initialization error: virtual machine sub folders is not found";
            // VMwareServerPool.logger.error(msg);
            // throw new VMMException(msg);
            // }
            // for (ManagedObjectReference mor :
            // subVmFolderRefs.getManagedObjectReference()) {
            // String folderName = (String) this.getDynamicProarray(mor,
            // "name")[0].getVal();
            // if (folderName.equals(vmTemplateFolderName)) {
            // this.vmTemplateFolderRef = mor;
            // break;
            // }
            // }
            // }

            // Find the host folder for this folder.
            this.hostFolderRef = (ManagedObjectReference) this.getDynamicProarray(this.datacenterRef, "hostFolder")[0].getVal();
            if (this.hostFolderRef == null) {
                String msg = "VMware Driver initialization error: host folder not found for datacenter " + this.datacenter;
                VMwareServerPool.logger.error(msg);
                throw new VMMException(msg);
            }

            // find the datastore
            ArrayOfManagedObjectReference dataStoreRefs = (ArrayOfManagedObjectReference) this.getDynamicProarray(
                this.datacenterRef, "datastore")[0].getVal();
            if (dataStoreRefs != null) {
                for (int i = 0; i < dataStoreRefs.getManagedObjectReference().length; i++) {
                    String dsName = (String) this.getDynamicProarray(dataStoreRefs.getManagedObjectReference(i), "info.name")[0]
                        .getVal();
                    if (dsName.equals(datastoreName)) {
                        this.datastoreRef = dataStoreRefs.getManagedObjectReference(i);
                        break;
                    }
                }
            }
            if (this.datastoreRef == null) {
                String msg = "VMware Driver initialization error: Datastore " + datastoreName + " cannot be found";
                VMwareServerPool.logger.error(msg);
                throw new VMMException(msg);
            }

            String vmImageStoreName = this.datacenter + "-VMIStore"; // TODO
            this.imageStore = new VMwareVMTemplateDataStore(MBeanObjectNamer.makeVMImageStoreName(vmImageStoreName),
                vmImageStoreName, this.datastoreRef);
            AgentCommon.getMBeanServer().registerMBean(this.imageStore, this.imageStore.getObjectName());

        } catch (Exception ex) {
            String msg = "VMware Driver initialization error: " + ex.getMessage();
            VMwareServerPool.logger.error(msg, ex);
            throw new VMMException(msg);
        }

        new VMwareEventCollector(this, new VMwareServiceConnection(hostName, userName, password), this.datacenter).start();

    }

    public ExecutorService getExecutorService() {
        return VMwareServerPool.executorService;
    }

    public ConnectionPool getConnectionPool() {
        return this.connectionPool;
    }

    @Override
    public VirtualMachineImageStoreMXBean getVMImageStore() {
        return this.imageStore;
    }

    public String getMountPoint() {
        return this.datacenter;
    }

    VMwareHost getHostByName(final String name) {
        return this.hostRefs.get(name);
    }

    @Override
    public HostMXBean newHost(final String hostName, final Map<String, String> props) {
        VMwareHost host = this.hostRefs.get(hostName);
        if (host != null) {
            return host;
        }

        ManagedObjectReference mor = null;
        try {
            synchronized (this.defaultConnection) {
                mor = this.defaultConnection.getDecendentMoRef(this.getHostFolderRef(), "HostSystem", hostName);
            }
        } catch (Exception ex) {
            VMwareServerPool.logger.error("Unable to find HostSystem managed object " + hostName + " in VMware VC inventory",
                ex);
            return null;
        }

        if (mor == null) {
            VMwareServerPool.logger.error("Host " + hostName + " not found");
            return null;
        }

        ObjectName mbeanObjectName = MBeanObjectNamer.makeHostName(this.getPath() + "/" + hostName, hostName);
        host = new VMwareHost(this, mor, mbeanObjectName, hostName);
        host.getResidentVMs();

        try {

            AgentCommon.getMBeanServer().registerMBean(host, mbeanObjectName);
        } catch (Exception ex) {
            VMwareServerPool.logger.error("Unable to register MBean " + mbeanObjectName, ex);
            return null;
        }
        this.hostRefs.put(hostName, host);
        // logger.info("Added host "+hostName+" "+host.getObjectName());
        this.addHost(host);
        try {
            this.connectionPool.ensureCapacity(1 + this.hostRefs.size() * 3); // XXX
        } catch (Exception ex) {
            VMwareServerPool.logger.error("Cannot increase connection pool", ex);
        }
        return host;
    }

    public ManagedObjectReference getDatacenterRef() {
        return this.datacenterRef;
    }

    public ManagedObjectReference getVmFolderRef() {
        return this.vmFolderRef;
    }

    public ManagedObjectReference getHostFolderRef() {
        return this.hostFolderRef;
    }

    public ManagedObjectReference getVmTemplateFolderRef() {
        return this.vmTemplateFolderRef;
    }

    public ManagedObjectReference getDatastoreRef() {
        return this.datastoreRef;
    }

    private DynamicProperty[] getDynamicProarray(final ManagedObjectReference MOR, final String pName) throws Exception {
        ObjectContent[] objContent;
        objContent = this.defaultConnection.getObjectProperties(null, MOR, new String[] {pName});
        ObjectContent contentObj = objContent[0];
        DynamicProperty[] objArr = contentObj.getPropSet();
        return objArr;
    }

    class ConnectionPool {
        private List<VMwareServiceConnection> connections = new ArrayList<VMwareServiceConnection>();

        private String hostName, userName, password;

        ConnectionPool(final int initialCapacity, final String hostName, final String userName, final String password)
            throws Exception {
            this.userName = userName;
            this.hostName = hostName;
            this.password = password;
            this.ensureCapacity(initialCapacity);
            VMwareServerPool.executorService.execute(new Runnable() {
                public void run() {
                    while (true) {
                        try {
                            Thread.sleep(VMwareServerPool.KEEP_ALIVE_PERIOD_MS);
                            synchronized (ConnectionPool.this) {
                                for (VMwareServiceConnection connection : ConnectionPool.this.connections) {
                                    synchronized (connection) {
                                        Calendar time = connection.getService().currentTime(connection.getServiceInstanceRef());
                                    }
                                }
                            }
                        } catch (Exception ex) {
                            VMwareServerPool.logger.error("error", ex);
                        }
                    }
                }
            });
        }

        synchronized void ensureCapacity(final int minCapacity) throws Exception {
            if (this.connections.size() < minCapacity) {
                for (int i = minCapacity - this.connections.size(); i > 0; i--) {
                    VMwareServiceConnection connection = new VMwareServiceConnection(this.hostName, this.userName,
                        this.password);
                    VMwareServerPool.logger.debug("Connecting to VMware VirtualCenter " + this.hostName + " ...");
                    connection.connect();
                    VMwareServerPool.logger.debug("Done");
                    this.connections.add(connection);
                }
            }
        }

        synchronized VMwareServiceConnection getConnection() {
            for (VMwareServiceConnection connection : this.connections) {
                if (connection.lease()) {
                    return connection;
                }
            }
            return null;
        }
    }

    class VMwareVMTemplateDataStore extends ManagedResource implements VirtualMachineImageStoreMXBean {
        private String name;

        private ArrayList<VirtualMachineImageMXBean> imageList = new ArrayList<VirtualMachineImageMXBean>();

        private ManagedObjectReference datastoreRef;

        private boolean needSync = true;

        public VMwareVMTemplateDataStore(final ObjectName objectName, final String name,
            final ManagedObjectReference datastoreRef) {
            super(objectName);
            this.name = name;
            this.datastoreRef = datastoreRef;
            this.listVMImageTemplates();
        }

        public long getCapacityMB() {
            try {
                long capacityInBytes = (Long) VMwareServerPool.this.getDynamicProarray(this.datastoreRef, "summary.capacity")[0]
                    .getVal();
                return capacityInBytes / 1024;
            } catch (Exception ex) {
                VMwareServerPool.logger.error("Failed to get datastore capacity", ex);
                return -1;
            }
        }

        public long getFreeSpaceMB() {
            try {
                long freeSpaceInBytes = (Long) VMwareServerPool.this.getDynamicProarray(this.datastoreRef, "summary.freeSpace")[0]
                    .getVal();
                return freeSpaceInBytes / 1024;
            } catch (Exception ex) {
                VMwareServerPool.logger.error("Failed to get datastore free space", ex);
                return -1;
            }
        }

        public String getName() {
            return this.name;
        }

        public synchronized void sync() {
            this.needSync = true;
        }

        public synchronized List<VirtualMachineImageMXBean> listVMImageTemplates() {
            if (this.needSync) {
                try {
                    ArrayList<ManagedObjectReference> vms = VMwareServerPool.this.defaultConnection.getDecendentMoRefs(
                        VMwareServerPool.this.getVmTemplateFolderRef(), "VirtualMachine");

                    for (ManagedObjectReference vmMor : vms) {
                        boolean isTemplate = (Boolean) VMwareServerPool.this.getDynamicProarray(vmMor,
                            "summary.config.template")[0].getVal();
                        if (isTemplate) {
                            DynamicProperty[] properties = VMwareServerPool.this.getDynamicProarray(vmMor, "config.name");
                            if (properties == null || properties.length < 1) {
                                continue;
                            }
                            String vmName = (String) properties[0].getVal();
                            String uuid = (String) VMwareServerPool.this.getDynamicProarray(vmMor, "config.uuid")[0].getVal();

                            boolean found = false;
                            for (VirtualMachineImageMXBean image : this.imageList) {
                                if (image.getUUID().equals(vmName)) {
                                    found = true;
                                }
                            }
                            if (!found) {
                                ObjectName on = MBeanObjectNamer.makeVMImageName(uuid);
                                VMwareVirtualMachineImage image = new VMwareVirtualMachineImage(on, vmName, uuid);
                                VMwareServerPool.logger.debug("New VMware template: " + image);
                                AgentCommon.getMBeanServer().registerMBean(image, on);
                                this.imageList.add(image);
                            }
                        }

                    }

                } catch (Exception ex) {
                    VMwareServerPool.logger.error("Failed to list VM templates", ex);
                }
                this.needSync = false;
            }
            return this.imageList;
        }

        public synchronized VirtualMachineImageMXBean lookUpByUUID(final String uuid) {
            for (VirtualMachineImageMXBean image : this.listVMImageTemplates()) {
                if (image.getUUID().equals(uuid)) {
                    return image;
                }
            }
            return null;
        }

        public synchronized void removeVMImageTemplate(final VirtualMachineImageMXBean vmImage) throws VMMException {
            String vmImageID = vmImage.getUUID();
            try {
                ManagedObjectReference vmMOR = VMwareServerPool.this.defaultConnection.getDecendentMoRef(null,
                    "VirtualMachine", vmImageID);
                // asynchronous op, forget about result ?
                VMwareServerPool.this.defaultConnection.getService().destroy_Task(vmMOR);
            } catch (Exception ex) {
                VMwareServerPool.logger.error("Failed to retrieve VM reference " + vmImageID, ex);
            }
        }

    }

    private static class VMwareVirtualMachineImage extends ManagedResource implements VirtualMachineImageMXBean {
        private String name, uuid;

        public VMwareVirtualMachineImage(final ObjectName objectName, final String name, final String uuid) {
            super(objectName);
            this.name = name;
            this.uuid = uuid;
        }

        public String getName() {
            return this.name;
        }

        // TODO we ignore the UUID and use the name as a unique
        public String getUUID() {
            return this.name;
        }

        public String getMetaData() {
            return "";
        }

        @Override
        public String toString() {
            return "VMwareTemplate(name=" + this.getName() + ",uuid=" + this.getUUID() + ")";
        }
    }

}
