/**
 * JASMINe VMMapi: JASMINe Virtual Machine Management API
 * Copyright (C) 2009 France Telecom R&D
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: AgentControl.java 3177 2009-03-20 13:30:34Z alitokmen $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.vmm.agent.main;

import java.lang.reflect.Proxy;

import javax.management.JMX;
import javax.management.MBeanServerConnection;
import javax.management.MBeanServerInvocationHandler;
import javax.management.ObjectName;
import javax.management.remote.JMXConnector;
import javax.management.remote.JMXConnectorFactory;
import javax.management.remote.JMXServiceURL;

import org.ow2.jasmine.vmm.api.DomainMXBean;
import org.ow2.jasmine.vmm.api.HostMXBean;
import org.ow2.jasmine.vmm.api.ServerPoolMXBean;
import org.ow2.jasmine.vmm.api.VMMAgentMXBean;
import org.ow2.jasmine.vmm.api.VirtualMachineImageMXBean;
import org.ow2.jasmine.vmm.api.VirtualMachineImageStoreMXBean;
import org.ow2.jasmine.vmm.api.VirtualMachineMXBean;

/**
 * AgentControl provides a basic CLI program to control a local or remote VMM
 * agent It currently accept the following commands: list : list recursively the
 * whole resource MXBean tree of the agent shutdown : asks the agent to shutdown
 */
public class AgentControl {
    private static final int DEFAULT_PORT = 9999;

    private static final String DEFAULT_HOST = "localhost";

    public static void main(final String[] args) {
        int port = DEFAULT_PORT;
        String host = DEFAULT_HOST;

        int i;
        for (i = 0; i < args.length; i++) {
            String s = args[i];
            if (!s.startsWith("-")) {
                break;
            }
            if (s.equals("-port")) {
                if (i < args.length - 1) {
                    port = Integer.parseInt(args[++i]);
                } else {
                    printUsageAndExit();
                }
            } else if (s.equals("-host")) {
                if (i < args.length - 1) {
                    host = args[++i];
                } else {
                    printUsageAndExit();
                }
            }
        }
        if (i != args.length - 1) {
            printUsageAndExit();
        }
        String command = args[i];

        try {
            JMXServiceURL url = new JMXServiceURL("service:jmx:rmi:///jndi/rmi://" + host + ":" + port + "/server");
            JMXConnector jmxc = JMXConnectorFactory.connect(url, null);
            MBeanServerConnection mbsc = jmxc.getMBeanServerConnection();
            ObjectName agentObjectName = new ObjectName("org.ow2.jasmine.vmmapi:type=Agent");
            VMMAgentMXBean agentMBean = JMX.newMXBeanProxy(mbsc, agentObjectName, VMMAgentMXBean.class);

            if (command.equals("list")) {
                DomainMXBean domain = agentMBean.getRootDomain();
                dumpDomain(domain);
            } else if (command.equals("shutdown")) {
                agentMBean.shutdown();
            } else {
                printUsageAndExit();
            }

            jmxc.close();
        } catch (Exception ex) {
            ex.printStackTrace();
        }
    }

    static private void dumpDomain(final DomainMXBean domain) throws Exception {
        System.out.println("Domain " + domain.getName() + "(" + domain.getObjectName() + ")");
        for (ServerPoolMXBean rp : domain.getServerPools()) {
            System.out.println("\tPool " + rp.getName() + " (" + rp.getObjectName() + ")");
            VirtualMachineImageStoreMXBean imageStore = rp.getVMImageStore();
            System.out.println("\t\tVMImageStore " + imageStore.getName());
            for (VirtualMachineImageMXBean image : imageStore.listVMImageTemplates()) {
                System.out.println("\t\t\tImage UUID=" + image.getUUID() + " name=" + image.getName());
            }
            System.out.println("\t\tHosts:");
            for (HostMXBean host : rp.getManagedHosts()) {
                System.out.println("\t\t\tHost " + host.getHostName() + " (" + proxyToObjectName(host) + ")");
                for (VirtualMachineMXBean vm : host.getResidentVMs()) {
                    try {
                        System.out.println("\t\t\t\tVM " + vm.getNameLabel() + " (" + proxyToObjectName(vm) + ")");
                    } catch (Throwable ex) {
                        ex.printStackTrace();
                    }
                }
            }
        }
        for (DomainMXBean subDomain : domain.getSubDomains()) {
            dumpDomain(subDomain);
        }
    }

    static private ObjectName proxyToObjectName(final Object proxy) {
        MBeanServerInvocationHandler ih = (MBeanServerInvocationHandler) Proxy.getInvocationHandler(proxy);
        return ih.getObjectName();
    }

    private static void printUsageAndExit() {
        System.out.println("Usage: java org.ow2.jasmine.vmm.agent.main.AgentControl [-port <port>] [-host <host>] <command>");
        System.out.println("       <command> : shutdown | list");
        System.exit(1);
    }
}
