/**
 * JASMINe VMMapi: JASMINe Virtual Machine Management API
 * Copyright (C) 2009 France Telecom R&D
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: DummyHost.java 3535 2009-04-23 13:07:22Z alitokmen $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.vmm.agent.driver.dummy;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Random;
import java.util.Set;
import java.util.Timer;
import java.util.TimerTask;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

import javax.management.MBeanNotificationInfo;
import javax.management.Notification;
import javax.management.ObjectName;

import org.apache.log4j.Logger;
import org.ow2.jasmine.vmm.agent.domain.ManagedResource;
import org.ow2.jasmine.vmm.agent.jmx.MBeanObjectNamer;
import org.ow2.jasmine.vmm.agent.main.AgentCommon;
import org.ow2.jasmine.vmm.api.HostMXBean;
import org.ow2.jasmine.vmm.api.InsufficientResourcesException;
import org.ow2.jasmine.vmm.api.InvalidVMConfigException;
import org.ow2.jasmine.vmm.api.NotificationTypes;
import org.ow2.jasmine.vmm.api.ServerPoolMXBean;
import org.ow2.jasmine.vmm.api.VMConfigSpec;
import org.ow2.jasmine.vmm.api.VMMException;
import org.ow2.jasmine.vmm.api.VirtualMachineImageStoreMXBean;
import org.ow2.jasmine.vmm.api.VirtualMachineMXBean;

/**
 * Dummy driver Host MXBean implementation
 */
class DummyHost extends ManagedResource implements HostMXBean {
    static Logger logger = Logger.getLogger(DummyHost.class);

    private List<DummyVirtualMachine> vms = new CopyOnWriteArrayList<DummyVirtualMachine>();

    private static ExecutorService executorService = Executors.newFixedThreadPool(3);

    private Timer perfTimer;

    private String hostName;

    private final ObjectName objectName;

    private int numCPU = 4;

    private int freeMemoryMB = 4 * 1024;

    private DummyServerPool pool;

    public DummyHost(final DummyServerPool pool, final String hostName, final ObjectName objectName) {
        super(objectName);
        this.pool = pool;
        this.hostName = hostName;
        this.objectName = objectName;
    }

    public ServerPoolMXBean getServerPool() {
        return this.pool;
    }

    private void addVM(final DummyVirtualMachine vm) {
        this.vms.add(vm);
    }

    private void removeVM(final DummyVirtualMachine vm) {
        this.vms.remove(vm);
    }

    void postMigrateVM(final DummyVirtualMachine vm, final DummyHost newHost) {
        this.removeVM(vm);
        newHost.addVM(vm);
        this.freeMemoryMB += vm.getMemorySizeMB();
        newHost.freeMemoryMB -= vm.getMemorySizeMB();
    }

    public void postDestroyVM(final DummyVirtualMachine vm) {
        this.removeVM(vm);
        this.freeMemoryMB += vm.getMemorySizeMB();
    }

    @Override
    public ObjectName getObjectName() {
        return this.objectName;
    }

    public synchronized void onDestroyVM(final DummyVirtualMachine vm) {
        try {
            AgentCommon.getMBeanServer().unregisterMBean(vm.getObjectName());
        } catch (Exception ex) {
            DummyHost.logger.error("Cannot unregister VM MBean", ex);
        }
        this.removeVM(vm);
        this.freeMemoryMB += vm.getMemorySizeMB();
        this.emitNotification(NotificationTypes.VM_DEL, "Destroyed", vm.getObjectName());
        DummyHost.logger.info("destroyed VM " + vm.getObjectName());
    }

    public VirtualMachineMXBean createVM(final VMConfigSpec vmSpec, final boolean sync) throws InsufficientResourcesException,
        InvalidVMConfigException, VMMException {
        final VMConfigSpec vmSpec2 = new VMConfigSpec(vmSpec);
        // check name uniqueness
        for (DummyVirtualMachine dvm : this.vms) {
            if (dvm.getNameLabel().equals(vmSpec2.getName())) {
                throw new InvalidVMConfigException("VM name already exists");
            }
        }

        if (!sync) {
            DummyHost.executorService.execute(new Runnable() {
                public void run() {
                    try {
                        DummyHost.this.createVM2(vmSpec2);
                    } catch (Exception ex) {
                    }
                };
            });
            return null;
        } else {
            return this.createVM2(vmSpec2);
        }

    }

    private VirtualMachineMXBean createVM2(final VMConfigSpec vmSpec) throws InsufficientResourcesException,
        InvalidVMConfigException, VMMException {
        DummyVirtualMachine vm = null;

        try {
            Thread.sleep(500);
            Random random = new Random(System.currentTimeMillis());
            String uuid = Integer.toString(random.nextInt(10000));
            ObjectName vmName = MBeanObjectNamer.makeVirtualMachineName(
                this.getServerPool().getPath() + "/" + vmSpec.getName(), uuid);
            vm = new DummyVirtualMachine(vmSpec.getName(), uuid, vmName, this);
            this.addVM(vm);
            vm.setDiskSizeMB(vmSpec.getDiskSizeMB());
            vm.setNumVCPUs(vmSpec.getNumVCPU());
            vm.setMemorySizeMB((int) vmSpec.getMemorySizeMB());
            vm.addUserData("imageID", vmSpec.getVmImageUUID());
            AgentCommon.getMBeanServer().registerMBean(vm, vm.getObjectName());
            DummyHost.logger.info("VM Creation: registered VirtualMachineMBean " + vm.getObjectName());
        } catch (Exception ex) {
            DummyHost.logger.error("Failed to create VM", ex);
            return null;
        }
        this.emitNotification(NotificationTypes.VM_ADD, "Created", vm.getObjectName());

        this.freeMemoryMB -= vmSpec.getMemorySizeMB();
        return vm;
    }

    HashMap<String, String> hypervisorInfo;

    public Map<String, String> getHypervisorInfo() {
        if (this.hypervisorInfo == null) {
            this.hypervisorInfo = new HashMap<String, String>();
            this.hypervisorInfo.put("name", "DummyHypervisor");
            this.hypervisorInfo.put("vendor", "");
            this.hypervisorInfo.put("version", "1.0");
        }
        return this.hypervisorInfo;
    }

    HashMap<String, String> cpuInfo;

    public Map<String, String> getCPUInfo() {
        if (this.cpuInfo == null) {
            this.cpuInfo = new HashMap<String, String>();
            this.cpuInfo.put("model", "Core 2 Duo");
            this.cpuInfo.put("speed", "2330");
            this.cpuInfo.put("vendor", "Intel");
        }
        return this.cpuInfo;
    }

    public int getNumCPU() {
        return this.numCPU;
    }

    public float[] getLoadPerCPU() {
        return new float[] {0, 0};
    }

    public float getCPULoad() {
        float l = this.vms.size() * 0.25f;
        return Math.min(1, l);
    }

    public Map<String, Float> getVMCPULoads() {
        HashMap<String, Float> cpuLoads = new HashMap<String, Float>();
        return cpuLoads;
    }

    public long getFreeMemoryMB() {
        return this.freeMemoryMB;
    }

    public String getHostName() {
        return this.hostName;
    }

    public List<VirtualMachineMXBean> getResidentVMs() {
        List<VirtualMachineMXBean> vmList = new ArrayList<VirtualMachineMXBean>();
        for (VirtualMachineMXBean vm : this.vms) {
            vmList.add(vm);
        }
        return vmList;
    }

    public long getTotalMemoryMB() {
        // TODO Auto-generated method stub
        return 4096;
    }

    public VirtualMachineImageStoreMXBean getVMImageStore() {
        return this.pool.getVMImageStore();
    }

    public String getUUID() {
        // TODO Auto-generated method stub
        return null;
    }

    @Override
    public MBeanNotificationInfo[] getNotificationInfo() {
        return new MBeanNotificationInfo[] {new MBeanNotificationInfo(new String[] {NotificationTypes.VM_STATE_CHANGE,
            NotificationTypes.VM_ADD, NotificationTypes.VM_DEL, NotificationTypes.LOG, NotificationTypes.ERROR,
            NotificationTypes.PERF_REPORT}, Notification.class.getName(), "Host event")};
    }

    private Set<PerfMetric> currentMonitoredMetrics;

    private long currentMonitoringPeriod;

    public void configurePerfMonitor(final Set<PerfMetric> metricsOfInterest, long periodMillis) {
        this.stopPerfMonitor();
        if (periodMillis < 1000) {
            periodMillis = 1000;
        }
        if (metricsOfInterest.contains(PerfMetric.VM_CPU_LOAD)) {
            this.currentMonitoringPeriod = periodMillis;
            this.currentMonitoredMetrics = metricsOfInterest;
            this.startPerfMonitor();
        }
    }

    private void startPerfMonitor() {
        if (this.perfTimer == null) {
            DummyHost.logger.info("Host " + this.hostName + ": starting periodic perf monitor task");
            this.perfTimer = new Timer();
            this.perfTimer.schedule(new PerfTimerTask(), 0, this.currentMonitoringPeriod);
        }
    }

    private void stopPerfMonitor() {
        if (this.perfTimer != null) {
            DummyHost.logger.info("Host " + this.hostName + ": stopping periodic perf monitor task");
            this.perfTimer.cancel();
        }
        this.perfTimer = null;
    }

    private class PerfTimerTask extends TimerTask {
        @Override
        public void run() {
            HashMap<String, Float> cpuLoads = new HashMap<String, Float>();
            int i = 0;
            for (DummyVirtualMachine vm : DummyHost.this.vms) {
                cpuLoads.put(vm.getNameLabel(), 0.25f);
                i++;
            }
            DummyHost.this.emitNotification(NotificationTypes.PERF_REPORT, "CPU load", cpuLoads);
        }
    }
}
