/**
 * JASMINe VMMapi: JASMINe Virtual Machine Management API
 * Copyright (C) 2009 France Telecom R&D
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: RemoteExec.java 3227 2009-04-03 13:03:14Z dangtran $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.vmm.agent.driver.util;

import java.io.InputStream;

import org.apache.log4j.Logger;

import com.jcraft.jsch.ChannelExec;
import com.jcraft.jsch.JSch;
import com.jcraft.jsch.Session;
import com.jcraft.jsch.UserInfo;

/**
 * Utility class allowing to invoke commands on a remote host using SSH
 */
public class RemoteExec {
    static Logger logger = Logger.getLogger(RemoteExec.class);

    /**
     * Result of a remote SSH invocation
     */
    static public class Result {
        /**
         * exit code of the remote command
         */
        public int exitCode;

        /**
         * standard output
         */
        public String output;

        /**
         * standard error
         */
        public String error;
    }

    /**
     * SSH authentification information
     */
    public static class SshAuthInfo {
        private String password;

        private String keyFile;

        /**
         * Constructs a new SshAuthInfo based on either a password or a private
         * key location
         * 
         * @param password password or null if a key file is provided
         * @param keyFile filename of a private SSH key file or null if a
         *        password is provided
         */
        public SshAuthInfo(final String password, final String keyFile) {
            this.password = password;
            this.keyFile = keyFile;
            if (password == null && keyFile == null) {
                throw new IllegalArgumentException();
            }
        }

        @Override
        public String toString() {
            return (this.password != null) ? "password=" + this.password : "keyfile=" + this.keyFile;
        }
    }

    static public class SshException extends Exception {
        private static final long serialVersionUID = -7284013737516587531L;
    }

    /**
     * Invoke a command through SSH on a remote host as root
     * 
     * @param host target host
     * @param password root password
     * @param command command to exec on the remote host as root
     * @return result of the invocation
     * @throws SshException raised if the SSH connection could not be
     *         established
     */
    static public Result commandAsRoot(final String host, final SshAuthInfo authInfo, final String command) throws SshException {
        return RemoteExec.command(host, "root", authInfo, command);
    }

    /**
     * Invoke a command through SSH on a remote host
     * 
     * @param host target host
     * @param user user login
     * @param password user password
     * @param command command to exec on the remote host
     * @return result of the invocation
     * @throws SshException raised if the SSH connection could not be
     *         established
     */
    static public Result command(final String host, final String user, final SshAuthInfo authInfo, final String command)
        throws SshException {
        try {
            JSch jsch = new JSch();
            if (authInfo.keyFile != null) {
                jsch.addIdentity(authInfo.keyFile, "passphrase");
            }
            Session session = jsch.getSession(user, host, 22);
            session.setUserInfo(new UserInfo() {
                public String getPassphrase() {
                    return null;
                }

                public String getPassword() {
                    return authInfo.password;
                }

                public boolean promptPassphrase(final String arg0) {
                    return true;
                }

                public boolean promptPassword(final String arg0) {
                    return true;
                }

                public boolean promptYesNo(final String arg0) {
                    return true;
                }

                public void showMessage(final String arg0) {
                }
            });
            session.connect();
            ChannelExec channel = (ChannelExec) session.openChannel("exec");
            (channel).setCommand(command);
            // channel.setInputStream(null);
            // ((ChannelExec) channel).setErrStream(System.err);
            InputStream in = channel.getInputStream();
            InputStream err = channel.getErrStream();
            channel.connect();
            Result result = new Result();
            StringBuffer output = new StringBuffer();
            StringBuffer error = new StringBuffer();
            byte[] tmp = new byte[1024];

            while (true) {
                while (in.available() > 0) {
                    int i = in.read(tmp, 0, 1024);
                    if (i < 0) {
                        break;
                    }
                    String s = new String(tmp, 0, i);
                    output.append(s);
                }
                if (channel.isClosed()) {
                    result.exitCode = channel.getExitStatus();
                    break;
                }
                try {
                    Thread.sleep(100);
                } catch (Exception ee) {
                }
            }

            while (true) {
                while (err.available() > 0) {
                    int i = err.read(tmp, 0, 1024);
                    if (i < 0) {
                        break;
                    }
                    String s = new String(tmp, 0, i);
                    error.append(s);
                }
                if (channel.isClosed()) {
                    result.exitCode = channel.getExitStatus();
                    break;
                }
                try {
                    Thread.sleep(100);
                } catch (Exception ee) {
                }
            }

            channel.disconnect();
            session.disconnect();
            result.output = new String(output);
            result.error = new String(error);
            return result;
        } catch (Exception ex) {
            RemoteExec.logger.error("SSH failure: user=[" + user + "] auth=[" + authInfo + "]");
            SshException e = new SshException();
            e.initCause(ex);
            throw e;
        }
    }

}
