/**
 * JASMINe VMMapi: JASMINe Virtual Machine Management API
 * Copyright (C) 2009 France Telecom R&D
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: VMwareHost.java 3535 2009-04-23 13:07:22Z alitokmen $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.vmm.agent.driver.vmware;

import java.math.BigInteger;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.Timer;

import javax.management.MBeanNotificationInfo;
import javax.management.Notification;
import javax.management.ObjectName;

import org.apache.log4j.Logger;
import org.ow2.jasmine.vmm.agent.domain.ManagedResource;
import org.ow2.jasmine.vmm.agent.jmx.MBeanObjectNamer;
import org.ow2.jasmine.vmm.agent.main.AgentCommon;
import org.ow2.jasmine.vmm.api.HostMXBean;
import org.ow2.jasmine.vmm.api.InsufficientResourcesException;
import org.ow2.jasmine.vmm.api.InvalidVMConfigException;
import org.ow2.jasmine.vmm.api.NotificationTypes;
import org.ow2.jasmine.vmm.api.ServerPoolMXBean;
import org.ow2.jasmine.vmm.api.VMConfigSpec;
import org.ow2.jasmine.vmm.api.VMMException;
import org.ow2.jasmine.vmm.api.VirtualMachineImageStoreMXBean;
import org.ow2.jasmine.vmm.api.VirtualMachineMXBean;
import org.w3c.dom.Element;

import com.vmware.vim.ArrayOfManagedObjectReference;
import com.vmware.vim.DynamicProperty;
import com.vmware.vim.HostCpuInfo;
import com.vmware.vim.HostCpuPackage;
import com.vmware.vim.HostHardwareInfo;
import com.vmware.vim.ManagedObjectReference;
import com.vmware.vim.ObjectContent;
import com.vmware.vim.VirtualMachineCloneSpec;
import com.vmware.vim.VirtualMachineConfigSpec;
import com.vmware.vim.VirtualMachineRelocateSpec;

/**
 * VMware driver Host MXBean implementation
 */
class VMwareHost extends ManagedResource implements HostMXBean {
    static Logger logger = Logger.getLogger(VMwareHost.class);

    private VMwareServerPool pool;

    private VMwareServiceConnection defaultConnection;

    private VMwareServerPool.ConnectionPool connectionPool;

    private String hostName;

    private int numCPU = -1;

    private ManagedObjectReference mor;

    private HashMap<String, VMwareVirtualMachine> vmRefs = new HashMap<String, VMwareVirtualMachine>();

    private VMwarePerfMonitor perfMon = null;

    private HashMap<String, BigInteger> hostMetricValues = null;

    private HashMap<String, HashMap<String, BigInteger>> metricValuesPerVM = null;

    private String[][] counterNames = null;

    private ManagedObjectReference myResourcePool;

    public VMwareHost(final VMwareServerPool pool, final ManagedObjectReference mor, final ObjectName objectName,
        final String hostName) {
        super(objectName);
        this.pool = pool;
        this.defaultConnection = pool.getConnectionPool().getConnection();
        this.connectionPool = pool.getConnectionPool();
        this.mor = mor;
        this.objectName = objectName;
        this.hostName = hostName;
        // this.PerfMon = new VMWarePerfMonitor(vmUtil.cb, this);
        this.counterNames = new String[][] {new String[] {"cpu", "usage", "average"}, // host
            // & vm
            new String[] {"mem", "usage", "average"}, // host & vm
            new String[] {"mem", "active", "average"}, // host & vm
            new String[] {"mem", "granted", "average"}, // host & vm
            new String[] {"mem", "state", "latest"}, // host only
            new String[] {"disk", "usage", "average"}, // host (& vm
            // ???)
            new String[] {"net", "transmitted", "average"}, // host & vm
            new String[] {"net", "received", "average"}, // host & vm
            new String[] {"sys", "uptime", "latest"} // host & vm
        };
    }

    public ManagedObjectReference getMOR() {
        return this.mor;
    }

    public ServerPoolMXBean getServerPool() {
        return this.pool;
    }

    @Override
    public ObjectName getObjectName() {
        return this.objectName;
    }

    // HostMBean Interface

    public String getHostName() {
        return this.hostName;
    }

    public VMwareServiceConnection getDefaultConnection() {
        return this.defaultConnection;
    }

    public VMwareServerPool.ConnectionPool getConnectionPool() {
        return this.pool.getConnectionPool();
    }

    HashMap<String, String> hypervisorInfo = null;

    public Map<String, String> getHypervisorInfo() { // TODO
        if (this.hypervisorInfo == null) {
            this.hypervisorInfo = new HashMap<String, String>();
            this.hypervisorInfo.put("name", "ESX");
            this.hypervisorInfo.put("vendor", "VMware");
            this.hypervisorInfo.put("version", "3.0.2");
        }
        return this.hypervisorInfo;
    }

    public int getNumCPU() {
        if (this.numCPU == -1) {
            this.getCPUInfo();
        }
        return this.numCPU;
    }

    HashMap<String, String> cpuInfo = null;

    public Map<String, String> getCPUInfo() { // TODO
        if (this.cpuInfo == null) {
            this.cpuInfo = new HashMap<String, String>();
            try {
                DynamicProperty[] cpuArry = this.getDynamicProarray(this.mor, "hardware");
                HostHardwareInfo hinfo = (HostHardwareInfo) cpuArry[0].getVal();
                HostCpuInfo tinfo = hinfo.getCpuInfo(); // ????
                HostCpuPackage cinfo = hinfo.getCpuPkg(0);

                this.cpuInfo.put("speed", Long.toString(tinfo.getHz() / 1000000));
                this.numCPU = tinfo.getNumCpuCores();
                this.cpuInfo.put("vendor", cinfo.getVendor());
            } catch (Exception ex) {
                VMwareHost.logger.debug("Host " + this.hostName, ex);
            }
        }
        return this.cpuInfo;
    }

    private VMwareVirtualMachine addVm(final ManagedObjectReference vmRef, final String vmName) {
        ObjectName vmMBeanObjectName = MBeanObjectNamer.makeVirtualMachineName(this.pool.getPath() + "/" + vmName,
            VMwareVirtualMachine.getUuid(this.defaultConnection, vmRef));
        VMwareVirtualMachine vm = new VMwareVirtualMachine(vmMBeanObjectName, this.pool, vmRef, this, vmName);
        try {
            AgentCommon.getMBeanServer().registerMBean(vm, vmMBeanObjectName);
        } catch (Exception ex) {
        }
        this.vmRefs.put(vmName, vm);
        VMwareHost.logger.info("Added VirtualMachine " + vmName + " " + vmMBeanObjectName);
        return vm;
    }

    public List<VirtualMachineMXBean> getResidentVMs() {
        List<VirtualMachineMXBean> result = new ArrayList<VirtualMachineMXBean>();

        try {
            ArrayOfManagedObjectReference vms = (ArrayOfManagedObjectReference) this.getDynamicProarray(this.mor, "vm")[0]
                .getVal();
            for (ManagedObjectReference vmMor : vms.getManagedObjectReference()) {
                boolean isTemplate = (Boolean) this.getDynamicProarray(vmMor, "summary.config.template")[0].getVal();
                if (isTemplate) {
                    continue;
                }
                DynamicProperty[] properties = this.getDynamicProarray(vmMor, "config.name");
                if (properties == null || properties.length < 1) {
                    continue;
                }
                String vmName = (String) properties[0].getVal();
                VMwareVirtualMachine vm = this.vmRefs.get(vmName);
                if (vm == null) {
                    vm = this.addVm(vmMor, vmName);
                }

                result.add(vm);
            }
        } catch (Exception ex) {
            VMwareHost.logger.error("Failed to retrieve resident VM of host " + this.hostName, ex);
        }

        return result;
    }

    // synchronized void onCreatedVM(ManagedObjectReference vmRef, String
    // vmName) {
    // VMwareVirtualMachine vm = vmRefs.get(vmName);
    // if (vm != null)
    // emitNotification(NotificationTypes.VM_ADD, "Created",
    // vm.getObjectName());
    // else logger.debug("VmCreatedEvent received for unknown VM " + vmName);
    // }
    //
    // synchronized void onRemovedVM(ManagedObjectReference vmRef, String
    // vmName) {
    // VMwareVirtualMachine vm = vmRefs.get(vmName);
    // if (vm != null) {
    // emitNotification(NotificationTypes.VM_DEL, "Destroyed",
    // vm.getObjectName());
    // logger.debug("Notification: VM removed: " + vmName);
    // } else
    // logger.debug("VmRemovedEvent received for unknown VM");
    // }

    synchronized void onPowerStateChangedVM(final ManagedObjectReference vmRef, final String vmName,
        final VirtualMachineMXBean.PowerState state, final long time) {
        VMwareVirtualMachine vm = this.vmRefs.get(vmName);
        if (vm != null) {
            vm.onPowerStateChangedVM(state, time);
        }
    }

    public long getTotalMemoryMB() {
        long total = 0;
        try {
            DynamicProperty[] cpuArry = this.getDynamicProarray(this.mor, "hardware");
            total = ((HostHardwareInfo) cpuArry[0].getVal()).getMemorySize();
        } catch (Exception ex) {
            VMwareHost.logger.debug("Host " + this.hostName, ex);
        }
        return total / (1024 * 1024);
    }

    public long getFreeMemoryMB() {
        if (this.hostMetricValues != null) {
            return this.hostMetricValues.get("mem.granted").longValue();
        } else {
            try {
                int overallMemoryUsage = (Integer) this.getDynamicProarray(this.mor, "summary.quickStats.overallMemoryUsage")[0]
                    .getVal();
                return this.getTotalMemoryMB() - overallMemoryUsage;
            } catch (Exception ex) {
                VMwareHost.logger.error("Cannot retrieve attribute summary.quickStats.overallMemoryUsage", ex);
                return -1;
            }
        }
    }

    public float[] getLoadPerCPU() {
        VMwareHost.logger.info("[VMWareHost.getLoadPerCPU] : Not Implemented");
        return null;
    }

    public float getCPULoad() { // [0,1] interval ?
        if (this.hostMetricValues != null) {
            return (this.hostMetricValues.get("cpu.usage").floatValue() / 10000);
        }
        return 0;
    }

    public VirtualMachineImageStoreMXBean getVMImageStore() {
        return this.pool.getVMImageStore();
    }

    private Timer perfTimer = null;

    private Set<PerfMetric> currentMonitoredMetrics;

    private long currentMonitoringPeriod;

    public void configurePerfMonitor(final Set<PerfMetric> metricsOfInterest, long periodMillis) {
        this.stopPerfMonitor();
        if (periodMillis < 20000) {
            periodMillis = 20000;
        }
        if (metricsOfInterest.contains(PerfMetric.VM_CPU_LOAD)) {
            this.currentMonitoringPeriod = periodMillis;
            this.currentMonitoredMetrics = metricsOfInterest;
            this.startPerfMonitor();
        }
    }

    public synchronized void startPerfMonitor() {
        if (this.perfTimer == null) {
            this.perfTimer = new Timer(true);
            synchronized (this.defaultConnection) {
                this.perfMon = new VMwarePerfMonitor(this.defaultConnection, this);
            }
            this.perfMon.configure(this.counterNames);
            this.perfTimer.schedule(this.perfMon, 0, 20000);
        }

    }

    public synchronized void stopPerfMonitor() {
        if (this.perfTimer != null) {
            this.perfTimer.cancel();
            this.hostMetricValues = null;
            this.metricValuesPerVM = null;
            for (VMwareVirtualMachine vm : this.vmRefs.values()) {
                vm.updateMetrics(null);
            }
            this.perfTimer = null;
        }
    }

    public void updateMetrics(final HashMap<String, BigInteger> hM, final HashMap<String, HashMap<String, BigInteger>> vmM) {
        this.hostMetricValues = hM;
        this.metricValuesPerVM = vmM;

        HashMap<String, Float> cpuLoads = new HashMap<String, Float>();

        for (VMwareVirtualMachine vm : this.vmRefs.values()) {
            String vmName = vm.getNameLabel();
            if (vmM.containsKey(vmName)) {
                vm.updateMetrics(vmM.get(vmName));
                cpuLoads.put(vmName, vm.getCPULoad() / this.getNumCPU());
            }
        }
        this.emitNotification(NotificationTypes.PERF_REPORT, "CPU load", cpuLoads);
    }

    // TODO revisit this function when VCPU >1
    public Map<String, Float> getVMCPULoads() {
        HashMap<String, Float> cpuLoads = new HashMap<String, Float>();

        long hostMhz = Long.parseLong(this.getCPUInfo().get("speed"));

        try {
            for (VMwareVirtualMachine vm : this.vmRefs.values()) {
                if (vm.getHostMBean() == this && vm.getState() == VirtualMachineMXBean.PowerState.RUNNING
                    && !(Boolean) this.getDynamicProarray(vm.getRef(), "summary.config.template")[0].getVal()) {

                    int loadMhz = (Integer) this.getDynamicProarray(vm.getRef(), "summary.quickStats.overallCpuUsage")[0]
                        .getVal();
                    float loadPercentage = (float) loadMhz / hostMhz;
                    if (loadPercentage > 1) {
                        loadPercentage = 1;
                    }
                    cpuLoads.put(vm.getNameLabel(), loadPercentage);
                }
            }

        } catch (Exception ex) {
            VMwareHost.logger.error("Failed to retrieve CPU loads", ex);
        }
        return cpuLoads;
    }

    public ManagedObjectReference getVCResourcePool() {
        if (this.myResourcePool == null) {
            DynamicProperty[] dProps;
            try {
                dProps = this.getDynamicProarray(this.mor, "parent");
                ManagedObjectReference computeResource = (ManagedObjectReference) dProps[0].getVal();
                dProps = this.getDynamicProarray(computeResource, "resourcePool");
                this.myResourcePool = (ManagedObjectReference) dProps[0].getVal();
                VMwareHost.logger.debug("Found resourcePool of host " + this.hostName);
            } catch (Exception e) {
                VMwareHost.logger.error("Failed to get Resource Pool of host " + this.hostName, e);
            }
        }
        return this.myResourcePool;
    }

    private ManagedObjectReference findVMTemplateMOR(final String name) {
        try {
            ArrayOfManagedObjectReference vms = (ArrayOfManagedObjectReference) this.getDynamicProarray(this.mor, "vm")[0]
                .getVal();
            for (ManagedObjectReference vmMor : vms.getManagedObjectReference()) {
                boolean isTemplate = (Boolean) this.getDynamicProarray(vmMor, "summary.config.template")[0].getVal();
                if (isTemplate) {
                    DynamicProperty[] properties = this.getDynamicProarray(vmMor, "config.name");
                    if (properties == null || properties.length < 1) {
                        continue;
                    }
                    String vmName = (String) properties[0].getVal();
                    if (vmName.equals(name)) {
                        return vmMor;
                    }
                }
            }
        } catch (Exception ex) {
            VMwareHost.logger.error(ex);
        }
        return null;
    }

    public VirtualMachineMXBean createVM(final VMConfigSpec vmSpecIn, final boolean sync)
        throws InsufficientResourcesException, InvalidVMConfigException, VMMException {
        final VMConfigSpec vmSpec = new VMConfigSpec(vmSpecIn);
        VMwareHost.logger.info("Creating new VM [name=" + vmSpec.getName() + ",memorySizeMB=" + vmSpec.getMemorySizeMB()
            + ",diskSize=" + vmSpec.getDiskSizeMB() + ",numVCPU=" + vmSpec.getNumVCPU() + ", ID=" + vmSpec.getVmImageUUID()
            + "]");
        if (this.pool.getVMImageStore().lookUpByUUID(vmSpec.getVmImageUUID()) == null) {
            VMwareHost.logger.debug("Invalid VM Image UUID " + vmSpec.getVmImageUUID());
            throw new InvalidVMConfigException("Invalid VM Image UUID");
        }

        ManagedObjectReference templateVMRef = this.findVMTemplateMOR(vmSpec.getVmImageUUID());
        if (templateVMRef == null) {
            VMwareHost.logger.debug("Template " + vmSpec.getVmImageUUID() + " not found");
            return null;
        }

        final VMwareServiceConnection connection = this.connectionPool.getConnection();
        if (connection == null) {
            VMwareHost.logger.debug("Out of connection");
            throw new VMMException("Too many VM creation requests, try again later");
        }

        VirtualMachineCloneSpec cloneSpec = new VirtualMachineCloneSpec();
        VirtualMachineRelocateSpec relocSpec = new VirtualMachineRelocateSpec();

        relocSpec.setPool(this.getVCResourcePool());
        relocSpec.setDatastore(this.pool.getDatastoreRef());
        cloneSpec.setLocation(relocSpec);
        cloneSpec.setPowerOn(false);
        cloneSpec.setTemplate(false);
        String clonedName = vmSpec.getName();

        try {
            synchronized (connection) {
                final ManagedObjectReference cloneTask = connection.getService().cloneVM_Task(templateVMRef,
                    this.pool.getVmFolderRef(), clonedName, cloneSpec);
                if (!sync) {
                    this.pool.getExecutorService().execute(new Runnable() {
                        public void run() {
                            try {
                                VMwareHost.this.createVM2(vmSpec, connection, cloneTask);
                            } catch (Exception ex) {
                            }
                        }
                    });
                    return null;
                } else {
                    return this.createVM2(vmSpec, connection, cloneTask);
                }
            }

        } catch (Exception e) {
            connection.release();
            if (e instanceof org.apache.axis.AxisFault) {
                VMwareHost.logger.debug("AxisFault Exception...");
                org.apache.axis.AxisFault fault = (org.apache.axis.AxisFault) e;
                org.w3c.dom.Element[] errors = fault.getFaultDetails();
                for (Element error : errors) {
                    if (error.toString().indexOf("DuplicateName") != -1) {
                        VMwareHost.logger.debug("Virtual Machine with the same name already exists");
                        throw new InvalidVMConfigException("Virtual Machine with the same name already exists");
                    } else if (error.toString().indexOf("InvalidArgument") != -1) {
                        VMwareHost.logger.debug("Specification is invalid");
                        throw new InvalidVMConfigException();
                    } else if (error.toString().indexOf("InvalidName") != -1) {
                        VMwareHost.logger.debug("Virtual Machine name is empty or too long");
                        throw new InvalidVMConfigException("Virtual Machine name is empty or too long");
                    } else if (error.toString().indexOf("RuntimeFault") != -1) {
                        VMwareHost.logger.debug(error.toString());
                        throw new VMMException();
                    } else {
                        VMwareHost.logger.debug(error.toString());
                        throw new VMMException();
                    }
                }
            } else {
                VMwareHost.logger.error("CloneVMTask Exception...", e);
                throw new VMMException(e);
            }
        }
        return null;
    }

    private VirtualMachineMXBean createVM2(final VMConfigSpec vmSpec, final VMwareServiceConnection connection,
        final ManagedObjectReference cloneTask) throws InsufficientResourcesException, InvalidVMConfigException, VMMException {
        VMwareVirtualMachine newVM;
        ManagedObjectReference vmMor = null;
        ObjectName name = null;
        String uuid = null;

        try {
            VMwareHost.logger.debug("CloneVMTask launched, waiting for result...");
            String status = null;
            synchronized (connection) {
                status = connection.waitForTask(cloneTask);
            }
            VMwareHost.logger.debug("CloneVMTask, status=" + status);

            if (status.equalsIgnoreCase("failure")) {
                String msg = "Failure -: Virtual Machine cannot be cloned";
                VMwareHost.logger.error(msg);
                throw new VMMException(msg);
            }

            if (status.equalsIgnoreCase("success")) {
                VMwareHost.logger.debug("Virtual Machine cloned  successfully.");
                // TaskInfo = getDynamicProarray(cloneTask,"info");
                vmMor = (ManagedObjectReference) this.getDynamicProarray(cloneTask, "info.result")[0].getVal();
                uuid = VMwareVirtualMachine.getUuid(this.defaultConnection, vmMor);
            }
        } catch (Exception e) {
            throw new VMMException(e);
        } finally {
            connection.release();
        }

        try {
            name = MBeanObjectNamer.makeVirtualMachineName(this.pool.getPath() + "/" + vmSpec.getName(), uuid);
            newVM = new VMwareVirtualMachine(name, this.pool, vmMor, this, vmSpec.getName());
            this.vmRefs.put(vmSpec.getName(), newVM);
            AgentCommon.getMBeanServer().registerMBean(newVM, name);
        } catch (Exception ex) {
            VMwareHost.logger.error("Failed to register new VM MBean " + name, ex);
            return null;
        }

        // before starting the VM set the memory and number of CPU parameters
        this.setResourceCapacity(vmMor, vmSpec);

        newVM.start();
        VMwareHost.logger.debug("[Host " + this.hostName + "] VM started");

        this.emitNotification(NotificationTypes.VM_ADD, "Created", name);

        return newVM;
    }

    void onVMDestroy(final VMwareVirtualMachine vm) {
        this.vmRefs.remove(vm.getNameLabel());
        this.emitNotification(NotificationTypes.VM_DEL, "Destroyed", vm.getObjectName());
    }

    private void setResourceCapacity(final ManagedObjectReference vmMor, final VMConfigSpec vmSpec) {
        VirtualMachineConfigSpec vmConfigSpec = new VirtualMachineConfigSpec();
        try {
            vmConfigSpec.setMemoryMB(vmSpec.getMemorySizeMB());
            vmConfigSpec.setNumCPUs(vmSpec.getNumVCPU());
            synchronized (this.defaultConnection) {
                ManagedObjectReference tmor = this.defaultConnection.getService().reconfigVM_Task(vmMor, vmConfigSpec);

                String status = this.defaultConnection.waitForTask(tmor);

                if (status.equalsIgnoreCase("failure")) {
                    String msg = "Failure -: Virtual Machine cannot be cloned";
                    VMwareHost.logger.error(msg);
                } else {
                    VMwareHost.logger.debug("VM " + vmSpec.getName() + ": Memory capacity set to " + vmSpec.getMemorySizeMB()
                        + "MB, number of CPU set to " + vmSpec.getNumVCPU());
                }
            }
        } catch (Exception e) {
            VMwareHost.logger.error("Failed to set VM memory and numCPU", e);
        }
    }

    private DynamicProperty[] getDynamicProarray(final ManagedObjectReference MOR, final String pName) throws Exception {
        ObjectContent[] objContent = this.defaultConnection.getObjectProperties(null, MOR, new String[] {pName});
        ObjectContent contentObj = objContent[0];
        DynamicProperty[] objArr = contentObj.getPropSet();
        return objArr;
    }

    @Override
    public MBeanNotificationInfo[] getNotificationInfo() {
        return new MBeanNotificationInfo[] {new MBeanNotificationInfo(new String[] {NotificationTypes.VM_STATE_CHANGE,
            NotificationTypes.VM_ADD, NotificationTypes.VM_DEL, NotificationTypes.ERROR, NotificationTypes.LOG},
            Notification.class.getName(), "Host event")};
    }

}
