/**
 * JASMINe VMMapi: JASMINe Virtual Machine Management API
 * Copyright (C) 2009 France Telecom R&D
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: DummyVirtualMachine.java 3973 2009-06-13 00:11:04Z dangtran $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.vmm.agent.driver.dummy;

import java.net.InetAddress;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;

import javax.management.MBeanNotificationInfo;
import javax.management.Notification;
import javax.management.ObjectName;

import org.apache.log4j.Logger;
import org.ow2.jasmine.vmm.agent.domain.ManagedResource;
import org.ow2.jasmine.vmm.api.BadVMPowerStateException;
import org.ow2.jasmine.vmm.api.HostMXBean;
import org.ow2.jasmine.vmm.api.IllegalOperationException;
import org.ow2.jasmine.vmm.api.InsufficientResourcesException;
import org.ow2.jasmine.vmm.api.NotificationTypes;
import org.ow2.jasmine.vmm.api.VMMException;
import org.ow2.jasmine.vmm.api.VirtualMachineMXBean;

/**
 * Dummy driver VirtualMachine MXBean implementation
 */
class DummyVirtualMachine extends ManagedResource implements VirtualMachineMXBean {
    static Logger logger = Logger.getLogger(DummyVirtualMachine.class);

    private String name;

    private DummyHost host;

    private String uuid;

    private String ipAddress;

    private int numVCPU;

    private int schedulingCap, schedulingWeight;

    private int diskSizeMB;

    private long memorySizeMB, memoryUsedMB;

    private PowerState status = PowerState.RUNNING;

    private Date startTime;

    public DummyVirtualMachine(final String name, final String uuid, final ObjectName objectName, final DummyHost host) {
        super(objectName);
        this.name = name;
        this.host = host;
        this.uuid = uuid;
        this.memorySizeMB = 1024;

        this.startTime = new Date(System.currentTimeMillis());

        try {
            this.ipAddress = InetAddress.getByName("198.7.9.66").getHostAddress();
        } catch (Exception ex) {
            DummyVirtualMachine.logger.error("Failed to get IP address", ex);
        }
    }

    public Date getStartTime() {
        return this.startTime;
    }

    public long getUpTimeSeconds() {
        return (System.currentTimeMillis() - this.startTime.getTime()) / 1000;
    }

    public int getNumVCPUs() {
        return this.numVCPU;
    }

    public void setNumVCPUs(final int numVCPU) {
        this.numVCPU = numVCPU;
    }

    public int getDiskSizeMB() {
        return this.diskSizeMB;
    }

    public void setDiskSizeMB(final int diskSizeMB) {
        this.diskSizeMB = diskSizeMB;
    }

    public long getMemorySizeMB() {
        return this.memorySizeMB;
    }

    public long getMemoryUsedMB() {
        return this.memoryUsedMB;
    }

    public void setMemorySizeMB(final long memorySizeMB) {
        this.memorySizeMB = memorySizeMB;
        this.memoryUsedMB = memorySizeMB / 2;
    }

    Map<String, String> userData = new HashMap<String, String>();

    public void addUserData(final String key, final String value) {
        this.userData.put(key, value);
    }

    public String getUserData(final String key) {
        return this.userData.get(key);
    }

    public void onVMStatusChanged() {
        DummyVirtualMachine.logger.info("VM " + this.getNameLabel() + " state=" + this.getState());
        this.emitNotification(NotificationTypes.VM_STATE_CHANGE, this.getState().toString(), null);
    }

    public boolean canLiveMigrateToHost(final HostMXBean targetHost) {
        return targetHost != this.host && this.getMemorySizeMB() <= targetHost.getFreeMemoryMB();
    }

    public void destroy() {
        this.host.onDestroyVM(this);
    }

    public float[] getLoadPerVCPU() {
        return new float[] {8, 10};
    }

    public float getCPULoad() {
        return 0;
    }

    public HostMXBean getHostMBean() {
        return this.host;
    }

    public String getGuestIpAddress() {
        return this.ipAddress;
    }

    public String getMacAddress() {
        return "8E:10:65:AB:CE:26";
    }

    public int getMemory() {
        return 0;
    }

    public String getNameLabel() {
        return this.name;
    }

    public String getPinnedVCPUs() {
        return null;
    }

    public int getSchedulingCap() {
        return this.schedulingCap;
    }

    public int getSchedulingWeight() {
        return this.schedulingWeight;
    }

    public PowerState getState() {
        return this.status;
    }

    public String getUuid() {
        return this.uuid;
    }

    public void reboot() {
    }

    public void resume() {
        if (this.status == PowerState.SUSPENDED) {
            this.status = PowerState.RUNNING;
            this.onVMStatusChanged();
            DummyVirtualMachine.logger.info("VM " + this.name + " resumed");
        }
    }

    public void setMacAddress(final String mac) {
    }

    public void setPinnedVCPUs(final String pinnedVCPUs) {
    }

    public void setSchedulingCap(final int schedulingCap) {
        this.schedulingCap = schedulingCap;
    }

    public void setSchedulingWeight(final int schedulingWeight) {
        this.schedulingWeight = schedulingWeight;
    }

    public void shutdown() {
        if (this.status == PowerState.RUNNING) {
            this.status = PowerState.HALTED;
            this.onVMStatusChanged();
            DummyVirtualMachine.logger.info("VM " + this.name + " shutdowned");
        }
    }

    public void start() {
        if (this.status == PowerState.HALTED) {
            this.status = PowerState.RUNNING;
            this.onVMStatusChanged();
            DummyVirtualMachine.logger.info("VM " + this.name + " started");
        }
    }

    public void suspend() throws BadVMPowerStateException {
        if (this.status == PowerState.RUNNING) {
            this.status = PowerState.SUSPENDED;
            this.onVMStatusChanged();
            DummyVirtualMachine.logger.info("VM " + this.name + " suspended");
        }
    }

    int count = 0;

    public void migrate(HostMXBean targetHost, final boolean live) throws IllegalOperationException, VMMException {
        String targetHostName = targetHost.getHostName();
        boolean foundHostInServerPool = false;
        // TODO need to found out how to check equality between MXBean whether
        // proxy or local
        for (HostMXBean h : this.host.getServerPool().getManagedHosts()) {
            if (h.getHostName().equals(targetHost.getHostName())) {
                foundHostInServerPool = true;
                targetHost = h;
                break;
            }
        }

        if (!foundHostInServerPool) {
            throw new IllegalOperationException("Source and target hosts belong to different server pools");
        }

        this.emitNotification(NotificationTypes.VM_MIGRATION_START, targetHostName, this.uuid);
        try {
            Thread.sleep(1000);
        } catch (Exception ex) {
        }

        DummyHost target = (DummyHost) targetHost;
        this.host.postMigrateVM(this, target);
        this.host = target;

        this.emitNotification(NotificationTypes.VM_MIGRATION, targetHostName, this.uuid);
    }

    @Override
    public MBeanNotificationInfo[] getNotificationInfo() {
        return new MBeanNotificationInfo[] {new MBeanNotificationInfo(new String[] {NotificationTypes.VM_STATE_CHANGE,
            NotificationTypes.VM_MIGRATION, NotificationTypes.LOG, NotificationTypes.ERROR}, Notification.class.getName(),
            "VM event")};
    }

    public void makeTemplate(final String vmImageID, final String name, final String description)
        throws InsufficientResourcesException, IllegalOperationException, VMMException {
        // TODO

    }
}
