/**
 * JASMINe VMMapi: JASMINe Virtual Machine Management API
 * Copyright (C) 2009 France Telecom R&D
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: VMwareVirtualMachine.java 3973 2009-06-13 00:11:04Z dangtran $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.vmm.agent.driver.vmware;

import java.math.BigInteger;
import java.util.Date;
import java.util.HashMap;

import javax.management.InstanceNotFoundException;
import javax.management.ObjectName;

import org.apache.log4j.Logger;
import org.ow2.jasmine.vmm.agent.domain.ManagedResource;
import org.ow2.jasmine.vmm.agent.main.AgentCommon;
import org.ow2.jasmine.vmm.api.HostMXBean;
import org.ow2.jasmine.vmm.api.IllegalOperationException;
import org.ow2.jasmine.vmm.api.InsufficientResourcesException;
import org.ow2.jasmine.vmm.api.NotificationTypes;
import org.ow2.jasmine.vmm.api.VMMException;
import org.ow2.jasmine.vmm.api.VirtualMachineMXBean;

import com.vmware.vim.DynamicProperty;
import com.vmware.vim.InvalidPowerState;
import com.vmware.vim.ManagedObjectReference;
import com.vmware.vim.ObjectContent;
import com.vmware.vim.VirtualDevice;
import com.vmware.vim.VirtualEthernetCard;
import com.vmware.vim.VirtualHardware;
import com.vmware.vim.VirtualMachineCloneSpec;
import com.vmware.vim.VirtualMachinePowerState;
import com.vmware.vim.VirtualMachineRelocateSpec;

/**
 * VMware driver VirtualMachine MXBean implementation
 */
class VMwareVirtualMachine extends ManagedResource implements VirtualMachineMXBean {
    static Logger logger = Logger.getLogger(VMwareVirtualMachine.class);

    private VMwareServerPool pool;

    private VMwareHost host;

    private ManagedObjectReference ref;

    private String nameLabel;

    private String uuid;

    private String macAddress;

    private String ipAddress;

    HashMap<String, BigInteger> metricValues = null;

    public VMwareVirtualMachine(final ObjectName objectName, final VMwareServerPool pool, final ManagedObjectReference mor,
        final VMwareHost host, final String nameLabel) {
        super(objectName);
        this.pool = pool;
        this.ref = mor;
        this.host = host;
        this.uuid = null;
        this.nameLabel = nameLabel;
    }

    public void updateMetrics(final HashMap<String, BigInteger> m) {
        this.metricValues = m;
    }

    public boolean canLiveMigrateToHost(final HostMXBean targetHost) {
        // TODO
        return false;
    }

    public ManagedObjectReference getRef() {
        return this.ref;
    }

    public void migrate(final HostMXBean targetHost, final boolean live) throws InstanceNotFoundException,
        IllegalOperationException, VMMException {
        // TODO
        throw new UnsupportedOperationException();
    }

    public float[] getLoadPerVCPU() {
        throw new UnsupportedOperationException("not implemented");
    }

    public String getUuid() {
        if (this.uuid == null) {
            final VMwareServiceConnection connection = this.pool.getConnectionPool().getConnection();
            try {
                this.uuid = VMwareVirtualMachine.getUuid(connection, this.ref);
            } finally {
                connection.release();
            }
        }
        return this.uuid;
    }

    public static String getUuid(final VMwareServiceConnection connection, final ManagedObjectReference ref) {
        String uuid = "";
        try {
            DynamicProperty[] props = VMwareVirtualMachine.getDynamicProarray(connection, ref, "config.uuid");
            if (props == null || props.length < 1) {
                VMwareVirtualMachine.logger.error("Cannot retrieve UUID of VM ");
                uuid = "-1";
            } else {
                uuid = (String) props[0].getVal();
            }
        } catch (Exception ex) {
            VMwareVirtualMachine.logger.error("[VMWareVirtualMachine.getUUID()] : ", ex);
        }
        return uuid;
    }

    public Date getStartTime() {
        VMwareVirtualMachine.logger.error("[VMWareVirtualMachine.getStartTime()] : Not Implemented ! ");
        return null;
    }

    public HostMXBean getHostMBean() {
        return this.host;
    }

    public String getNameLabel() {
        return this.nameLabel;
    }

    public String getMacAddress() {
        if (this.macAddress == null) {
            final VMwareServiceConnection connection = this.pool.getConnectionPool().getConnection();
            try {
                VirtualHardware hw = (VirtualHardware) VMwareVirtualMachine.getDynamicProarray(connection, this.ref,
                    "config.hardware")[0].getVal();
                for (VirtualDevice vd : hw.getDevice()) {
                    if (vd instanceof VirtualEthernetCard) {
                        this.macAddress = ((VirtualEthernetCard) vd).getMacAddress();
                        break;
                    }
                }
            } catch (Exception ex) {
                VMwareVirtualMachine.logger.error("[VMWareVirtualMachine.getMacAddress()] : ", ex);
            } finally {
                connection.release();
            }
        }
        return this.macAddress;
    }

    public String getGuestIpAddress() {
        if (this.ipAddress == null) {
            // first try to get IP address thru VMware API
            // assuming that VMware Tools are running in the guest
            int tryNumber = 24;
            final VMwareServiceConnection connection = this.pool.getConnectionPool().getConnection();
            try {
                while (tryNumber-- > 0) {
                    VMwareVirtualMachine.logger.debug("Retrieving IP address of VM " + this.nameLabel + " ...");
                    DynamicProperty[] props = VMwareVirtualMachine.getDynamicProarray(connection, this.ref, "guest.ipAddress");
                    if (props != null && props.length > 0) {
                        this.ipAddress = (String) props[0].getVal();
                        VMwareVirtualMachine.logger.debug("IP address of VM " + this.nameLabel + " is " + this.ipAddress);
                        break;
                    }
                    Thread.sleep(10 * 1000);
                }
            } catch (Exception ex) {
                VMwareVirtualMachine.logger.error("[VMWareVirtualMachine.getGuestIpAddress()] : ", ex);
            } finally {
                connection.release();
            }
        }
        return this.ipAddress;
    }

    public float getCPULoad() {
        // FIXME only works once perf monitor is started
        if (this.metricValues != null && this.metricValues.get("cpu.usage") != null) {
            return (this.metricValues.get("cpu.usage").floatValue() / 10000);
        }
        return 0;
    }

    public long getUpTimeSeconds() {
        if (this.metricValues != null && this.metricValues.get("sys.uptime") != null) {
            return this.metricValues.get("sys.uptime").longValue();
        }
        return 0;
    }

    public long getMemorySizeMB() {
        long result = 0;
        final VMwareServiceConnection connection = this.pool.getConnectionPool().getConnection();
        try {
            result = (Integer) VMwareVirtualMachine.getDynamicProarray(connection, this.ref, "config.hardware.memoryMB")[0]
                .getVal();
        } catch (Exception ex) {
            VMwareVirtualMachine.logger.error("[VMWareVirtualMachine.getMemorySizeMB()] : ", ex);
        } finally {
            connection.release();
        }
        return result;
    }

    public void setMemorySizeMB(final long size) {
        VMwareVirtualMachine.logger.error("[VMWareVirtualMachine.setMemorySizeMB(...)] : Not Implemented ! ");
    }

    public long getMemoryUsedMB() {
        if (this.metricValues != null) {
            return this.metricValues.get("mem.active").longValue() * 1024;
        }
        return 0;
    }

    public int getNumVCPUs() {
        int result = 0;
        final VMwareServiceConnection connection = this.pool.getConnectionPool().getConnection();
        try {
            result = (Integer) VMwareVirtualMachine.getDynamicProarray(connection, this.ref, "config.hardware.numCPU")[0]
                .getVal();
        } catch (Exception ex) {
            VMwareVirtualMachine.logger.error("[VMWareVirtualMachine.getNumVCPUs()] : ", ex);
        } finally {
            connection.release();
        }
        return result;
    }

    public void setNumVCPUs(final int numVCPUs) {
        // TODO
        throw new UnsupportedOperationException();
    }

    public int getSchedulingCap() {
        // TODO
        int result = 0;
        VMwareVirtualMachine.logger.error("[VMWareVirtualMachine.getSchedulingCap()] : Not Implemented ! ");
        return result;
    }

    public void setSchedulingCap(final int schedulingCap) {
        // TODO
        VMwareVirtualMachine.logger.error("[VMWareVirtualMachine.setSchedulingCap(...)] : Not Implemented ! ");
    }

    public int getSchedulingWeight() {
        // TODO
        int result = 0;
        VMwareVirtualMachine.logger.error("[VMWareVirtualMachine.getSchedulingWeight()] : Not Implemented ! ");
        return result;
    }

    public void setSchedulingWeight(final int schedulingWeight) {
        // TODO
        VMwareVirtualMachine.logger.error("[VMWareVirtualMachine.setSchedulingWeight(...)] : Not Implemented ! ");
    }

    public PowerState getState() {
        final VMwareServiceConnection connection = this.pool.getConnectionPool().getConnection();
        try {
            DynamicProperty props[] = VMwareVirtualMachine.getDynamicProarray(connection, this.ref, "runtime.powerState");
            if (props != null) {
                VirtualMachinePowerState state = (VirtualMachinePowerState) props[0].getVal();
                if (state.getValue().equals("poweredOff")) {
                    return PowerState.HALTED;
                } else if (state.getValue().equals("poweredOn")) {
                    return PowerState.RUNNING;
                } else if (state.getValue().equals("suspended")) {
                    return PowerState.SUSPENDED;
                }
            } else {
                return PowerState.UNKNOWN;
            }
        } catch (Exception ex) {
            VMwareVirtualMachine.logger.error("[VMWareVirtualMachine.getState()]: VM " + this.getNameLabel(), ex);
        } finally {
            connection.release();
        }

        return PowerState.UNKNOWN;
    }

    private PowerState lastState = PowerState.UNKNOWN;

    void onPowerStateChangedVM(final VirtualMachineMXBean.PowerState state, final long time) {
        if (state != this.lastState) {
            this.emitNotification(NotificationTypes.VM_STATE_CHANGE, state.toString(), null);
            VMwareVirtualMachine.logger.debug("VM " + this.getNameLabel() + " new power state: " + state.toString());
            this.lastState = state;
        }
    }

    public String getUserData(final String key) {
        throw new UnsupportedOperationException("not yet implemented");
    }

    public void addUserData(final String key, final String value) {
        throw new UnsupportedOperationException("not yet implemented");
    }

    public void suspend() {
        final VMwareServiceConnection connection = this.pool.getConnectionPool().getConnection();
        try {
            ManagedObjectReference taskmor = connection.getService().suspendVM_Task(this.ref);
            boolean result = this.getTaskInfo(connection, taskmor);
            if (result) {
                VMwareVirtualMachine.logger.debug("Virtual Machine " + this.nameLabel + " suspended on successfully");
            }
        } catch (Exception ex) {
            VMwareVirtualMachine.logger.error("Error suspending VM", ex);
        } finally {
            connection.release();
        }
    }

    public void resume() {
        this.start();
    }

    public void shutdown() {
        final VMwareServiceConnection connection = this.pool.getConnectionPool().getConnection();
        try {
            ManagedObjectReference taskmor = connection.getService().powerOffVM_Task(this.ref);
            boolean result = this.getTaskInfo(connection, taskmor);
            if (result) {
                VMwareVirtualMachine.logger.debug("Virtual Machine " + this.nameLabel + " powered off successfuly");
            }
        } catch (Exception ex) {
            VMwareVirtualMachine.logger.error("Error shutting down VM", ex);
        } finally {
            connection.release();
        }
    }

    public void start() {
        final VMwareServiceConnection connection = this.pool.getConnectionPool().getConnection();
        try {
            ManagedObjectReference taskmor = connection.getService().powerOnVM_Task(this.ref, null);
            boolean result = this.getTaskInfo(connection, taskmor);
            if (result) {
                VMwareVirtualMachine.logger.debug("Virtual Machine " + this.nameLabel + " powered on successfully");
            }
        } catch (InvalidPowerState e) {
            VMwareVirtualMachine.logger.debug("Virtual Machine is already powered on");
        } catch (Exception ex) {
            VMwareVirtualMachine.logger.error("Error starting VM", ex);
        } finally {
            connection.release();
        }
    }

    public void reboot() {
        final VMwareServiceConnection connection = this.pool.getConnectionPool().getConnection();
        try {
            // TODO this is a hard reboot
            // should we use reboot_guest ?
            connection.getService().resetVM_Task(this.ref);
        } catch (Exception ex) {
            VMwareVirtualMachine.logger.error("Error rebooting VM", ex);
        } finally {
            connection.release();
        }
    }

    public void destroy() {
        if (this.getState() == PowerState.RUNNING) {
            this.shutdown();
        }

        final VMwareServiceConnection connection = this.pool.getConnectionPool().getConnection();
        try {
            ManagedObjectReference taskmor = connection.getService().destroy_Task(this.ref);
            boolean result = this.getTaskInfo(connection, taskmor);
            if (result) {
                VMwareVirtualMachine.logger.debug("Virtual Machine " + this.nameLabel + " destroyed successfully");
            }
            this.host.onVMDestroy(this);
            AgentCommon.getMBeanServer().unregisterMBean(this.objectName);

        } catch (InvalidPowerState e) {
            VMwareVirtualMachine.logger.debug("Invalid power state");
        } catch (Exception ex) {
            VMwareVirtualMachine.logger.error("Error destroying VM", ex);
        } finally {
            connection.release();
        }
    }

    private static DynamicProperty[] getDynamicProarray(final VMwareServiceConnection connection,
        final ManagedObjectReference MOR, final String pName) throws Exception {
        ObjectContent[] objContent;
        objContent = connection.getObjectProperties(null, MOR, new String[] {pName});
        if (objContent != null) {
            ObjectContent contentObj = objContent[0];
            DynamicProperty[] objArr = contentObj.getPropSet();
            return objArr;
        } else {
            return null;
        }
    }

    private boolean getTaskInfo(final VMwareServiceConnection connection, final ManagedObjectReference taskmor)
        throws Exception {
        boolean valid = false;
        String res;
        res = connection.waitForTask(taskmor);
        if (res.equalsIgnoreCase("success")) {
            valid = true;
        } else {
            valid = false;
        }
        return valid;
    }

    // TODO store name & description in user data fields of the VMware template
    public void makeTemplate(final String vmImageID, final String name, final String description)
        throws InsufficientResourcesException, IllegalOperationException, VMMException {

        VMwareServiceConnection connection = this.host.getConnectionPool().getConnection();
        if (connection == null) {
            VMwareVirtualMachine.logger.debug("Out of connection");
            throw new VMMException("Too many VM creation requests, try again later");
        }

        VirtualMachineCloneSpec cloneSpec = new VirtualMachineCloneSpec();
        VirtualMachineRelocateSpec relocSpec = new VirtualMachineRelocateSpec();
        try {

            this.shutdown();

            relocSpec.setPool(this.host.getVCResourcePool(connection));
            relocSpec.setDatastore(this.pool.getDatastoreRef());
            cloneSpec.setLocation(relocSpec);
            cloneSpec.setPowerOn(false);
            cloneSpec.setTemplate(true);

            synchronized (connection) {
                VMwareVirtualMachine.logger.debug("Making template " + vmImageID + " from VM " + this.getNameLabel());
                String status = "";
                try {
                    ManagedObjectReference cloneTask = connection.getService().cloneVM_Task(this.ref,
                        this.pool.getVmTemplateFolderRef(), vmImageID, cloneSpec);
                    status = connection.waitForTask(cloneTask);
                } catch (Exception e) {
                    VMwareVirtualMachine.logger.error("Failed to create template", e);
                    throw new VMMException(e);
                }

                VMwareVirtualMachine.logger.debug("CloneVMTask, status=" + status);

                if (status.equalsIgnoreCase("failure")) {
                    String msg = "Failure -: Virtual Machine cannot be cloned to template";
                    VMwareVirtualMachine.logger.error(msg);
                    throw new VMMException(msg);
                }
            }
        } finally {
            connection.release();
            this.start();
        }

    }
}
