/**
 * JASMINe VMMapi: JASMINe Virtual Machine Management API
 * Copyright (C) 2009 France Telecom R&D
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: XenServerPool.java 3973 2009-06-13 00:11:04Z dangtran $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.vmm.agent.driver.xen;

import java.util.Map;
import java.util.Properties;

import javax.management.ObjectName;

import org.ow2.jasmine.vmm.agent.domain.ServerPool;
import org.ow2.jasmine.vmm.agent.driver.util.RemoteExec;
import org.ow2.jasmine.vmm.agent.jmx.MBeanObjectNamer;
import org.ow2.jasmine.vmm.agent.main.AgentCommon;
import org.ow2.jasmine.vmm.api.HostMXBean;
import org.ow2.jasmine.vmm.api.VirtualMachineImageStoreMXBean;

/**
 * The XenServerPool is the Xen-driver-specific implementation of the
 * ServerPoolMXBean interface.
 * <p>
 * The Xen driver resorts to a combination of SSH commands and XenAPI
 * connections to control each Xen host of a XenServerPool. The XenServerPool
 * accepts the following properties:
 * <ul>
 * <li>sshDefaultRootPassword : default root ssh password
 * <li>vmmDomain0HomeDir : directory where the driver host-side shell scripts
 * are installed
 * <li>xenapiDefaultUser : XenAPI default login
 * <li>xenapiDefaultPassword : XenAPI default password
 * <li>xenapiDefaulPort : XenAPI default port
 * </ul>
 * These properties will be used to connect to each host member of the
 * XenServerPool.
 */
public class XenServerPool extends ServerPool {
    private String xenAPIPort;

    private String xenAPIUser;

    private String xenAPIPassword;

    private VirtualMachineImageStoreMXBean imageStore;

    private String vmImageStoreDir;

    private RemoteExec.SshAuthInfo defaultAuthInfo;

    public XenServerPool(final String name, final ObjectName objectName, final Properties props) {
        super(name, objectName);
        this.xenAPIPort = props.getProperty("xenapiDefaulPort", "9363");
        this.xenAPIUser = props.getProperty("xenapiDefaultUser", "root");
        this.xenAPIPassword = props.getProperty("xenapiDefaultPassword");
        this.vmImageStoreDir = props.getProperty("vmImageStoreDir", "/var/xen/images/templates");
        String sshPassword = props.getProperty("sshDefaultRootPassword");
        String sshPrivateKeyFile = props.getProperty("sshDefaultPrivateKeyFile");
        this.defaultAuthInfo = new RemoteExec.SshAuthInfo(sshPassword, sshPrivateKeyFile);
    }

    @Override
    public HostMXBean newHost(final String hostName, final Map<String, String> props) {
        try {
            for (HostMXBean host : this.hosts) {
                if (host.getHostName().equals(hostName)) {
                    return host;
                }
            }
            ObjectName mbeanObjectName = MBeanObjectNamer.makeHostName(this.getPath() + "/" + hostName, hostName);
            RemoteExec.SshAuthInfo hostAuthInfo;
            if (props.get("sshRootPassword") != null) {
                hostAuthInfo = new RemoteExec.SshAuthInfo(props.get("sshRootPassword"), null);
            } else if (props.get("sshPrivateKeyFile") != null) {
                hostAuthInfo = new RemoteExec.SshAuthInfo(null, props.get("sshPrivateKeyFile"));
            } else {
                hostAuthInfo = this.defaultAuthInfo;
            }

            XenHost host = XenHost.newHost(this, mbeanObjectName, hostName, new XenHost.XenAPIParams(this.xenAPIPort,
                this.xenAPIUser, this.xenAPIPassword), hostAuthInfo);
            if (host != null) {
                AgentCommon.getMBeanServer().registerMBean(host, mbeanObjectName);
                this.hosts.add(host);
                ServerPool.logger.info("In server pool " + this.name + ": added Host MBean " + mbeanObjectName);
                if (this.imageStore == null) {
                    ObjectName defaultDataStoreName = MBeanObjectNamer.makeVMImageStoreName(hostName + ".default");
                    this.imageStore = new XenVMImageStore(defaultDataStoreName, this.vmImageStoreDir, hostName,
                        this.defaultAuthInfo);
                    AgentCommon.getMBeanServer().registerMBean(this.imageStore, defaultDataStoreName);
                }
            }

            return host;
        } catch (Exception ex) {
            ServerPool.logger.error("Failed to add XenHost", ex);
            return null;
        }
    }

    @Override
    public VirtualMachineImageStoreMXBean getVMImageStore() {
        return this.imageStore;
    }

}
