/**
 * JASMINe VMMapi: JASMINe Virtual Machine Management API
 * Copyright (C) 2009 France Telecom R&D
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: ServerPool.java 5509 2009-10-13 16:28:37Z dangtran $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.vmm.agent.domain;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import javax.management.ObjectName;

import org.apache.log4j.Logger;
import org.ow2.jasmine.vmm.agent.domain.policy.PolicyRegistry;
import org.ow2.jasmine.vmm.agent.domain.policy.VMPlacementPolicy;
import org.ow2.jasmine.vmm.api.HostMXBean;
import org.ow2.jasmine.vmm.api.InsufficientResourcesException;
import org.ow2.jasmine.vmm.api.InvalidVMConfigException;
import org.ow2.jasmine.vmm.api.ServerPoolMXBean;
import org.ow2.jasmine.vmm.api.VMConfigSpec;
import org.ow2.jasmine.vmm.api.VMMException;
import org.ow2.jasmine.vmm.api.VirtualMachineImageStoreMXBean;
import org.ow2.jasmine.vmm.api.VirtualMachineMXBean;

/**
 * Base class partially implementing the ServerPoolMXBean interface and expected
 * to be sub-classed on a per driver basis.
 * <p>
 * A ServerPool is associated with a VM placement policy which determines where
 * a provisioned VM is to be placed on one host among the members of the server
 * pool
 */
public abstract class ServerPool extends ManagedResource implements ServerPoolMXBean {
    static protected Logger logger = Logger.getLogger(ServerPool.class);

    protected String name;

    protected final List<HostMXBean> hosts = new ArrayList<HostMXBean>();

    protected VMPlacementPolicy placementPolicy;

    /**
     * Constructs a new server pool
     * 
     * @param name symbolic name of the server pool
     * @param objectName JMX object name of the server pool
     */
    public ServerPool(final String name, final ObjectName objectName) {
        super(objectName);
        this.name = name;
    }

    /*
     * @see org.ow2.jasmine.vmm.api.ServerPoolMXBean#getName()
     */
    public String getName() {
        return this.name;
    }

    abstract public HostMXBean newHost(String hostName, Map<String, String> props);

    /**
     * Adds a new host to the server pool
     * 
     * @param host host to add
     */
    protected void addHost(final HostMXBean host) {
        this.hosts.add(host);
        ServerPool.logger.info("Added host " + host.getHostName());
    }

    public void setProvisioningPolicy(final String policyId) {
        if (policyId != null) {
            this.placementPolicy = PolicyRegistry.getInstance().newPolicy(policyId);
            ServerPool.logger.info("Domain " + this.name + ": policy set to " + policyId);
        } else {
            this.placementPolicy = PolicyRegistry.getInstance().newDefaultPolicy();
        }
    }

    /*
     * @see
     * org.ow2.jasmine.vmm.api.ServerPoolMXBean#provisionVM(org.ow2.jasmine.
     * vmm.api.VMConfigSpec, java.util.Map, boolean)
     */
    public VirtualMachineMXBean provisionVM(final VMConfigSpec vmSpec, final Map<String, String> constraints, final boolean sync)
        throws InsufficientResourcesException, InvalidVMConfigException, VMMException {
        if (this.getVMImageStore().lookUpByUUID(vmSpec.getVmImageUUID()) == null) {
            throw new InvalidVMConfigException("Illegal VMImage UUID " + vmSpec.getVmImageUUID());
        }
        HostMXBean candidateHost = this.placementPolicy.placeVM(this, vmSpec);
        if (candidateHost == null) {
            throw new InsufficientResourcesException();
        }
        return candidateHost.createVM(vmSpec, sync);
    }

    /*
     * @see org.ow2.jasmine.vmm.api.ServerPoolMXBean#getVMImageStore()
     */
    abstract public VirtualMachineImageStoreMXBean getVMImageStore();

    /*
     * @see org.ow2.jasmine.vmm.api.ServerPoolMXBean#getManagedHosts()
     */
    public List<HostMXBean> getManagedHosts() {
        return this.hosts;
    }

}
