package org.ow2.jasmine.vmm.agent.driver.hyperv;

import java.util.HashMap;
import java.util.HashSet;
import java.util.Set;
import java.util.UUID;

import javax.management.InstanceNotFoundException;

import org.apache.log4j.Logger;
import org.jinterop.dcom.common.JIException;
import org.jinterop.dcom.core.IJIComObject;
import org.jinterop.dcom.core.JIArray;
import org.jinterop.dcom.core.JIComServer;
import org.jinterop.dcom.core.JIProgId;
import org.jinterop.dcom.core.JISession;
import org.jinterop.dcom.core.JIString;
import org.jinterop.dcom.core.JIVariant;
import org.jinterop.dcom.impls.JIObjectFactory;
import org.jinterop.dcom.impls.automation.IJIDispatch;
import org.jinterop.dcom.impls.automation.IJIEnumVariant;
import org.ow2.jasmine.vmm.api.BadVMPowerStateException;

import com.hyper9.jwbem.SWbemLocator;
import com.hyper9.jwbem.SWbemMethod;
import com.hyper9.jwbem.SWbemObject;
import com.hyper9.jwbem.SWbemObjectSet;
import com.hyper9.jwbem.SWbemServices;
import com.hyper9.jwbem.msvm.MsvmObject;
import com.hyper9.jwbem.msvm.virtualsystem.MsvmComputerSystem;
import com.hyper9.jwbem.msvm.virtualsystem.MsvmVirtualSystemSettingData;
import com.hyper9.jwbem.msvm.virtualsystemmanagement.MsvmVirtualSystemManagementService;

public class HyperVConnection {
    static Logger logger = Logger.getLogger(HyperVConnection.class);

    private String hostname, username, password;

    private IJIDispatch services_dispatch;

    private SWbemServices services = null;

    public HyperVConnection(final String hostName, final String username, final String password) {
        this.username = username;
        this.password = password;
        this.hostname = hostName;

        this.connectToHyperv();
        this.connectToWin32();

    }

    // Connexion au nameSpace "ROOT\\Virtualization"
    public synchronized void connectToHyperv() {
        HyperVConnection.logger.debug("connectHyperV");

        SWbemLocator locator = new SWbemLocator();
        try {
            this.services = locator.connect(this.hostname, this.hostname, "ROOT\\Virtualization", this.username, this.password);
        } catch (Exception ex) {
            HyperVConnection.logger.error("Failed to establish Hyper-V-API connection with host " + this.hostname
                + " with login " + this.username, ex);
        }
        // this.locator.session.setGlobalSocketTimeout(0);
    }

    // Connexion au nameSpace "ROOT\\CIMV2"
    public synchronized void connectToWin32() {
        HyperVConnection.logger.debug("connectToWin32");

        try {
            JISession session = JISession.createSession(this.hostname, this.username, this.password);
            session.useSessionSecurity(true);
            // this.session.setGlobalSocketTimeout(30000);
            JIComServer comStub = new JIComServer(JIProgId.valueOf("WbemScripting.SWbemLocator"), this.hostname, session);
            IJIComObject unknown = comStub.createInstance();
            IJIComObject comObject = unknown.queryInterface("76A6415B-CB41-11d1-8B02-00600806D9B6");// ISWbemLocator
            // This will obtain the dispatch interface
            IJIDispatch idispatch = (IJIDispatch) JIObjectFactory.narrowObject(comObject.queryInterface(IJIDispatch.IID));
            JIVariant results[] = idispatch.callMethodA("ConnectServer", new Object[] {new JIString(this.hostname),
                new JIString("ROOT\\CIMV2"), JIVariant.OPTIONAL_PARAM(), JIVariant.OPTIONAL_PARAM(),
                JIVariant.OPTIONAL_PARAM(), JIVariant.OPTIONAL_PARAM(), new Integer(0), JIVariant.OPTIONAL_PARAM()});

            this.services_dispatch = (IJIDispatch) JIObjectFactory.narrowObject((results[0]).getObjectAsComObject());

        } catch (Exception ex) {
            HyperVConnection.logger.error("Failed to establish Hyper-V-API connection with host " + this.hostname
                + " with login " + this.username, ex);
        }

    }

    // Fermeture des connexions
    public synchronized void closeConnection() {
        HyperVConnection.logger.debug("close");
        if (this.services_dispatch.getAssociatedSession() != null) {
            try {
                JISession.destroySession(this.services_dispatch.getAssociatedSession());
            } catch (JIException ex) {
                HyperVConnection.logger.error("Failed to close Hyper-V-API connection");
            }
        }
        if (this.services.getLocator() != null) {
            this.services.getLocator().disconnect();
        }
    }

    private synchronized void testConnectionHyperv() {
        try {
            String wql = "SELECT * FROM Msvm_VirtualSystemManagementService";
            this.services.execQuery(wql, MsvmObject.class);
        } catch (Exception ex) {
            HyperVConnection.logger.debug("Re-open the 'ROOT\\Virtualization' connection");
            this.connectToHyperv();
        }
    }

    private synchronized void testConnectionWin32() {
        try {
            String wql = "SELECT * FROM Win32_Processor";
            this.services_dispatch.callMethodA("ExecQuery", new Object[] {new JIString(wql), new JIString("WQL"),
                JIVariant.OPTIONAL_PARAM(), JIVariant.OPTIONAL_PARAM()});
        } catch (Exception ex) {
            HyperVConnection.logger.debug("Re-open the 'ROOT\\CIMV2' connection");
            this.connectToWin32();
        }
    }

    public synchronized Set<VM> getAllVM() throws JIException {
        HyperVConnection.logger.info("getAllVM");
        Set<VM> setVM = new HashSet<VM>();

        String wql = "SELECT * FROM Msvm_ComputerSystem WHERE Caption = 'Virtual machine'";

        this.testConnectionHyperv();

        SWbemObjectSet<MsvmComputerSystem> compSysSet = this.services.execQuery(wql, MsvmComputerSystem.class);

        for (MsvmComputerSystem cs : compSysSet) {
            VM vm = new VM();
            vm.setNameLabel(cs.getElementName());
            vm.setUuid(cs.getName());
            setVM.add(vm);
        }

        return setVM;
    }

    public synchronized boolean startVM(final HyperVHost host, final String ElementName) throws Exception {
        HyperVConnection.logger.info("startVM");

        return this.changeState(VmState.Enabled, host, ElementName);
    }

    public synchronized boolean shutdownVM(final HyperVHost host, final String ElementName) throws Exception {
        HyperVConnection.logger.info("shutdownVM");

        return this.changeState(VmState.Disabled, host, ElementName);
    }

    public synchronized boolean rebootVM(final HyperVHost host, final String ElementName) throws Exception {
        HyperVConnection.logger.info("rebootVM");

        return this.changeState(VmState.Reboot, host, ElementName);
    }

    public synchronized boolean suspendVM(final HyperVHost host, final String ElementName) throws Exception {
        HyperVConnection.logger.info("suspendVM");

        return this.changeState(VmState.Suspended, host, ElementName);
    }

    public synchronized boolean destroyVM(final HyperVHost host, final String ElementName) throws Exception {
        HyperVConnection.logger.info("destroyVM");
        String wql = "SELECT * FROM Msvm_ComputerSystem WHERE ElementName = '" + ElementName + "'";

        this.testConnectionHyperv();

        SWbemObjectSet<MsvmComputerSystem> compSysSet = this.services.execQuery(wql, MsvmComputerSystem.class);

        if (compSysSet.getSize() != 1) {
            HyperVConnection.logger.debug("VM " + ElementName + " missing on " + host);
            throw new Exception("VM " + ElementName + " missing on " + host);
        }

        MsvmComputerSystem cs = compSysSet.iterator().next();

        wql = "SELECT * FROM Msvm_VirtualSystemManagementService";
        SWbemObjectSet<MsvmVirtualSystemManagementService> virtualSystemSet = this.services.execQuery(wql,
            MsvmVirtualSystemManagementService.class);

        MsvmVirtualSystemManagementService vs = virtualSystemSet.iterator().next();
        SWbemMethod destroyVirtualSystemMethod = null;

        for (final SWbemMethod m : vs.getMethods()) {
            if (m.getName().equals("DestroyVirtualSystem")) {
                destroyVirtualSystemMethod = m;
            }
        }

        SWbemObject inParams = destroyVirtualSystemMethod.getInParameters();

        inParams.getObjectDispatcher().put("ComputerSystem", new JIVariant(cs.getObjectPath().getPath(), true));

        Object[] methodParams = new Object[] {new JIString("DestroyVirtualSystem"),
            new JIVariant(inParams.getObjectDispatcher()), new Integer(0), JIVariant.NULL(),};

        JIVariant variant3 = vs.getObjectDispatcher().callMethodA("ExecMethod_", methodParams)[0];

        IJIDispatch wbemProperty_dispatch = (IJIDispatch) JIObjectFactory.narrowObject((variant3).getObjectAsComObject());
        JIVariant returnValue = wbemProperty_dispatch.get("ReturnValue");
        int resultat = returnValue.getObjectAsInt();

        HyperVConnection.logger.debug("resultat = " + resultat);
        if (resultat == 0) {
            return true;
        } else {
            return false;
        }
    }

    private synchronized boolean changeState(final int state, final HyperVHost host, final String ElementName) throws Exception {
        String wql = "SELECT * FROM Msvm_ComputerSystem WHERE ElementName = '" + ElementName + "'";

        this.testConnectionHyperv();

        SWbemObjectSet<MsvmComputerSystem> compSysSet = this.services.execQuery(wql, MsvmComputerSystem.class);

        if (compSysSet.getSize() != 1) {
            HyperVConnection.logger.debug("VM " + ElementName + " missing on " + host);
            throw new Exception("VM " + ElementName + " missing on " + host);
        }
        MsvmComputerSystem cs = compSysSet.iterator().next();
        SWbemMethod requestStateChange = null;

        for (final SWbemMethod m : cs.getMethods()) {
            if (m.getName().equals("RequestStateChange")) {
                requestStateChange = m;
            }
        }

        SWbemObject inParams = requestStateChange.getInParameters();
        inParams.getObjectDispatcher().put("RequestedState", new JIVariant(state));
        inParams.getObjectDispatcher().put("TimeoutPeriod", JIVariant.NULL());

        Object[] methodParams = new Object[] {new JIString("RequestStateChange"),
            new JIVariant(inParams.getObjectDispatcher()), new Integer(0), JIVariant.NULL(),};

        JIVariant result = cs.getObjectDispatcher().callMethodA("ExecMethod_", methodParams)[0];
        IJIDispatch dispatch = (IJIDispatch) JIObjectFactory.narrowObject(result.getObjectAsComObject());
        JIVariant returnValue = dispatch.get("ReturnValue");
        int resultat = returnValue.getObjectAsInt();

        HyperVConnection.logger.debug("resultat = " + resultat);
        if (resultat != 0) {
            if (resultat == 32775) {

                throw new BadVMPowerStateException("Bad power state for VM " + ElementName);
            }
            if (resultat == 4096) {
                // String job = dispatch.get("Job").getObjectAsString2();
                // JIVariant var =
                // this.services.getObjectDispatcher().callMethodA("Get",
                // new Object[] {new JIVariant(job), JIVariant.OPTIONAL_PARAM(),
                // JIVariant.OPTIONAL_PARAM()})[0];
                //
                // IJIDispatch dispach = (IJIDispatch)
                // JIObjectFactory.narrowObject(var.getObjectAsComObject());
                // System.out.println(new MsvmVirtualSystemSettingData(dispach,
                // this.services).getObjectText());

                return true;
            }

            return false;
        }

        return true;
    }

    public synchronized boolean createVM(final String vmName, final String VHDPath) throws JIException {
        HyperVConnection.logger.info("createVM");

        String wql = "SELECT * FROM Msvm_VirtualSystemManagementService";

        this.testConnectionHyperv();

        SWbemObjectSet<MsvmVirtualSystemManagementService> virtualSystemSet = this.services.execQuery(wql,
            MsvmVirtualSystemManagementService.class);
        MsvmVirtualSystemManagementService virtualSystem = virtualSystemSet.iterator().next();

        JIVariant[] var = this.services.getObjectDispatcher().callMethodA(
            "Get",
            new Object[] {new JIVariant("Msvm_VirtualSystemGlobalSettingData"), JIVariant.OPTIONAL_PARAM(),
                JIVariant.OPTIONAL_PARAM()});

        IJIDispatch dispach = (IJIDispatch) JIObjectFactory.narrowObject(var[0].getObjectAsComObject());

        JIVariant varnt = dispach.callMethodA("SpawnInstance_");
        IJIDispatch dispath = (IJIDispatch) JIObjectFactory.narrowObject(varnt.getObjectAsComObject());
        MsvmObject globalSettingData = new MsvmObject(dispath, this.services);

        globalSettingData.getObjectDispatcher().put("ElementName", new JIVariant(vmName));

        if (!this.defineVirtualSystem(virtualSystem, globalSettingData)) {

            return false;
        }

        wql = "SELECT * FROM Msvm_ComputerSystem WHERE ElementName = '" + vmName + "'";
        SWbemObjectSet<MsvmComputerSystem> compSysSet = this.services.execQuery(wql, MsvmComputerSystem.class);
        MsvmComputerSystem compSys = compSysSet.iterator().next();

        wql = "SELECT * FROM Msvm_VirtualSystemSettingData WHERE ElementName = '" + vmName + "'";
        SWbemObjectSet<MsvmVirtualSystemSettingData> virtualSettingSet = this.services.execQuery(wql,
            MsvmVirtualSystemSettingData.class);
        MsvmVirtualSystemSettingData virtualSettingData = virtualSettingSet.iterator().next();

        if (!this.associateVHD(vmName, VHDPath, virtualSystem, compSys, virtualSettingData)) {

            return false;
        }
        if (!this.addNetworkAdapter(vmName, virtualSystem, compSys)) {

            return false;
        }

        return true;
    }

    private synchronized boolean defineVirtualSystem(final MsvmVirtualSystemManagementService virtualSystem,
        final MsvmObject globalSettingData) throws JIException {
        SWbemMethod defineVirtualSystem = null;

        for (final SWbemMethod m : virtualSystem.getMethods()) {
            if (m.getName().equals("DefineVirtualSystem")) {
                defineVirtualSystem = m;
            }
        }

        JIVariant[] varnt = globalSettingData.getObjectDispatcher().callMethodA("GetText_",
            new Object[] {new JIVariant(1), JIVariant.OPTIONAL_PARAM(), JIVariant.OPTIONAL_PARAM()});
        String settingDataText = varnt[0].getObjectAsString().getString();
        SWbemObject inParams = defineVirtualSystem.getInParameters();

        inParams.getObjectDispatcher().put("SystemSettingdata", new JIVariant(settingDataText));
        inParams.getObjectDispatcher().put("ResourceSettingData", JIVariant.OPTIONAL_PARAM());
        inParams.getObjectDispatcher().put("SourceSetting", JIVariant.OPTIONAL_PARAM());

        Object[] methodParams = new Object[] {new JIString("DefineVirtualSystem"),
            new JIVariant(inParams.getObjectDispatcher()), new Integer(0), JIVariant.NULL(),};

        JIVariant result = virtualSystem.getObjectDispatcher().callMethodA("ExecMethod_", methodParams)[0];
        IJIDispatch dispatch = (IJIDispatch) JIObjectFactory.narrowObject(result.getObjectAsComObject());
        int resultat = dispatch.get("ReturnValue").getObjectAsInt();

        HyperVConnection.logger.debug("resultat = " + resultat);
        if (resultat == 0) {
            return true;
        }
        return false;

    }

    private synchronized boolean associateVHD(final String vmName, final String VHDPath,
        final MsvmVirtualSystemManagementService virtualSystem, final MsvmComputerSystem compSys,
        final MsvmVirtualSystemSettingData virtualSettingData) throws JIException {
        HyperVConnection.logger.info("associateVHD");

        // Step 2

        JIVariant[] varnt = this.services.getObjectDispatcher().callMethodA(
            "AssociatorsOf",
            new Object[] {new JIVariant(virtualSettingData.getObjectPath().getPath()), JIVariant.OPTIONAL_PARAM(),
                new JIVariant("Msvm_ResourceAllocationSettingData")});

        IJIDispatch wbemObjectSet_dispatch = (IJIDispatch) JIObjectFactory.narrowObject((varnt[0]).getObjectAsComObject());

        JIVariant variant = wbemObjectSet_dispatch.get("_NewEnum");
        IJIComObject object2 = JIObjectFactory.narrowObject(variant.getObjectAsComObject());
        IJIEnumVariant enumVARIANT = (IJIEnumVariant) JIObjectFactory.narrowObject(object2.queryInterface(IJIEnumVariant.IID));

        JIVariant Count = wbemObjectSet_dispatch.get("Count");
        int count = Count.getObjectAsInt();
        for (int i = 0; i < count; i++) {
            Object[] values = enumVARIANT.next(1);
            JIArray array = (JIArray) values[0];
            Object[] arrayObj = (Object[]) array.getArrayInstance();
            for (Object element : arrayObj) {
                JIVariant rasdVariant = (JIVariant) element;
                IJIDispatch rasd_dispatch = (IJIDispatch) JIObjectFactory.narrowObject((rasdVariant).getObjectAsComObject());
                JIVariant addressVariant = rasd_dispatch.get("Address");

                if (addressVariant.getType() != JIVariant.VT_NULL && addressVariant.getObjectAsString2().equals("0")
                    && rasd_dispatch.get("ResourceSubType").getObjectAsString2().equals("Microsoft Emulated IDE Controller")) {

                    MsvmObject IDEController = new MsvmObject(rasd_dispatch, this.services);

                    // Step 3

                    MsvmObject diskRASDDefault = this.newVMRasd("Microsoft Synthetic Disk Drive");

                    diskRASDDefault.getObjectDispatcher().put("Parent", new JIVariant(IDEController.getObjectPath().getPath()));
                    diskRASDDefault.getObjectDispatcher().put("Address", new JIVariant("0"));

                    JIVariant[] varint = diskRASDDefault.getObjectDispatcher().callMethodA("GetText_",
                        new Object[] {new JIVariant(1), JIVariant.OPTIONAL_PARAM(), JIVariant.OPTIONAL_PARAM()});
                    String diskRASDDefaultText = varint[0].getObjectAsString().getString();

                    SWbemMethod addVirtualSystemResources = null;

                    for (final SWbemMethod m : virtualSystem.getMethods()) {
                        if (m.getName().equals("AddVirtualSystemResources")) {
                            addVirtualSystemResources = m;
                            break;
                        }
                    }

                    SWbemObject inParams = addVirtualSystemResources.getInParameters();

                    inParams.getObjectDispatcher().put("TargetSystem", new JIVariant(compSys.getObjectPath().getPath()));
                    inParams.getObjectDispatcher().put("ResourceSettingData",
                        new JIVariant(new JIArray(new JIString[] {new JIString(diskRASDDefaultText)}, true)));

                    Object[] methodParams = new Object[] {new JIString("AddVirtualSystemResources"),
                        new JIVariant(inParams.getObjectDispatcher()), new Integer(0), JIVariant.NULL(),};

                    JIVariant result = virtualSystem.getObjectDispatcher().callMethodA("ExecMethod_", methodParams)[0];

                    IJIDispatch dispach1 = (IJIDispatch) JIObjectFactory.narrowObject(result.getObjectAsComObject());

                    JIVariant returnValue = dispach1.get("ReturnValue");
                    int resultat = returnValue.getObjectAsInt();

                    HyperVConnection.logger.debug("resultat = " + resultat);
                    if (resultat != 0) {
                        return false;
                    }

                    JIVariant NewResourcesArray = dispach1.get("NewResources");

                    JIVariant newResources = ((JIVariant[]) NewResourcesArray.getObjectAsArray().getArrayInstance())[0];

                    String newDiskDrivePath = newResources.getObjectAsString2();

                    MsvmObject VHDRASDDefault = this.newVMRasd("Microsoft Virtual Hard Disk");

                    VHDRASDDefault.getObjectDispatcher().put("Parent", new JIVariant(newDiskDrivePath));
                    VHDRASDDefault.getObjectDispatcher().put("Connection",
                        new JIVariant(new JIArray(new JIString[] {new JIString(VHDPath)}, true)));

                    varint = VHDRASDDefault.getObjectDispatcher().callMethodA("GetText_",
                        new Object[] {new JIVariant(1), JIVariant.OPTIONAL_PARAM(), JIVariant.OPTIONAL_PARAM()});
                    String VHDRASDDefaultText = varint[0].getObjectAsString().getString();

                    inParams = addVirtualSystemResources.getInParameters();

                    inParams.getObjectDispatcher().put("TargetSystem", new JIVariant(compSys.getObjectPath().getPath()));
                    inParams.getObjectDispatcher().put("ResourceSettingData",
                        new JIVariant(new JIArray(new JIString[] {new JIString(VHDRASDDefaultText)}, true)));

                    methodParams = new Object[] {new JIString("AddVirtualSystemResources"),
                        new JIVariant(inParams.getObjectDispatcher()), new Integer(0), JIVariant.NULL(),};

                    result = virtualSystem.getObjectDispatcher().callMethodA("ExecMethod_", methodParams)[0];
                    IJIDispatch dispatch = (IJIDispatch) JIObjectFactory.narrowObject(result.getObjectAsComObject());
                    returnValue = dispatch.get("ReturnValue");
                    resultat = returnValue.getObjectAsInt();

                    HyperVConnection.logger.debug("resultat = " + resultat);
                    if (resultat != 0) {
                        return false;
                    }
                    return true;
                }
            }
        }

        return true;
    }

    private synchronized boolean addNetworkAdapter(final String vmName, final MsvmVirtualSystemManagementService virtualSystem,
        final MsvmComputerSystem compSys) throws JIException {

        HyperVConnection.logger.info("addNetworkAdapter");

        String wql = "SELECT * FROM Msvm_VirtualSwitchManagementService";
        SWbemObjectSet<MsvmObject> virtualSwitchMngtSet = this.services.execQuery(wql, MsvmObject.class);
        MsvmObject virtualSwitchMngt = virtualSwitchMngtSet.iterator().next();

        wql = "SELECT * FROM Msvm_VirtualSwitch";
        SWbemObjectSet<MsvmObject> virtualSwitchSet = this.services.execQuery(wql, MsvmObject.class);
        MsvmObject virtualSwitch = virtualSwitchSet.iterator().next();

        SWbemMethod createSwitchPort = null;

        for (final SWbemMethod m : virtualSwitchMngt.getMethods()) {
            if (m.getName().equals("CreateSwitchPort")) {
                createSwitchPort = m;
                break;
            }
        }

        SWbemObject inParams = createSwitchPort.getInParameters();
        inParams.getObjectDispatcher().put("VirtualSwitch", new JIVariant(virtualSwitch.getObjectPath().getPath(), true));

        UUID uuid = UUID.randomUUID();
        String randomUUIDString = uuid.toString();

        inParams.getObjectDispatcher().put("FriendlyName", new JIVariant(randomUUIDString));
        inParams.getObjectDispatcher().put("Name", new JIVariant(randomUUIDString));

        Object[] methodParams = new Object[] {new JIString("CreateSwitchPort"), new JIVariant(inParams.getObjectDispatcher()),
            new Integer(0), JIVariant.NULL(),};

        JIVariant result = virtualSwitchMngt.getObjectDispatcher().callMethodA("ExecMethod_", methodParams)[0];

        IJIDispatch wbemObject_dispatch = (IJIDispatch) JIObjectFactory.narrowObject(result.getObjectAsComObject());
        JIVariant returnValue = wbemObject_dispatch.get("ReturnValue");
        int resultat = returnValue.getObjectAsInt();

        HyperVConnection.logger.debug("resultat = " + resultat);
        if (resultat != 0) {
            return false;
        }

        String createdSwitch = wbemObject_dispatch.get("CreatedSwitchPort").getObjectAsString2();

        // Step 3
        MsvmObject newRASDDefault = null;
        if (HyperVServerPool.LEGACY_NETWORK_ADAPTER) {
            newRASDDefault = this.newVMRasd("Microsoft Emulated Ethernet Port");

            newRASDDefault.getObjectDispatcher().put("ElementName", new JIVariant(HyperVServerPool.LEGACY_NETWORK_NAME));
        } else {
            newRASDDefault = this.newVMRasd("Microsoft Synthetic Ethernet Port");

            UUID guid = UUID.randomUUID();
            String randomGUIDString = guid.toString();
            newRASDDefault.getObjectDispatcher().put("VirtualSystemIdentifiers",
                new JIVariant(new JIArray(new JIVariant[] {new JIVariant("{" + randomGUIDString + "}")}, true)));
            newRASDDefault.getObjectDispatcher().put("ElementName", new JIVariant(HyperVServerPool.SYNTHETIC_NETWORK_NAME));
        }

        newRASDDefault.getObjectDispatcher().put("Connection",
            new JIVariant(new JIArray(new JIVariant[] {new JIVariant(createdSwitch)}, true)));

        JIVariant[] varint = newRASDDefault.getObjectDispatcher().callMethodA("GetText_",
            new Object[] {new JIVariant(1), JIVariant.OPTIONAL_PARAM(), JIVariant.OPTIONAL_PARAM()});
        String newRASDDefaultText = varint[0].getObjectAsString().getString();

        SWbemMethod addVirtualSystemResources = null;

        for (final SWbemMethod m : virtualSystem.getMethods()) {
            if (m.getName().equals("AddVirtualSystemResources")) {
                addVirtualSystemResources = m;
                break;
            }
        }

        inParams = addVirtualSystemResources.getInParameters();

        inParams.getObjectDispatcher().put("TargetSystem", new JIVariant(compSys.getObjectPath().getPath()));
        inParams.getObjectDispatcher().put("ResourceSettingData",
            new JIVariant(new JIArray(new JIString[] {new JIString(newRASDDefaultText)}, true)));

        methodParams = new Object[] {new JIString("AddVirtualSystemResources"), new JIVariant(inParams.getObjectDispatcher()),
            new Integer(0), JIVariant.NULL(),};

        result = virtualSystem.getObjectDispatcher().callMethodA("ExecMethod_", methodParams)[0];

        wbemObject_dispatch = (IJIDispatch) JIObjectFactory.narrowObject(result.getObjectAsComObject());
        returnValue = wbemObject_dispatch.get("ReturnValue");
        resultat = returnValue.getObjectAsInt();

        HyperVConnection.logger.debug("resultat = " + resultat);
        if (resultat != 0) {
            return false;
        }

        return true;
    }

    private synchronized MsvmObject newVMRasd(final String resourceSubType) throws JIException {
        HyperVConnection.logger.debug("newVMRasd");

        String wql = "SELECT * FROM Msvm_AllocationCapabilities WHERE ResourceSubtype='" + resourceSubType + "'";
        SWbemObjectSet<MsvmObject> allocCapSet = this.services.execQuery(wql, MsvmObject.class);
        MsvmObject allocCap = allocCapSet.iterator().next();

        String allocCapPath = allocCap.getObjectPath().getPath().replace("\\", "\\\\");
        wql = "SELECT * FROM Msvm_SettingsDefineCapabilities WHERE valueRange=0 AND GroupComponent='" + allocCapPath + "'";
        SWbemObjectSet<MsvmObject> setDefCapSet = this.services.execQuery(wql, MsvmObject.class);
        MsvmObject setDefCap = setDefCapSet.iterator().next();

        String partComponent = setDefCap.getObjectDispatcher().get("PartComponent").getObjectAsString2();

        JIVariant[] var = this.services.getObjectDispatcher().callMethodA("Get",
            new Object[] {new JIVariant(partComponent), JIVariant.OPTIONAL_PARAM(), JIVariant.OPTIONAL_PARAM()});

        IJIDispatch dispach = (IJIDispatch) JIObjectFactory.narrowObject(var[0].getObjectAsComObject());
        return new MsvmObject(dispach, this.services);

    }

    public synchronized int getEnabledState(final HyperVHost host, final String ElementName) throws Exception {
        HyperVConnection.logger.info("getEnabledState");

        String wql = "SELECT * FROM Msvm_ComputerSystem WHERE ElementName = '" + ElementName + "'";

        this.testConnectionHyperv();

        SWbemObjectSet<MsvmComputerSystem> compSysSet = this.services.execQuery(wql, MsvmComputerSystem.class);

        if (compSysSet.getSize() != 1) {
            HyperVConnection.logger.debug("VM " + ElementName + " missing on " + host);
            throw new Exception("VM " + ElementName + " missing on " + host);
        }
        MsvmComputerSystem cs = compSysSet.iterator().next();
        int state = cs.getEnabledState();

        HyperVConnection.logger.debug("VM " + ElementName + " state result : " + state);

        return state;
    }

    public synchronized boolean setCPUCount(final String ElementName, final int numVCPU) throws Exception {
        HyperVConnection.logger.info("setCPUCount");

        this.testConnectionHyperv();

        String wql = "SELECT * FROM Msvm_ComputerSystem WHERE ElementName = '" + ElementName + "'";
        SWbemObjectSet<MsvmComputerSystem> compSysSet = this.services.execQuery(wql, MsvmComputerSystem.class);
        MsvmComputerSystem compSys = compSysSet.iterator().next();

        // Turning off the VM
        int state = compSys.getEnabledState();
        if (state != VmState.Disabled) {
            compSys.requestStateChange(VmState.Disabled);
        }

        MsvmObject processorSettingData = this.associatorsOf(ElementName, "Msvm_ProcessorSettingData");

        processorSettingData.getObjectDispatcher().put("VirtualQuantity", new JIVariant(numVCPU));

        JIVariant[] varint = processorSettingData.getObjectDispatcher().callMethodA("GetText_",
            new Object[] {new JIVariant(1), JIVariant.OPTIONAL_PARAM(), JIVariant.OPTIONAL_PARAM()});
        String processorSettingDataText = varint[0].getObjectAsString().getString();

        boolean result = this.modifySystemResources(compSys, new JIArray(
            new JIString[] {new JIString(processorSettingDataText)}, true));

        compSys.requestStateChange(state);

        return result;

    }

    public synchronized boolean setMemorySize(final String ElementName, final long memorySizeMB) throws Exception {
        HyperVConnection.logger.info("setMemorySize");

        this.testConnectionHyperv();

        String wql = "SELECT * FROM Msvm_ComputerSystem WHERE ElementName = '" + ElementName + "'";
        SWbemObjectSet<MsvmComputerSystem> compSysSet = this.services.execQuery(wql, MsvmComputerSystem.class);
        MsvmComputerSystem compSys = compSysSet.iterator().next();

        // Turning off the VM
        int state = compSys.getEnabledState();
        if (state != VmState.Disabled) {
            compSys.requestStateChange(VmState.Disabled);
        }

        MsvmObject memorySettingData = this.associatorsOf(ElementName, "Msvm_MemorySettingData");

        memorySettingData.getObjectDispatcher().put("VirtualQuantity", new JIVariant(memorySizeMB));

        JIVariant[] varint = memorySettingData.getObjectDispatcher().callMethodA("GetText_",
            new Object[] {new JIVariant(1), JIVariant.OPTIONAL_PARAM(), JIVariant.OPTIONAL_PARAM()});
        String memorySettingDataText = varint[0].getObjectAsString().getString();

        boolean result = this.modifySystemResources(compSys, new JIArray(new JIString[] {new JIString(memorySettingDataText)},
            true));

        compSys.requestStateChange(state);

        return result;
    }

    public synchronized boolean setSchedulingCap(final String ElementName, final int schedulingCap) throws Exception {
        HyperVConnection.logger.info("setSchedulingCap");

        this.testConnectionHyperv();

        String wql = "SELECT * FROM Msvm_ComputerSystem WHERE ElementName = '" + ElementName + "'";
        SWbemObjectSet<MsvmComputerSystem> compSysSet = this.services.execQuery(wql, MsvmComputerSystem.class);
        MsvmComputerSystem compSys = compSysSet.iterator().next();

        // Turning off the VM
        int state = compSys.getEnabledState();
        if (state != VmState.Disabled) {
            compSys.requestStateChange(VmState.Disabled);
        }

        MsvmObject processorSettingData = this.associatorsOf(ElementName, "Msvm_ProcessorSettingData");

        processorSettingData.getObjectDispatcher().put("Limit", new JIVariant(schedulingCap));

        JIVariant[] varint = processorSettingData.getObjectDispatcher().callMethodA("GetText_",
            new Object[] {new JIVariant(1), JIVariant.OPTIONAL_PARAM(), JIVariant.OPTIONAL_PARAM()});
        String processorSettingDataText = varint[0].getObjectAsString().getString();

        boolean result = this.modifySystemResources(compSys, new JIArray(
            new JIString[] {new JIString(processorSettingDataText)}, true));

        compSys.requestStateChange(state);

        return result;
    }

    public synchronized boolean setSchedulingWeight(final String ElementName, final int schedulingWeight) throws Exception {
        HyperVConnection.logger.info("setSchedulingWeight");

        this.testConnectionHyperv();

        String wql = "SELECT * FROM Msvm_ComputerSystem WHERE ElementName = '" + ElementName + "'";
        SWbemObjectSet<MsvmComputerSystem> compSysSet = this.services.execQuery(wql, MsvmComputerSystem.class);
        MsvmComputerSystem compSys = compSysSet.iterator().next();

        // Turning off the VM
        int state = compSys.getEnabledState();
        if (state != VmState.Disabled) {
            compSys.requestStateChange(VmState.Disabled);
        }

        MsvmObject processorSettingData = this.associatorsOf(ElementName, "Msvm_ProcessorSettingData");

        processorSettingData.getObjectDispatcher().put("Weight", new JIVariant(schedulingWeight));

        JIVariant[] varint = processorSettingData.getObjectDispatcher().callMethodA("GetText_",
            new Object[] {new JIVariant(1), JIVariant.OPTIONAL_PARAM(), JIVariant.OPTIONAL_PARAM()});
        String processorSettingDataText = varint[0].getObjectAsString().getString();

        boolean result = this.modifySystemResources(compSys, new JIArray(
            new JIString[] {new JIString(processorSettingDataText)}, true));

        compSys.requestStateChange(state);

        return result;
    }

    private synchronized MsvmObject associatorsOf(final String ElementName, final String classeName) throws JIException {

        String wql = "SELECT * FROM Msvm_VirtualSystemSettingData WHERE ElementName = '" + ElementName + "'";
        SWbemObjectSet<MsvmComputerSystem> imgMngmtServSet = this.services.execQuery(wql, MsvmComputerSystem.class);
        MsvmComputerSystem virtualSetting = imgMngmtServSet.iterator().next();

        JIVariant[] varnt = this.services.getObjectDispatcher().callMethodA(
            "AssociatorsOf",
            new Object[] {new JIVariant(virtualSetting.getObjectPath().getPath()), JIVariant.OPTIONAL_PARAM(),
                new JIVariant(classeName)});

        IJIDispatch wbemObjectSet_dispatch = (IJIDispatch) JIObjectFactory.narrowObject((varnt[0]).getObjectAsComObject());

        JIVariant variant = wbemObjectSet_dispatch.get("_NewEnum");
        IJIComObject object2 = JIObjectFactory.narrowObject(variant.getObjectAsComObject());
        IJIEnumVariant enumVARIANT = (IJIEnumVariant) JIObjectFactory.narrowObject(object2.queryInterface(IJIEnumVariant.IID));

        Object[] arrayObj = (Object[]) (((JIArray) enumVARIANT.next(1)[0]).getArrayInstance());
        IJIDispatch wbemObject_dispatch = (IJIDispatch) JIObjectFactory.narrowObject(((JIVariant) arrayObj[0])
            .getObjectAsComObject());

        return new MsvmObject(wbemObject_dispatch, this.services);
    }

    private synchronized boolean modifySystemResources(final MsvmComputerSystem compSys, final JIArray resourceSettingDataArray)
        throws JIException {

        String wql = "SELECT * FROM Msvm_VirtualSystemManagementService";
        SWbemObjectSet<MsvmVirtualSystemManagementService> virtualSystemSet = this.services.execQuery(wql,
            MsvmVirtualSystemManagementService.class);
        MsvmVirtualSystemManagementService virtualSystem = virtualSystemSet.iterator().next();

        SWbemMethod modifyVirtualSystemResources = null;

        for (final SWbemMethod m : virtualSystem.getMethods()) {
            if (m.getName().equals("ModifyVirtualSystemResources")) {
                modifyVirtualSystemResources = m;
            }
        }

        SWbemObject inParams = modifyVirtualSystemResources.getInParameters();

        inParams.getObjectDispatcher().put("ComputerSystem", new JIVariant(compSys.getObjectPath().getPath()));
        inParams.getObjectDispatcher().put("ResourceSettingData", new JIVariant(resourceSettingDataArray));

        Object[] methodParams = new Object[] {new JIString("ModifyVirtualSystemResources"),
            new JIVariant(inParams.getObjectDispatcher()), new Integer(0), JIVariant.NULL(),};

        JIVariant result = virtualSystem.getObjectDispatcher().callMethodA("ExecMethod_", methodParams)[0];
        IJIDispatch dispatch = (IJIDispatch) JIObjectFactory.narrowObject(result.getObjectAsComObject());
        JIVariant returnValue = dispatch.get("ReturnValue");
        int resultat = returnValue.getObjectAsInt();

        HyperVConnection.logger.debug("resultat = " + resultat);
        if (resultat != 0) {
            return false;
        }
        return true;

    }

    public synchronized int getSchedulingWeight(final String nameLabel) throws JIException {
        HyperVConnection.logger.info("getSchedulingWeight");

        this.testConnectionHyperv();

        MsvmObject processorSettingData = this.associatorsOf(nameLabel, "Msvm_ProcessorSettingData");

        int schedulingWeight = processorSettingData.getObjectDispatcher().get("Weight").getObjectAsInt();

        return schedulingWeight;
    }

    public synchronized int getSchedulingCap(final String nameLabel) throws JIException {
        HyperVConnection.logger.info("getSchedulingCap");

        this.testConnectionHyperv();

        MsvmObject processorSettingData = this.associatorsOf(nameLabel, "Msvm_ProcessorSettingData");

        int schedulingCap = Integer.parseInt(processorSettingData.getObjectDispatcher().get("Limit").getObjectAsString2());

        return schedulingCap;
    }

    public synchronized String getMacAddress(final String nameLabel) throws JIException {
        HyperVConnection.logger.info("getMacAddress");

        this.testConnectionHyperv();

        MsvmObject settingData = null;
        if (HyperVServerPool.LEGACY_NETWORK_ADAPTER) {
            settingData = this.associatorsOf(nameLabel, "Msvm_EmulatedEthernetPortSettingData");
        } else {
            settingData = this.associatorsOf(nameLabel, "Msvm_SyntheticEthernetPortSettingData");
        }
        String macAddress = settingData.getObjectDispatcher().get("Address").getObjectAsString2();

        return macAddress;
    }

    public synchronized String getStartTime(final HyperVHost host, final String ElementName) throws Exception {
        HyperVConnection.logger.info("getStartTime");
        String wql = "SELECT * FROM Msvm_ComputerSystem WHERE ElementName = '" + ElementName + "'";

        this.testConnectionHyperv();

        SWbemObjectSet<MsvmComputerSystem> compSysSet = this.services.execQuery(wql, MsvmComputerSystem.class);

        if (compSysSet.getSize() != 1) {
            HyperVConnection.logger.debug("VM " + ElementName + " missing on " + host);
            throw new InstanceNotFoundException("VM " + ElementName + " missing on " + host);
        }
        MsvmComputerSystem cs = compSysSet.iterator().next();

        String startTime;
        startTime = cs.getProperties().getItem("InstallDate").toString();

        HyperVConnection.logger.debug("VM " + ElementName + " StartTime : " + startTime);

        return startTime;
    }

    public synchronized int getNumberOfProcessors(final String nameLabel) throws JIException {
        HyperVConnection.logger.debug("getNumVCPUs");

        this.testConnectionHyperv();

        JIVariant summary_variant = this.getSummaryInfo(nameLabel)[0];
        IJIDispatch dispatch = (IJIDispatch) JIObjectFactory.narrowObject(summary_variant.getObjectAsComObject());
        int numberOfProcessors = dispatch.get("NumberOfProcessors").getObjectAsInt();

        return numberOfProcessors;
    }

    public synchronized float getCPULoad(final String nameLabel) throws JIException {
        HyperVConnection.logger.debug("getCPULoad");

        this.testConnectionHyperv();

        IJIDispatch dispatch = (IJIDispatch) JIObjectFactory.narrowObject(this.getSummaryInfo(nameLabel)[0]
            .getObjectAsComObject());
        float processorLoad = dispatch.get("ProcessorLoad").getObjectAsInt();
        return processorLoad;
    }

    public synchronized long getMemorySize(final String nameLabel) throws Exception {
        HyperVConnection.logger.debug("getMemorySize");

        this.testConnectionHyperv();

        MsvmObject memorySettingData = this.associatorsOf(nameLabel, "Msvm_MemorySettingData");
        JIVariant memoryVariant = memorySettingData.getObjectDispatcher().get("VirtualQuantity");

        long memorySize = 0;
        if (memoryVariant.getType() != JIVariant.VT_NULL) {
            memorySize = Long.parseLong(memoryVariant.getObjectAsString2());
        }

        return memorySize;
    }

    public synchronized long getMemoryUsage(final String nameLabel) throws JIException {
        HyperVConnection.logger.debug("getMemoryUsage");

        this.testConnectionHyperv();

        JIVariant summary_variant = this.getSummaryInfo(nameLabel)[0];
        IJIDispatch dispatch = (IJIDispatch) JIObjectFactory.narrowObject(summary_variant.getObjectAsComObject());
        JIVariant memoryUsageVariant = dispatch.get("MemoryUsage");

        long memoryUsage = 0;
        if (memoryUsageVariant.getType() != JIVariant.VT_NULL) {
            memoryUsage = Long.parseLong(memoryUsageVariant.getObjectAsString().getString());
        }

        return memoryUsage;
    }

    public synchronized HashMap<String, Float> getVMCPULoads() throws JIException {
        HyperVConnection.logger.debug("getVMCPULoads");
        HashMap<String, Float> cpuLoads = new HashMap<String, Float>();

        this.testConnectionHyperv();

        JIVariant[] summary_variantArray = this.getSummaryInfo(null);

        for (JIVariant element : summary_variantArray) {
            IJIDispatch dispatch = (IJIDispatch) JIObjectFactory.narrowObject(element.getObjectAsComObject());
            float processorLoad = dispatch.get("ProcessorLoad").getObjectAsInt();
            String elementName = dispatch.get("ElementName").getObjectAsString2();
            cpuLoads.put(elementName, processorLoad);
        }
        return cpuLoads;
    }

    // Return an array of JIVariant representing SummaryIformation of the VMs
    private synchronized JIVariant[] getSummaryInfo(final String nameLabel) throws JIException {
        HyperVConnection.logger.debug("getSummaryInfo");

        String wql = "SELECT * FROM Msvm_VirtualSystemManagementService";
        SWbemObjectSet<MsvmVirtualSystemManagementService> virtualSystemSet = this.services.execQuery(wql,
            MsvmVirtualSystemManagementService.class);
        MsvmVirtualSystemManagementService virtualSystem = virtualSystemSet.iterator().next();

        if (nameLabel == null) {
            wql = "SELECT * FROM Msvm_VirtualSystemSettingData";
        } else {
            wql = "SELECT * FROM Msvm_VirtualSystemSettingData WHERE ElementName = '" + nameLabel + "'";
        }
        SWbemObjectSet<MsvmVirtualSystemSettingData> settingDataSet = this.services.execQuery(wql,
            MsvmVirtualSystemSettingData.class);

        JIVariant[] VirtualSystemVariantArray = new JIVariant[settingDataSet.getSize()];
        int i = 0;
        for (MsvmVirtualSystemSettingData vssd : settingDataSet) {
            VirtualSystemVariantArray[i] = new JIVariant(vssd.getObjectPath().getPath());
            i++;
        }

        SWbemMethod getSummaryInformation = null;

        for (final SWbemMethod m : virtualSystem.getMethods()) {
            if (m.getName().equals("GetSummaryInformation")) {
                getSummaryInformation = m;
            }
        }

        SWbemObject inParams = getSummaryInformation.getInParameters();

        inParams.getObjectDispatcher().put("SettingData", new JIVariant(new JIArray(VirtualSystemVariantArray, true)));
        inParams.getObjectDispatcher().put(
            "RequestedInformation",
            new JIVariant(new JIArray(
                new Integer[] {0, 1, 2, 3, 4, 100, 101, 102, 103, 104, 105, 106, 107, 108, 109, 110, 111}, true)));

        Object[] methodParams = new Object[] {new JIString("GetSummaryInformation"),
            new JIVariant(inParams.getObjectDispatcher()), new Integer(0), JIVariant.NULL(),};

        JIVariant result = virtualSystem.getObjectDispatcher().callMethodA("ExecMethod_", methodParams)[0];
        IJIDispatch dispatch = (IJIDispatch) JIObjectFactory.narrowObject(result.getObjectAsComObject());
        JIVariant summaryInformationValue = dispatch.get("SummaryInformation");

        return (JIVariant[]) summaryInformationValue.getObjectAsArray().getArrayInstance();
    }

    public synchronized Set<Template> getHostVMTemplates(final String imageStorePath) throws JIException {
        HyperVConnection.logger.info("getHostVMTemplates");
        Set<Template> setTemplate = new HashSet<Template>();

        String wql = "SELECT * FROM CIM_DataFile WHERE Path = '" + imageStorePath + "' AND FileType = 'vhd File'";

        this.testConnectionWin32();

        JIVariant results[] = this.services_dispatch.callMethodA("ExecQuery", new Object[] {new JIString(wql),
            new JIString("WQL"), JIVariant.OPTIONAL_PARAM(), JIVariant.OPTIONAL_PARAM()});
        IJIDispatch wbemObjectSet_dispatch = (IJIDispatch) JIObjectFactory.narrowObject((results[0]).getObjectAsComObject());

        JIVariant variant = wbemObjectSet_dispatch.get("_NewEnum");
        IJIComObject object = variant.getObjectAsComObject();

        IJIEnumVariant enumVARIANT = (IJIEnumVariant) JIObjectFactory.narrowObject(object.queryInterface(IJIEnumVariant.IID));

        JIVariant Count = wbemObjectSet_dispatch.get("Count");
        int count = Count.getObjectAsInt();

        for (int i = 0; i < count; i++) {
            Template temp = new Template();

            Object[] values = enumVARIANT.next(1);

            JIArray array = (JIArray) values[0];
            Object[] arrayObj = (Object[]) array.getArrayInstance();
            for (Object element : arrayObj) {
                IJIDispatch wbemObject_dispatch = (IJIDispatch) JIObjectFactory.narrowObject(((JIVariant) element)
                    .getObjectAsComObject());
                // Name
                JIVariant elementName = wbemObject_dispatch.get("FileName");
                temp.setName(elementName.getObjectAsString().getString());

                // MetaData
                JIVariant variant2 = (wbemObject_dispatch.callMethodA("GetObjectText_", new Object[] {new Integer(1)}))[0];
                temp.setMetaData(variant2.getObjectAsString().getString());
            }
            setTemplate.add(temp);
        }

        return setTemplate;
    }

    public synchronized boolean hostDeleteFile(final String folderPath, final String name) throws JIException {
        HyperVConnection.logger.info("hostDeleteFile " + folderPath + name);
        String wql = "SELECT * FROM CIM_DataFile WHERE Name = '" + folderPath + name + "'";

        this.testConnectionWin32();

        JIVariant results2[] = this.services_dispatch.callMethodA("ExecQuery", new Object[] {new JIString(wql),
            new JIString("WQL"), JIVariant.OPTIONAL_PARAM(), JIVariant.OPTIONAL_PARAM()});
        IJIDispatch wbemObjectSet_dispatch = (IJIDispatch) JIObjectFactory.narrowObject((results2[0]).getObjectAsComObject());

        JIVariant variant = wbemObjectSet_dispatch.get("_NewEnum");
        IJIComObject object = JIObjectFactory.narrowObject(variant.getObjectAsComObject());
        IJIEnumVariant enumVARIANT = (IJIEnumVariant) JIObjectFactory.narrowObject(object.queryInterface(IJIEnumVariant.IID));

        Object[] arrayObj = (Object[]) (((JIArray) enumVARIANT.next(1)[0]).getArrayInstance());
        IJIDispatch wbemObject_dispatch = (IJIDispatch) JIObjectFactory.narrowObject(((JIVariant) arrayObj[0])
            .getObjectAsComObject());

        Object[] methodParams = new Object[] {new JIString("Delete"), JIVariant.NULL(), new Integer(0), JIVariant.NULL(),};

        JIVariant result = wbemObject_dispatch.callMethodA("ExecMethod_", methodParams)[0];
        IJIDispatch dispatch = (IJIDispatch) JIObjectFactory.narrowObject(result.getObjectAsComObject());
        JIVariant returnValue = dispatch.get("ReturnValue");
        int resultat = returnValue.getObjectAsInt();

        HyperVConnection.logger.debug("resultat = " + resultat);
        if (resultat == 0 || resultat == 4096) {
            return true;
        } else {
            return false;
        }

    }

    public synchronized boolean hostCopyFile(final String originPath, final String destinationPath) throws JIException {
        HyperVConnection.logger.debug("hostCopyFile from " + originPath + " to " + destinationPath);

        String wql = "SELECT * FROM CIM_DataFile WHERE Name = '" + originPath + "'";

        this.testConnectionWin32();

        JIVariant results2[] = this.services_dispatch.callMethodA("ExecQuery", new Object[] {new JIString(wql),
            new JIString("WQL"), JIVariant.OPTIONAL_PARAM(), JIVariant.OPTIONAL_PARAM()});
        IJIDispatch wbemObjectSet_dispatch = (IJIDispatch) JIObjectFactory.narrowObject((results2[0]).getObjectAsComObject());

        JIVariant variant = wbemObjectSet_dispatch.get("_NewEnum");
        IJIComObject object2 = JIObjectFactory.narrowObject(variant.getObjectAsComObject());
        IJIEnumVariant enumVARIANT = (IJIEnumVariant) JIObjectFactory.narrowObject(object2.queryInterface(IJIEnumVariant.IID));

        Object[] arrayObj = (Object[]) (((JIArray) enumVARIANT.next(1)[0]).getArrayInstance());
        IJIDispatch wbemObject_dispatch = (IJIDispatch) JIObjectFactory.narrowObject(((JIVariant) arrayObj[0])
            .getObjectAsComObject());

        JIVariant variant2 = wbemObject_dispatch.get("Methods_");
        IJIDispatch dispatch2 = (IJIDispatch) JIObjectFactory.narrowObject(variant2.getObjectAsComObject());

        JIVariant jiCount = dispatch2.get("Count");
        int count = jiCount.getObjectAsInt();

        JIVariant variant3 = dispatch2.get("_NewEnum");
        IJIEnumVariant enumVARIANT2 = (IJIEnumVariant) JIObjectFactory.narrowObject(variant3.getObjectAsComObject()
            .queryInterface(IJIEnumVariant.IID));

        for (int i = 0; i < count; i++) {
            Object[] values = enumVARIANT2.next(1);

            JIArray array = (JIArray) values[0];
            Object[] arrayObj2 = (Object[]) array.getArrayInstance();
            IJIDispatch dispatch3 = (IJIDispatch) JIObjectFactory.narrowObject(((JIVariant) arrayObj2[0])
                .getObjectAsComObject());

            if (dispatch3.get("Name").getObjectAsString2().equals("Copy")) {
                JIVariant variant4 = dispatch3.get("InParameters");
                IJIDispatch inParamsDispatch = (IJIDispatch) JIObjectFactory.narrowObject(variant4.getObjectAsComObject());

                inParamsDispatch.put("FileName", new JIVariant(destinationPath));

                Object[] methodParams = new Object[] {new JIString("Copy"), new JIVariant(inParamsDispatch), new Integer(0),
                    JIVariant.NULL(),};

                JIVariant result = wbemObject_dispatch.callMethodA("ExecMethod_", methodParams)[0];
                IJIDispatch dispatch = (IJIDispatch) JIObjectFactory.narrowObject(result.getObjectAsComObject());
                JIVariant returnValue = dispatch.get("ReturnValue");
                int resultat = returnValue.getObjectAsInt();

                HyperVConnection.logger.debug("resultat = " + resultat);
                if (resultat != 0) {
                    return false;
                }
                return true;

            }
        }
        return false;
    }

    public synchronized long getHostSizeMB(final String path) throws JIException {
        HyperVConnection.logger.debug("getHostSizeMB");

        String wql = "SELECT * FROM Win32_LogicalDisk WHERE Name = '" + path + "'";

        this.testConnectionWin32();

        JIVariant results2[] = this.services_dispatch.callMethodA("ExecQuery", new Object[] {new JIString(wql),
            new JIString("WQL"), JIVariant.OPTIONAL_PARAM(), JIVariant.OPTIONAL_PARAM()});
        IJIDispatch wbemObjectSet_dispatch = (IJIDispatch) JIObjectFactory.narrowObject((results2[0]).getObjectAsComObject());

        JIVariant variant = wbemObjectSet_dispatch.get("_NewEnum");
        IJIComObject object2 = JIObjectFactory.narrowObject(variant.getObjectAsComObject());
        IJIEnumVariant enumVARIANT = (IJIEnumVariant) JIObjectFactory.narrowObject(object2.queryInterface(IJIEnumVariant.IID));

        Object[] arrayObj = (Object[]) (((JIArray) enumVARIANT.next(1)[0]).getArrayInstance());
        IJIDispatch dispatch = (IJIDispatch) JIObjectFactory.narrowObject(((JIVariant) arrayObj[0]).getObjectAsComObject());
        JIVariant returnValue = dispatch.get("Size");
        long resultat = Long.parseLong(returnValue.getObjectAsString().getString()) / 1048576;

        return resultat;
    }

    public synchronized long getHostFreeSpaceMB(final String path) throws JIException {
        HyperVConnection.logger.debug("getHostFreeSpaceMB");

        String wql = "SELECT * FROM Win32_LogicalDisk WHERE Name = '" + path + "'";

        this.testConnectionWin32();

        JIVariant results2[] = this.services_dispatch.callMethodA("ExecQuery", new Object[] {new JIString(wql),
            new JIString("WQL"), JIVariant.OPTIONAL_PARAM(), JIVariant.OPTIONAL_PARAM()});
        IJIDispatch wbemObjectSet_dispatch = (IJIDispatch) JIObjectFactory.narrowObject((results2[0]).getObjectAsComObject());

        JIVariant variant = wbemObjectSet_dispatch.get("_NewEnum");
        IJIComObject object2 = JIObjectFactory.narrowObject(variant.getObjectAsComObject());
        IJIEnumVariant enumVARIANT = (IJIEnumVariant) JIObjectFactory.narrowObject(object2.queryInterface(IJIEnumVariant.IID));

        Object[] arrayObj = (Object[]) (((JIArray) enumVARIANT.next(1)[0]).getArrayInstance());
        IJIDispatch dispatch = (IJIDispatch) JIObjectFactory.narrowObject(((JIVariant) arrayObj[0]).getObjectAsComObject());
        JIVariant returnValue = dispatch.get("FreeSpace");
        long resultat = Long.parseLong(returnValue.getObjectAsString().getString()) / 1048576;

        return resultat;
    }

    public synchronized int getHostNumCPU() throws JIException {
        HyperVConnection.logger.debug("getHostNumCPU");
        String wql = "SELECT * FROM Win32_Processor";

        this.testConnectionWin32();

        JIVariant results[] = this.services_dispatch.callMethodA("ExecQuery", new Object[] {new JIString(wql),
            new JIString("WQL"), JIVariant.OPTIONAL_PARAM(), JIVariant.OPTIONAL_PARAM()});
        IJIDispatch wbemObjectSet_dispatch = (IJIDispatch) JIObjectFactory.narrowObject((results[0]).getObjectAsComObject());

        int numCPU = wbemObjectSet_dispatch.get("Count").getObjectAsInt();

        return numCPU;
    }

    public synchronized HashMap<String, String> getHostCPUInfo() throws JIException {
        HyperVConnection.logger.debug("getHostCPUInfo");
        String wql = "SELECT * FROM Win32_Processor";

        this.testConnectionWin32();

        JIVariant results[] = this.services_dispatch.callMethodA("ExecQuery", new Object[] {new JIString(wql),
            new JIString("WQL"), JIVariant.OPTIONAL_PARAM(), JIVariant.OPTIONAL_PARAM()});
        IJIDispatch wbemObjectSet_dispatch = (IJIDispatch) JIObjectFactory.narrowObject((results[0]).getObjectAsComObject());

        JIVariant variant = wbemObjectSet_dispatch.get("_NewEnum");
        IJIComObject object2 = JIObjectFactory.narrowObject(variant.getObjectAsComObject());
        IJIEnumVariant enumVARIANT = (IJIEnumVariant) JIObjectFactory.narrowObject(object2.queryInterface(IJIEnumVariant.IID));

        Object[] arrayObj = (Object[]) (((JIArray) enumVARIANT.next(1)[0]).getArrayInstance());
        IJIDispatch wbemObject_dispatch = (IJIDispatch) JIObjectFactory.narrowObject(((JIVariant) arrayObj[0])
            .getObjectAsComObject());

        HashMap<String, String> cpuInfo = new HashMap<String, String>();
        cpuInfo.put("model", wbemObject_dispatch.get("Name").getObjectAsString2());
        cpuInfo.put("vendor", wbemObject_dispatch.get("Manufacturer").getObjectAsString2());
        cpuInfo.put("speed", Integer.toString(wbemObject_dispatch.get("CurrentClockSpeed").getObjectAsInt()));

        return cpuInfo;
    }

    public synchronized float[] getHostCPULoad() throws JIException {
        HyperVConnection.logger.debug("getHostCPULoad");
        String wql = "SELECT * FROM Win32_Processor";
        float[] loadPerCPU = new float[0];
        int numCPU = 0;

        this.testConnectionWin32();

        JIVariant results[] = this.services_dispatch.callMethodA("ExecQuery", new Object[] {new JIString(wql),
            new JIString("WQL"), JIVariant.OPTIONAL_PARAM(), JIVariant.OPTIONAL_PARAM()});
        IJIDispatch wbemObjectSet_dispatch = (IJIDispatch) JIObjectFactory.narrowObject((results[0]).getObjectAsComObject());

        JIVariant variant = wbemObjectSet_dispatch.get("_NewEnum");
        IJIComObject object2 = JIObjectFactory.narrowObject(variant.getObjectAsComObject());
        IJIEnumVariant enumVARIANT = (IJIEnumVariant) JIObjectFactory.narrowObject(object2.queryInterface(IJIEnumVariant.IID));

        JIVariant Count = wbemObjectSet_dispatch.get("Count");
        int count = Count.getObjectAsInt();
        for (int i = 0; i < count; i++) {
            Object[] values = enumVARIANT.next(1);
            JIArray array = (JIArray) values[0];
            Object[] arrayObj = (Object[]) array.getArrayInstance();
            for (Object element : arrayObj) {
                IJIDispatch processor_dispatch = (IJIDispatch) JIObjectFactory.narrowObject(((JIVariant) element)
                    .getObjectAsComObject());

                float[] temp = new float[numCPU + 1];
                int j = 0;
                for (float load : loadPerCPU) {
                    temp[j] = load;
                }
                temp[numCPU] = processor_dispatch.get("LoadPercentage").getObjectAsInt();
                loadPerCPU = temp;
                numCPU++;
            }
        }

        return loadPerCPU;
    }

    public class VM {

        private String nameLabel;

        private String uuid;

        public VM() {
            this.nameLabel = null;
            this.uuid = null;
        }

        public VM(final String nameLabel, final String uuid) {
            this.nameLabel = nameLabel;
            this.uuid = uuid;
        }

        public String getNameLabel() {

            return this.nameLabel;
        }

        public String getUuid() {

            return this.uuid;
        }

        public void setNameLabel(final String nameLabel) {

            this.nameLabel = nameLabel;
        }

        public void setUuid(final String uuid) {

            this.uuid = uuid;
        }
    }

    public class Template {

        private String name;

        private String metadata;

        public Template() {
            this.name = null;
            this.metadata = null;
        }

        public Template(final String name, final String metadata) {
            this.name = name;
            this.metadata = metadata;
        }

        public String getName() {

            return this.name;
        }

        public String getMetaData() {

            return this.metadata;
        }

        public void setName(final String name) {

            this.name = name;
        }

        public void setMetaData(final String metadata) {

            this.metadata = metadata;
        }
    }

    interface VmState {
        final int Enabled = 2;

        final int Disabled = 3;

        final int Reboot = 10;

        final int Paused = 32768;

        final int Suspended = 32769;
    }

}
