package org.ow2.jasmine.vmm.agent.driver.hyperv;

import java.util.ArrayList;
import java.util.List;
import java.util.Set;

import javax.management.ObjectName;

import org.apache.log4j.Logger;
import org.ow2.jasmine.vmm.agent.domain.ManagedResource;
import org.ow2.jasmine.vmm.agent.jmx.MBeanObjectNamer;
import org.ow2.jasmine.vmm.agent.main.AgentCommon;
import org.ow2.jasmine.vmm.api.VMMException;
import org.ow2.jasmine.vmm.api.VirtualMachineImageMXBean;
import org.ow2.jasmine.vmm.api.VirtualMachineImageStoreMXBean;

public class HyperVVMImageStore extends ManagedResource implements VirtualMachineImageStoreMXBean {
    static Logger logger = Logger.getLogger(HyperVVMImageStore.class);

    private String imageStorePath, disk;

    private HyperVConnection connection;

    private ArrayList<HypervVirtualMachineImage> vmList = new ArrayList<HypervVirtualMachineImage>();

    public HyperVVMImageStore(final ObjectName objectName, final String imageStorePath, final String disk,
        final HyperVConnection connection) throws VMMException {
        super(objectName);
        this.imageStorePath = imageStorePath;
        this.connection = connection;
        this.disk = disk;
        this.sync();
    }

    @Override
    public String getPath() {
        return this.imageStorePath;
    }

    public void removeVMImageTemplate(final VirtualMachineImageMXBean vmImage) throws VMMException {
        String vmImageID = vmImage.getUUID();

        boolean success = false;
        try {
            HyperVVMImageStore.logger.debug("Removing VM image template : " + vmImageID);
            success = this.connection.hostDeleteFile(this.getPath(), vmImageID.concat(".vhd"));

        } catch (Exception ex) {
            HyperVHost.logger.error("Removing VM image template failed: VMI connection failure", ex);
            throw new VMMException("Removing VM image template failed: VMI connection failure", ex);
        }
        if (!success) {
            HyperVVMImageStore.logger.error("Failed to removing VM image template : " + vmImageID);
            throw new VMMException("Failed to removing VM image template : " + vmImageID);
        }
        this.sync();

    }

    synchronized void sync() throws VMMException {
        Set<HyperVConnection.Template> tempSet;

        try {
            tempSet = this.connection.getHostVMTemplates(this.imageStorePath.substring(2));
            HyperVVMImageStore.logger.debug("Listing VM templates");
        } catch (Exception ex) {
            throw new VMMException(ex.getMessage());
        }
        for (HyperVConnection.Template temp : tempSet) {

            String uuid = temp.getName();
            if (this.lookUpByUUID(uuid) == null) {
                HypervVirtualMachineImage image = this.newVMImage(MBeanObjectNamer.makeVMImageName(uuid), temp.getMetaData(),
                    uuid, temp.getName());

                if (image == null) {
                    continue;
                }
                try {
                    AgentCommon.getMBeanServer().registerMBean(image, image.getObjectName());
                    HyperVVMImageStore.logger.info("Added VMImage " + image.name);
                } catch (Exception ex) {
                    HyperVVMImageStore.logger.error("Failed to register VMImage MBean", ex);
                    continue;
                }
                this.vmList.add(image);
            }

        }

        // get rid of deleted images
        ArrayList<HypervVirtualMachineImage> vmListTemp = new ArrayList<HypervVirtualMachineImage>(this.vmList);
        for (HypervVirtualMachineImage image : vmListTemp) {
            boolean match = false;
            for (HyperVConnection.Template temp : tempSet) {
                String uuid = temp.getName();
                if (image.getUUID().equals(uuid)) {
                    match = true;
                    break;
                }
            }
            if (!match) {
                try {
                    AgentCommon.getMBeanServer().unregisterMBean(MBeanObjectNamer.makeVMImageName(image.getName()));
                } catch (Exception ex) {
                    HyperVVMImageStore.logger.error("Failed to unregister VMImage MBean", ex);
                }
                this.vmList.remove(image);
            }
        }

    }

    public VirtualMachineImageMXBean lookUpByUUID(final String uuid) {
        for (HypervVirtualMachineImage image : this.vmList) {
            if (image.getUUID().equals(uuid)) {
                return image;
            }
        }
        return null;
    }

    public List<VirtualMachineImageMXBean> listVMImageTemplates() {
        ArrayList<VirtualMachineImageMXBean> result = new ArrayList<VirtualMachineImageMXBean>(this.vmList);
        return result;
    }

    public long getCapacityMB() {
        long size = 0;

        try {
            size = this.connection.getHostSizeMB(this.disk);
            HyperVVMImageStore.logger.debug("GetCapacityMB of imageStore : " + size);
        } catch (Exception ex) {
            HyperVVMImageStore.logger.error("Failed to get CapacityMB of imageStore", ex);
        }

        return size;
    }

    public long getFreeSpaceMB() {
        long freeSpace = 0;

        try {
            freeSpace = this.connection.getHostFreeSpaceMB(this.disk);
            HyperVVMImageStore.logger.debug("GetFreeSpaceMB of imageStore : " + freeSpace);
        } catch (Exception ex) {
            HyperVVMImageStore.logger.error("Failed to get FreeSpaceMB of imageStore", ex);
        }

        return freeSpace;
    }

    public String getName() {
        return "DiskImageStore(" + this.getObjectName() + ")";
    }

    public HypervVirtualMachineImage newVMImage(final ObjectName objectName, final String metadata, final String uuid,
        final String name) {
        HypervVirtualMachineImage vmi = new HypervVirtualMachineImage(objectName);
        vmi.metadata = new String(metadata);
        vmi.uuid = uuid;
        vmi.name = name;
        return vmi;
    }

    class HypervVirtualMachineImage extends ManagedResource implements VirtualMachineImageMXBean {
        private String name, uuid, metadata;

        private HypervVirtualMachineImage(final ObjectName objectName) {
            super(objectName);
        }

        public String getMetaData() {
            return this.metadata;
        }

        public String getName() {
            return this.name;
        }

        public String getUUID() {
            return this.uuid;
        }

        public String getFileName() {
            return HyperVVMImageStore.this.imageStorePath + this.name;
        }

    }

}
