/**
 * JASMINe VMMapi: JASMINe Virtual Machine Management API
 * Copyright (C) 2009 France Telecom R&D
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: XenHost.java 5690 2009-12-07 12:20:55Z dangtran $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.vmm.agent.driver.xen;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

import javax.management.InstanceNotFoundException;
import javax.management.MBeanNotificationInfo;
import javax.management.Notification;
import javax.management.NotificationEmitter;
import javax.management.ObjectName;

import org.apache.log4j.Logger;
import org.ow2.jasmine.vmm.agent.domain.ManagedResource;
import org.ow2.jasmine.vmm.agent.driver.util.LocalExec;
import org.ow2.jasmine.vmm.agent.driver.util.RemoteExec;
import org.ow2.jasmine.vmm.agent.driver.xen.XenVMImageStore.XenVirtualMachineImage;
import org.ow2.jasmine.vmm.agent.jmx.MBeanObjectNamer;
import org.ow2.jasmine.vmm.agent.main.AgentCommon;
import org.ow2.jasmine.vmm.api.HostMXBean;
import org.ow2.jasmine.vmm.api.InsufficientResourcesException;
import org.ow2.jasmine.vmm.api.InvalidVMConfigException;
import org.ow2.jasmine.vmm.api.NotificationTypes;
import org.ow2.jasmine.vmm.api.ResourceUsage;
import org.ow2.jasmine.vmm.api.ServerPoolMXBean;
import org.ow2.jasmine.vmm.api.VMConfigSpec;
import org.ow2.jasmine.vmm.api.VMCustomizationSpec;
import org.ow2.jasmine.vmm.api.VMMException;
import org.ow2.jasmine.vmm.api.VirtualMachineImageStoreMXBean;
import org.ow2.jasmine.vmm.api.VirtualMachineMXBean;

import com.xensource.xenapi.Connection;
import com.xensource.xenapi.HostCpu;
import com.xensource.xenapi.VM;

/**
 * Xen driver Host MXBean implementation
 */
public class XenHost extends ManagedResource implements HostMXBean, NotificationEmitter {
    static Logger logger = Logger.getLogger(XenHost.class);

    static String XEN_CONFIG_HOME = "/etc/xen/auto";

    // the thread pool size determines the maximum number of
    // simultaneous VM creation on the same host
    private static final int MAX_PARALELL_VM_CREATION_PER_HOST = 3;

    // one extra thread for the event reader
    private static final int THREADPOOL_SIZE = 1 + XenHost.MAX_PARALELL_VM_CREATION_PER_HOST;

    private static ExecutorService executorService = Executors.newFixedThreadPool(XenHost.THREADPOOL_SIZE);

    private List<XenVirtualMachine> vmList = new CopyOnWriteArrayList<XenVirtualMachine>();

    private XenServerPool serverPool;

    private Connection connection;

    private XenPerfCollector perfCollectionTask;

    private com.xensource.xenapi.Host host;

    private String hostName;

    private int numCPU = -1;

    private long cpuFrequencyMhz = -1;

    private RemoteExec.SshAuthInfo authInfo;

    public static class XenAPIParams {
        private final String xenAPIPort;

        private final String xenAPIUser;

        private final String xenAPIPassword;

        public XenAPIParams(final String xenAPIPort, final String xenAPIUser, final String xenAPIPassword) {
            this.xenAPIPort = xenAPIPort;
            this.xenAPIUser = xenAPIUser;
            this.xenAPIPassword = xenAPIPassword;
        }

        public String getXenAPIPort() {
            return this.xenAPIPort;
        }

        public String getXenAPIUser() {
            return this.xenAPIUser;
        }

        public String getXenAPIPassword() {
            return this.xenAPIPassword;
        }

    }

    public static XenHost newHost(final XenServerPool resourcePool, final ObjectName objectName, final String hostName,
        final XenAPIParams xenAPIParams, final RemoteExec.SshAuthInfo authInfo) {

        try {
            String xenAPIURL = "http://" + hostName + ":" + xenAPIParams.getXenAPIPort();
            Connection connection = new Connection(xenAPIURL, xenAPIParams.getXenAPIUser(), xenAPIParams.getXenAPIPassword());
            XenHost xenHost = new XenHost(resourcePool, connection, objectName, hostName, authInfo);
            return xenHost;
        } catch (Exception ex) {
            XenHost.logger.error("Failed to establish Xen-API connection with host " + hostName + " with login "
                + xenAPIParams.getXenAPIUser(), ex);
            return null;
        }

    }

    private XenHost(final XenServerPool serverPool, final Connection connection, final ObjectName objectName,
        final String hostName, final RemoteExec.SshAuthInfo authInfo) {
        super(objectName);
        this.serverPool = serverPool;
        this.connection = connection;
        this.hostName = hostName;
        this.authInfo = authInfo;

        try {
            Set<com.xensource.xenapi.Host> hosts = com.xensource.xenapi.Host.getAll(connection);
            this.host = hosts.iterator().next();
            this.populate();
        } catch (Exception ex) {
            XenHost.logger.debug("Host " + hostName, ex);
        }

        this.startPerfMonitor();
    }

    RemoteExec.SshAuthInfo getSshAuthInfo() {
        return this.authInfo;
    }

    Connection getXenAPIConnection() {
        return this.connection;
    }

    List<XenVirtualMachine> getVMs() {
        return this.vmList;
    }

    private void addVM(final XenVirtualMachine vm) {
        this.vmList.add(vm);
    }

    private void removeVM(final XenVirtualMachine vm) {
        this.vmList.remove(vm);
    }

    public ServerPoolMXBean getServerPool() {
        return this.serverPool;
    }

    public void postMigrateVM(final XenVirtualMachine vm, final XenHost newHost) {
        this.removeVM(vm);
        newHost.addVM(vm);
    }

    public void onVMDestroy(final XenVirtualMachine vm) {
        this.removeVM(vm);
        this.emitNotification(NotificationTypes.VM_DEL, "Destroyed", vm.getObjectName());
    }

    private Set<PerfMetric> currentMonitoredMetrics;

    private long currentMonitoringPeriod;

    public void configurePerfMonitor(final Set<PerfMetric> metricsOfInterest, long periodMillis) {
        this.stopPerfMonitor();
        if (periodMillis < 1000) {
            periodMillis = 1000;
        }
        if (metricsOfInterest.contains(PerfMetric.VM_CPU_LOAD)) {
            this.currentMonitoringPeriod = periodMillis;
            this.currentMonitoredMetrics = metricsOfInterest;
            this.startPerfMonitor();
        }
    }

    String getIPAddress(String macAddress) {
        macAddress = macAddress.toUpperCase();

        XenHost.logger.debug("Determining IP address from MAC address " + macAddress + " ...");

        RemoteExec.Result result = null;
        String command = "getIPfromMAC.sh " + macAddress;

        XenHost.logger.debug("Launching command: " + command);

        if (this.serverPool.runGetIpScriptRemotely()) {
            try {
                result = RemoteExec.commandAsRoot(this.hostName, this.authInfo, command);
            } catch (RemoteExec.SshException ex) {
                XenHost.logger.debug("SSH failure", ex);
                return null;
            }
        } else {
            try {
                XenHost.logger.debug("Local execution!");
                result = LocalExec.command(command);
            } catch (Exception ex) {
                XenHost.logger.debug("Local command failure", ex);
                return null;
            }
        }

        if (result.exitCode != 0) {
            XenHost.logger.error("Cannot get IP from MAC " + result.output);
            return null;
        }

        String ipAddress = result.output;
        if (ipAddress != null && ipAddress.length() > 0 && ipAddress.endsWith("\n")) {
            ipAddress = ipAddress.substring(0, ipAddress.length() - 1);
        }
        XenHost.logger.debug("Mac-to-IP " + macAddress + " -> " + ipAddress);
        return ipAddress;
    }

    private synchronized void startPerfMonitor() {
        if (this.perfCollectionTask == null) {
            // XenHost.executorService.execute(new XenPerfCollector(this));
            this.perfCollectionTask = new XenPerfCollector(this);
            this.perfCollectionTask.start();
        }
    }

    private synchronized void stopPerfMonitor() {
        if (this.perfCollectionTask != null) {
            this.perfCollectionTask.stop();
            this.perfCollectionTask = null;
        }
    }

    @Override
    public ObjectName getObjectName() {
        return this.objectName;
    }

    HashMap<String, String> hypervisorInfo;

    // TODO
    public Map<String, String> getHypervisorInfo() {
        if (this.hypervisorInfo == null) {
            this.hypervisorInfo = new HashMap<String, String>();
            this.hypervisorInfo.put("vendor", "");
            this.hypervisorInfo.put("name", "Xen");
            this.hypervisorInfo.put("version", "3.2");
        }
        return this.hypervisorInfo;
    }

    HashMap<String, String> cpuInfo;

    public Map<String, String> getCPUInfo() {
        if (this.cpuInfo == null) {
            this.cpuInfo = new HashMap<String, String>();
            synchronized (this.connection) {
                try {
                    HostCpu[] cpus = this.host.getHostCPUs(this.connection).toArray(new HostCpu[0]);
                    this.numCPU = cpus.length;
                    HostCpu.Record rec = cpus[0].getRecord(this.connection);
                    this.cpuInfo.put("model", rec.modelname);
                    this.cpuFrequencyMhz = rec.speed;
                    this.cpuInfo.put("speed", Long.toString(rec.speed));
                    this.cpuInfo.put("vendor", rec.vendor);
                    // cpuInfo.put("features",rec.features);
                    // cpuInfo.put("flags",rec.flags);
                } catch (Exception ex) {
                    XenHost.logger.debug("Host " + this.hostName, ex);
                }
            }
        }
        return this.cpuInfo;
    }

    public long getCPUFrequencyMhz() {
        if (this.cpuFrequencyMhz == -1) {
            this.getCPUInfo();
        }
        return this.cpuFrequencyMhz;

    }

    public int getNumCPU() {
        if (this.numCPU == -1) {
            this.getCPUInfo();
        }
        return this.numCPU;
    }

    public String getHostName() {
        return this.hostName;
    }

    public VirtualMachineImageStoreMXBean getVMImageStore() {
        return this.serverPool.getVMImageStore();
    }

    public long getFreeMemoryMB() {
        RemoteExec.Result result = null;
        String command = "xm info | grep ^free_memory | awk '{print $3}'";
        try {
            result = RemoteExec.commandAsRoot(this.hostName, this.authInfo, command);
        } catch (RemoteExec.SshException ex) {
            XenHost.logger.error("SSH exception: ", ex);
        }
        if (result.exitCode != 0) {
            XenHost.logger.error("getFreeMemoryMB: SSH " + command + " failed");
            return 0;
        }
        if (result.output.endsWith("\n")) {
            result.output = result.output.substring(0, result.output.length() - 1);
        }
        long freeMem = Long.parseLong(result.output);

        for (VirtualMachineMXBean vm : this.vmList) {
            XenVirtualMachine xvm = (XenVirtualMachine) vm;
            if (xvm.getDomID() == 0) {
                freeMem += xvm.getMemorySizeMB() - (196 + 350); // XXX
            }
        }

        return freeMem;
    }

    private void populate() {
        try {
            synchronized (this.connection) {
                Set<VM> vms = VM.getAll(this.connection);
                for (VM vm : vms) {
                    ObjectName name;
                    if (vm.getDomid(this.connection) == 0) {
                        // for Domain 0, we use the host UUID
                        name = MBeanObjectNamer.makeVirtualMachineName(this.serverPool.getPath() + "/"
                            + vm.getNameLabel(this.connection), this.host.getUuid(this.connection));
                    } else {
                        name = MBeanObjectNamer.makeVirtualMachineName(this.serverPool.getPath() + "/"
                            + vm.getNameLabel(this.connection), vm.getUuid(this.connection));
                    }

                    XenVirtualMachine xenVM = new XenVirtualMachine(name, this, this.connection, vm, null);
                    this.addVM(xenVM);
                    AgentCommon.getMBeanServer().registerMBean(xenVM, name);
                    XenHost.logger.info("Added XenVirtualMachineMBean " + name);
                }
            }
        } catch (Exception ex) {
            XenHost.logger.debug("Host " + this.hostName, ex);
        }
    }

    public List<VirtualMachineMXBean> getResidentVMs() {
        List<VirtualMachineMXBean> result = new ArrayList<VirtualMachineMXBean>(this.vmList);
        return result;
    }

    public VirtualMachineMXBean createVM(final VMConfigSpec vmSpecIn, final boolean sync) throws InvalidVMConfigException,
        VMMException {
        final VMConfigSpec vmSpec = new VMConfigSpec(vmSpecIn);
        XenHost.logger.info("Creating new VM [name=" + vmSpec.getName() + ",memorySizeMB=" + vmSpec.getMemorySizeMB()
            + ",diskSize=" + vmSpec.getDiskSizeMB() + ",numVCPU=" + vmSpec.getNumVCPU() + "]");

        final XenVirtualMachineImage template = (XenVirtualMachineImage) this.serverPool.getVMImageStore().lookUpByUUID(
            vmSpec.getVmImageUUID());
        if (template == null) {
            throw new InvalidVMConfigException("Invalid VM Image UUID");
        }

        final String vmName = vmSpec.getName();

        if (!sync) {
            XenHost.executorService.execute(new Runnable() {
                public void run() {
                    try {
                        XenHost.this.createVM2(vmSpec, vmName, template);
                    } catch (Exception ex) {
                    }
                }
            });
            return null;
        } else {
            return this.createVM2(vmSpec, vmName, template);
        }
    }

    private VirtualMachineMXBean createVM2(final VMConfigSpec vmSpec, final String vmName, final XenVirtualMachineImage template)
        throws InvalidVMConfigException, VMMException {

        RemoteExec.Result result;
        try {
            String diskCreationHost = this.serverPool.getVmImageStoreHostname();
            if (diskCreationHost == null) {
                diskCreationHost = this.getHostName();
            }

            result = XenSSH.createVMDiskImages(diskCreationHost, this.getSshAuthInfo(), vmName, vmSpec, template.getFileName());

            if (result.exitCode == 0) {
                result = XenSSH.createVM(this, vmName, vmSpec, template.getFileName());
            } else {
                XenHost.logger.error("Command createVMDiskImages failed: " + result.error);
                throw new VMMException("VM's disk images creation failed: " + result.error);
            }

        } catch (RemoteExec.SshException ex) {
            XenHost.logger.error("SSH failed: ", ex);
            throw new VMMException("VM creation failed: SSH connection failure", ex);
        }
        if (result.exitCode != 0) {
            XenHost.logger.error("Command createVM failed: " + result.error);
            // TODO parse error message
            throw new VMMException("VM creation failed: " + result.error);
        }

        VM newVM = null;
        String vmUUID = "";
        synchronized (this.connection) {
            try {
                newVM = VM.getByNameLabel(this.connection, vmName).iterator().next();
                vmUUID = newVM.getUuid(this.connection);
            } catch (Exception ex) {
                XenHost.logger.error("Failed to get VM uuid: ", ex);
                throw new VMMException("VM creation failure: Failed to get VM uuid");
            }
        }
        ObjectName name = null;
        XenVirtualMachine xenVM = null;
        try {
            name = MBeanObjectNamer.makeVirtualMachineName(this.serverPool.getPath() + "/" + vmName, vmUUID);
            HashMap<String, String> map = new HashMap<String, String>();
            map.put("imageID", vmSpec.getVmImageUUID());
            xenVM = new XenVirtualMachine(name, this, this.connection, newVM, map);
            this.addVM(xenVM);

            AgentCommon.getMBeanServer().registerMBean(xenVM, name);
        } catch (Exception ex) {
            XenHost.logger.debug("Host " + this.hostName, ex);
            throw new VMMException(ex);
        }

        this.emitNotification(NotificationTypes.VM_ADD, "Created", name);

        return xenVM;
    }

    public VirtualMachineMXBean cloneVM(final String targetVmName, final String clonedVmName,
        final VMCustomizationSpec custSpec, final boolean sync) throws InstanceNotFoundException,
        InsufficientResourcesException, VMMException {
        try {
            RemoteExec.Result result = XenSSH.cloneVM(this, targetVmName, clonedVmName, custSpec);
            if (result.exitCode != 0) {
                XenVirtualMachine.logger.error("Command cloneVM failed: " + result.error);
                throw new VMMException("VM cloning failed: " + result.output);
            }
            XenVirtualMachine.logger.debug(result.output);
        } catch (Exception ex) {
            XenVirtualMachine.logger.error("ssh failure", ex);
            throw new VMMException("VM creation failed: SSH connection failure", ex);
        }

        VM newVM = null;
        String vmUUID = "";
        synchronized (this.connection) {
            try {
                newVM = VM.getByNameLabel(this.connection, clonedVmName).iterator().next();
                vmUUID = newVM.getUuid(this.connection);
            } catch (Exception ex) {
                XenHost.logger.error("Failed to get VM uuid: ", ex);
                throw new VMMException("VM creation failure: Failed to get VM uuid");
            }
        }
        ObjectName name = null;
        XenVirtualMachine xenVM = null;
        try {
            name = MBeanObjectNamer.makeVirtualMachineName(this.serverPool.getPath() + "/" + clonedVmName, vmUUID);
            HashMap<String, String> map = new HashMap<String, String>();
            xenVM = new XenVirtualMachine(name, this, this.connection, newVM, map);
            this.addVM(xenVM);

            AgentCommon.getMBeanServer().registerMBean(xenVM, name);
        } catch (Exception ex) {
            XenHost.logger.debug("Host " + this.hostName, ex);
            throw new VMMException(ex);
        }

        this.emitNotification(NotificationTypes.VM_ADD, "Created", name);

        return xenVM;
    }

    public long getTotalMemoryMB() {
        synchronized (this.connection) {
            try {
                return this.host.getMetrics(this.connection).getMemoryTotal(this.connection) / (1024 * 1024);
            } catch (Exception ex) {
                XenHost.logger.debug("Host " + this.hostName, ex);
            }
        }
        return 0;
    }

    public float[] getLoadPerCPU() {
        float[] result = new float[0];
        synchronized (this.connection) {
            try {
                Set<HostCpu> cpus = this.host.getHostCPUs(this.connection);
                result = new float[cpus.size()];
                int i = 0;
                for (HostCpu cpu : cpus) {
                    result[i++] = (float) (cpu.getUtilisation(this.connection) * 100);
                }
            } catch (Exception ex) {
                XenHost.logger.debug("Host " + this.hostName, ex);
            }
        }
        return result;
    }

    public float getCPULoad() {
        float aggregatedCPULoad = 0;
        for (XenVirtualMachine vm : this.vmList) {
            aggregatedCPULoad += vm.getResourceUsage().getCpuLoad();
        }
        return aggregatedCPULoad;
    }

    @Override
    public MBeanNotificationInfo[] getNotificationInfo() {
        return new MBeanNotificationInfo[] {new MBeanNotificationInfo(new String[] {NotificationTypes.VM_STATE_CHANGE,
            NotificationTypes.VM_ADD, NotificationTypes.VM_DEL, NotificationTypes.ERROR, NotificationTypes.LOG},
            Notification.class.getName(), "Host event")};
    }

    public Map<String, ResourceUsage> getVMResourceUsage() {
        HashMap<String, ResourceUsage> result = new HashMap<String, ResourceUsage>();
        for (XenVirtualMachine vm : this.vmList) {
            result.put(vm.getNameLabel(), vm.getResourceUsage());
        }
        return result;
    }

    public Map<String, Float> getVMCPULoads() {
        HashMap<String, Float> cpuLoads = new HashMap<String, Float>();
        for (XenVirtualMachine vm : this.vmList) {
            cpuLoads.put(vm.getNameLabel(), vm.getResourceUsage().getCpuLoad());
        }
        return cpuLoads;
    }

}
