package org.ow2.jasmine.vmm.agent.driver.xen;

import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.StringTokenizer;

import org.apache.log4j.Logger;
import org.ow2.jasmine.vmm.agent.driver.util.RemoteExec;
import org.ow2.jasmine.vmm.agent.driver.util.ResourceUsageHelper;
import org.ow2.jasmine.vmm.api.NotificationTypes;
import org.ow2.jasmine.vmm.api.ResourceUsage;
import org.ow2.jasmine.vmm.api.ResourceUsage.DiskStats;
import org.ow2.jasmine.vmm.api.VirtualMachineMXBean.PowerState;

public class XenPerfCollector {
    static Logger logger = Logger.getLogger(XenPerfCollector.class);

    private static String PERF_DAEMON_COMMAND = "perfmondemon -a -i 5 -n 1000000";

    private static String ITERATION_SEPARATOR = "-----";

    private XenHost host;

    private RemoteExec.SSHExec sshExec;

    public XenPerfCollector(final XenHost host) {
        this.host = host;
        this.sshExec = new RemoteExec.SSHExec(host.getHostName(), "root", host.getSshAuthInfo(),
            XenPerfCollector.PERF_DAEMON_COMMAND);
    }

    private class ParseError extends Exception {
        public ParseError(final String msg) {
            super(msg);
        }
    }

    private String readKeyStringValue(final StringTokenizer st, final String expectedKeyName) throws ParseError {
        String key = st.nextToken();
        if (!key.equalsIgnoreCase(expectedKeyName)) {
            throw new ParseError("expecting " + expectedKeyName + " read " + key);
        }
        return st.nextToken();
    }

    private long readKeyLongValue(final StringTokenizer st, final String expectedKeyName) throws ParseError {
        String key = st.nextToken();
        if (!key.equalsIgnoreCase(expectedKeyName)) {
            throw new ParseError("expecting " + expectedKeyName + " read " + key);
        }
        return Long.parseLong(st.nextToken());
    }

    private float readKeyFloatValue(final StringTokenizer st, final String expectedKeyName) throws ParseError {
        String key = st.nextToken();
        if (!key.equalsIgnoreCase(expectedKeyName)) {
            throw new ParseError("expecting " + expectedKeyName + " read " + key);
        }
        return Float.parseFloat(st.nextToken());
    }

    private String skipVM(final StringTokenizer st) {
        String t = null;
        while (true) {
            if (!st.hasMoreTokens()) {
                return null;
            }
            t = st.nextToken();
            if (t.equalsIgnoreCase("IdVm") || t.equalsIgnoreCase("Iteration")) {
                return t;
            }
        }
    }

    private String readVMStats(final StringTokenizer st, final Date samplingTime) throws ParseError {
        long domId = Long.parseLong(st.nextToken());

        XenVirtualMachine currentXenVM = null;
        for (XenVirtualMachine vm : this.host.getVMs()) {
            PowerState vmState = vm.getState();
            if ((vmState == PowerState.RUNNING || vmState == PowerState.PAUSED) && vm.getDomID() == domId) {
                currentXenVM = vm;
                break;
            }
        }
        if (currentXenVM == null) {
            XenHost.logger.debug("Skipping perf info for unknown VM domainID=" + domId);
            return this.skipVM(st);
        }
        ResourceUsage vmUsage = new ResourceUsage();
        vmUsage.setSamplingTime(samplingTime);
        this.readKeyLongValue(st, "interval");
        // read CPU load and memory occupation
        vmUsage.setCpuLoad((this.readKeyFloatValue(st, "cpu")) / 100.0f);
        vmUsage.setMemoryUsedKBytes(this.readKeyLongValue(st, "memory"));
        // read disk and net IO stats
        List<DiskStats> diskStatsList = new ArrayList<DiskStats>();
        vmUsage.setDiskStats(diskStatsList);
        List<ResourceUsage.NetworkStats> netStatsList = new ArrayList<ResourceUsage.NetworkStats>();
        vmUsage.setNetworkStats(netStatsList);
        String t = null;
        while (st.hasMoreTokens()) {
            t = st.nextToken();
            if (t.equalsIgnoreCase("IO")) {
                String deviceName = st.nextToken();
                long readKBytePerSec = this.readKeyLongValue(st, "read");
                long writtenKBytePerSec = this.readKeyLongValue(st, "write");
                ResourceUsage.DiskStats diskStats = new ResourceUsage.DiskStats();
                diskStats.setDeviceName(deviceName);
                diskStats.setDiskReadKBytePerSec(readKBytePerSec);
                diskStats.setDiskWrittenKBytesPerSec(writtenKBytePerSec);
                diskStatsList.add(diskStats);
            } else if (t.equalsIgnoreCase("net")) {
                String deviceName = st.nextToken();
                long readKBytePerSec = this.readKeyLongValue(st, "rcv");
                long writtenKBytePerSec = this.readKeyLongValue(st, "tx");
                ResourceUsage.NetworkStats netStats = new ResourceUsage.NetworkStats();
                netStats.setDeviceName(deviceName);
                netStats.setNetReceivedKbitPerSec(readKBytePerSec * 8);
                netStats.setNetTransmittedKbitPerSec(writtenKBytePerSec * 8);
                netStatsList.add(netStats);
            } else {
                currentXenVM.updateCurrentResourceUsage(vmUsage);
                return t;
            }
        }
        currentXenVM.updateCurrentResourceUsage(vmUsage);
        return null;
    }

    private void readAllVMStats(final String input) {
        StringTokenizer st = new StringTokenizer(input);
        // XenPerfCollector.logger.debug("INPUT=" + input);

        try {
            String t = st.nextToken();
            if (!t.equalsIgnoreCase("Iteration")) {
                throw new ParseError("Cannot find iteration token");
            }
            // read iteration and time
            t = st.nextToken();
            t = this.readKeyStringValue(st, "time");
            Date time = new Date(Long.parseLong(t) * 1000);

            t = st.nextToken();
            if (!t.equalsIgnoreCase("IdVm")) {
                throw new ParseError("Cannot find token IdVm");
            }
            while (true) {
                t = this.readVMStats(st, time);
                if (t == null || !t.equalsIgnoreCase("IdVm")) {
                    if (t != null) {
                        XenPerfCollector.logger.debug("???? " + t);
                    }
                    break;
                }
            }

            // send notification
            Map<String, Object> notifUserData = new HashMap<String, Object>();
            for (XenVirtualMachine vm : this.host.getVMs()) {
                if (vm.getState() == PowerState.RUNNING && vm.getResourceUsage() != null) {
                    notifUserData.put(vm.getNameLabel(), ResourceUsageHelper.serialize(vm.getResourceUsage()));
                }
            }
            this.host.emitNotification(NotificationTypes.PERF_REPORT, "Resource Usage", notifUserData);

        } catch (ParseError ex) {
            XenHost.logger.error("parse error", ex);
        }
    }

    private StringBuffer buffer = new StringBuffer();

    public void start() {
        RemoteExec.SSHExecCallback stdoutCb = new RemoteExec.SSHExecCallback() {
            @Override
            public void newData(String s) {
                while (true) {
                    int i = s.indexOf(XenPerfCollector.ITERATION_SEPARATOR);
                    if (i == -1) {
                        XenPerfCollector.this.buffer.append(s);
                        break;
                    } else {
                        if (i > 0) {
                            XenPerfCollector.this.buffer.append(s.substring(0, i - 1));
                            XenPerfCollector.this.readAllVMStats(XenPerfCollector.this.buffer.toString());
                            XenPerfCollector.this.buffer.setLength(0);
                            s = s.substring(i + XenPerfCollector.ITERATION_SEPARATOR.length());
                        }

                        // XenPerfCollector.this.buffer.append(s.substring(i +
                        // XenPerfCollector.ITERATION_SEPARATOR.length()));
                    }
                }
            }
        };
        RemoteExec.SSHExecCallback stderrCb = new RemoteExec.SSHExecCallback() {
            @Override
            public void newData(final String s) {
                // XenPerfCollector.logger.error("Error: " + s);
            }
        };
        XenPerfCollector.logger.debug("Starting perf collector daemon on host " + this.host.getHostName());
        try {
            this.sshExec.connect(stdoutCb, stderrCb);
        } catch (RemoteExec.SshException ex) {
            XenPerfCollector.logger.error("Failed to connect ", ex);
        }
    }

    public void stop() {
        this.sshExec.disconnect();
    }

}
