/**
 * JASMINe VMMapi: JASMINe Virtual Machine Management API
 * Copyright (C) 2009 France Telecom R&D
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id$
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.vmm.agent.driver.xen;

import java.util.UUID;

import org.apache.log4j.Logger;
import org.ow2.jasmine.vmm.agent.driver.util.RemoteExec;
import org.ow2.jasmine.vmm.agent.driver.util.RemoteExec.SshAuthInfo;
import org.ow2.jasmine.vmm.api.VMConfigSpec;
import org.ow2.jasmine.vmm.api.VMCustomizationSpec;
import org.ow2.jasmine.vmm.api.VnicIPSettings;
import org.ow2.jasmine.vmm.api.VnicIPSettings.IpAssignmentMode;

/**
 * The XenSSH class gathers static methods wrapping SSH remote invocation to
 * shell scripts that must be installed on Xen backend hosts.
 */
public class XenSSH {
    static Logger logger = Logger.getLogger(XenSSH.class);

    /**
     * Creates a new VM
     * 
     * @param host target host
     * @param vmName VM label
     * @param vmSpec VM parameters
     * @param templateFileName absolute path of the template image
     * @return SSH result
     * @throws RemoteExec.SshException raised if the SSH connection cannot be
     *         established
     */
    public static RemoteExec.Result createVM(final XenHost host, final String vmName, final VMConfigSpec vmSpec,
        final String templateFileName) throws RemoteExec.SshException {
        String command = "createVM -d " + vmSpec.getDiskSizeMB() + " -t " + templateFileName + " -m "
            + vmSpec.getMemorySizeMB() + " -c " + vmSpec.getNumVCPU() + " -u " + UUID.randomUUID().toString() + " " + vmName;
        XenSSH.logger.debug("host " + host.getHostName() + " launching command " + command);
        return RemoteExec.commandAsRoot(host.getHostName(), host.getSshAuthInfo(), command);
    }

    /**
     * @param hostname target's name
     * @param authoInfo
     * @param vmName VM label
     * @param vmSpec VM parameters
     * @param templateFileName absolute path of the template image
     * @return SSH result
     * @throws RemoteExec.SshException raised if the SSH connection cannot be
     *         established
     */
    public static RemoteExec.Result createVMDiskImages(final String hostname, final SshAuthInfo authoInfo, final String vmName,
        final VMConfigSpec vmSpec, final String templateFileName) throws RemoteExec.SshException {
        String command = "createVMDiskImages " + " " + vmName + " " + vmSpec.getDiskSizeMB() + " " + templateFileName + " "
            + vmSpec.getMemorySizeMB();
        XenSSH.logger.debug("host " + hostname + " launching command " + command);
        return RemoteExec.commandAsRoot(hostname, authoInfo, command);
    }

    public static RemoteExec.Result cloneVM(final XenHost host, final String vmName, final String clonedVmName,
        final VMCustomizationSpec custSpec) throws RemoteExec.SshException {
        String command = "cloneVM --src " + vmName + " --name " + clonedVmName + " --force ";
        if (custSpec != null) {
            if (custSpec.getGuestOsHostName() != null && !custSpec.getGuestOsHostName().equals("")) {
                command += " --hostname " + custSpec.getGuestOsHostName();
            }
            // XXX configure only first NIC
            if (custSpec.getVnicIpSettingsList() != null) {
                VnicIPSettings nic0Settings = custSpec.getVnicIpSettingsList().get(0);
                if (nic0Settings.getIpAssignmentMode() == IpAssignmentMode.DHCP) {
                    command += " --net eth0/dhcp";
                } else if (nic0Settings.getIpAssignmentMode() == IpAssignmentMode.FIXED) {
                    command += " --net eth0/static/" + nic0Settings.getIpAddress() + "/" + nic0Settings.getSubnetMask() + "/"
                        + nic0Settings.getGateway();
                }
            }
        }
        XenSSH.logger.debug("host " + host.getHostName() + " launching command " + command);
        return RemoteExec.commandAsRoot(host.getHostName(), host.getSshAuthInfo(), command);
    }

    /**
     * Suspends a VM
     * 
     * @param host host containing the VM
     * @param vmName label of the VM to suspend
     * @return SSH result
     * @throws RemoteExec.SshException raised if the SSH connection cannot be
     *         established
     */
    public static RemoteExec.Result suspendVM(final XenHost host, final String vmName) throws RemoteExec.SshException {
        String command = "virsh save " + vmName + " /tmp/" + vmName + ".snapshot";
        XenSSH.logger.debug("host " + host.getHostName() + " Launching command " + command);
        return RemoteExec.commandAsRoot(host.getHostName(), host.getSshAuthInfo(), command);
    }

    /**
     * Resumes a suspended VM
     * 
     * @param host host containing the VM
     * @param vmName label of the VM to resume
     * @return SSH result
     * @throws RemoteExec.SshException raised if the SSH connection cannot be
     *         established
     */
    public static RemoteExec.Result resumeVM(final XenHost host, final String vmName) throws RemoteExec.SshException {
        String command = "virsh restore /tmp/" + vmName + ".snapshot";
        XenSSH.logger.debug("host " + host.getHostName() + " Launching command " + command);
        return RemoteExec.commandAsRoot(host.getHostName(), host.getSshAuthInfo(), command);
    }

    /**
     * Suspends a VM
     * 
     * @param host host containing the VM
     * @param vmName label of the VM to suspend
     * @return SSH result
     * @throws RemoteExec.SshException raised if the SSH connection cannot be
     *         established
     */
    public static RemoteExec.Result pauseVM(final XenHost host, final String vmName) throws RemoteExec.SshException {
        String command = "virsh suspend " + vmName;
        XenSSH.logger.debug("host " + host.getHostName() + " Launching command " + command);
        return RemoteExec.commandAsRoot(host.getHostName(), host.getSshAuthInfo(), command);
    }

    /**
     * Resumes a suspended VM
     * 
     * @param host host containing the VM
     * @param vmName label of the VM to resume
     * @return SSH result
     * @throws RemoteExec.SshException raised if the SSH connection cannot be
     *         established
     */
    public static RemoteExec.Result unpauseVM(final XenHost host, final String vmName) throws RemoteExec.SshException {
        String command = "virsh resume " + vmName;
        XenSSH.logger.debug("host " + host.getHostName() + " Launching command " + command);
        return RemoteExec.commandAsRoot(host.getHostName(), host.getSshAuthInfo(), command);
    }

    /**
     * Starts a stopped VM
     * 
     * @param host host containing the VM
     * @param vmName label of the VM to start
     * @return SSH result
     * @throws RemoteExec.SshException raised if the SSH connection cannot be
     *         established
     */
    public static RemoteExec.Result startVM(final XenHost host, final String vmName) throws RemoteExec.SshException {
        String command = "virsh start " + vmName;
        XenSSH.logger.debug("host " + host.getHostName() + " Launching command " + command);
        return RemoteExec.commandAsRoot(host.getHostName(), host.getSshAuthInfo(), command);
    }

    /**
     * Shutdowns a VM
     * 
     * @param host host containing the VM
     * @param vmName label of the VM to shutdown
     * @return SSH result
     * @throws RemoteExec.SshException raised if the SSH connection cannot be
     *         established
     */
    public static RemoteExec.Result shutdownVM(final XenHost host, final String vmName) throws RemoteExec.SshException {
        String command = "virsh shutdown " + vmName;
        XenSSH.logger.debug("host " + host.getHostName() + " Launching command " + command);
        return RemoteExec.commandAsRoot(host.getHostName(), host.getSshAuthInfo(), command);
    }

    /**
     * Destroys a VM
     * 
     * @param host host containing the VM
     * @param vmName label of the VM to destroy
     * @return SSH result
     * @throws RemoteExec.SshException raised if the SSH connection cannot be
     *         established
     */
    public static RemoteExec.Result destroyVM(final XenHost host, final String vmName) throws RemoteExec.SshException {
        String command = "destroyVM " + vmName;
        XenSSH.logger.debug("host " + host.getHostName() + " Launching command " + command);
        return RemoteExec.commandAsRoot(host.getHostName(), host.getSshAuthInfo(), command);
    }

    /**
     * Migrates a VM
     * 
     * @param sourceHost host containing the VM to migrate
     * @param vmName label of the VM to migrate
     * @param targetHostName host to which the VM will migrate
     * @return SSH result
     * @throws RemoteExec.SshException raised if the SSH connection cannot be
     *         established
     */
    public static RemoteExec.Result migrateVM(final XenHost sourceHost, final String vmName, final String targetHostName)
        throws RemoteExec.SshException {
        String command = "migrateVM " + vmName + " " + targetHostName;
        XenSSH.logger.debug("host " + sourceHost.getHostName() + " Launching command " + command);
        return RemoteExec.commandAsRoot(sourceHost.getHostName(), sourceHost.getSshAuthInfo(), command);
    }

}
