/**
 * JASMINe VMMapi: JASMINe Virtual Machine Management API
 * Copyright (C) 2009 France Telecom R&D
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: JMXNotificationAppender.java 3177 2009-03-20 13:30:34Z alitokmen $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.vmm.agent.jmx;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CopyOnWriteArrayList;

import javax.management.ListenerNotFoundException;
import javax.management.MBeanNotificationInfo;
import javax.management.MBeanServer;
import javax.management.Notification;
import javax.management.NotificationBroadcaster;
import javax.management.NotificationFilter;
import javax.management.NotificationListener;
import javax.management.ObjectName;

import org.apache.log4j.AppenderSkeleton;
import org.apache.log4j.spi.LoggingEvent;
import org.ow2.jasmine.vmm.agent.main.AgentCommon;
import org.ow2.jasmine.vmm.agent.main.VirtManagerAgent;

/**
 * The JMXNotificationAppender MBean acts as a log4j appender and emits JMX
 * notifications for every log4j message
 * <p>
 * The <tt>ObjectName</tt> for identifying the unique JMXNotificationAppender of
 * a VMM agent follows the following template: <blockquote>
 * org.ow2.jasmine.vmm.agent:type=Logger </blockquote>
 */
public class JMXNotificationAppender extends AppenderSkeleton implements NotificationBroadcaster, JMXNotificationAppenderMBean {
    private String objectName;

    private ObjectName oName;

    private String notificationType;

    private boolean isStarted;

    private long notificationSequence;

    private List<ListenerInfo> notificationListeners = new CopyOnWriteArrayList<ListenerInfo>();

    public String getObjectName() {
        return objectName;
    }

    public void setObjectName(final String objectName) {
        this.objectName = objectName;
    }

    public String getNotificationType() {
        return notificationType;
    }

    public void setNotificationType(final String notificationType) {
        this.notificationType = notificationType;
    }

    private boolean start() {
        if (isStarted) {
            return true;
        }
        try {
            MBeanServer mbeanServer = AgentCommon.getMBeanServer();
            if (mbeanServer != null) {
                oName = ObjectName.getInstance(objectName);
                mbeanServer.registerMBean(this, oName);
                isStarted = true;
            }
        } catch (Exception ex) {
            errorHandler.error("Cannot register JMXNotificationAppenderMBean: " + ex);
        }
        return isStarted;
    }

    @Override
    protected synchronized void append(final LoggingEvent event) {
        if (start()) {
            StringBuffer message = new StringBuffer(layout.format(event));
            if (layout.ignoresThrowable() && event.getThrowableStrRep() != null) {
                for (String exceptionStringRep : event.getThrowableStrRep()) {
                    message.append(exceptionStringRep);
                }
            }
            Notification notification = new Notification(notificationType, oName, notificationSequence++, message.toString());
            fireNotification(notification);
        }
    }

    @Override
    public void close() {
        if (isStarted) {
            try {
                MBeanServer mbeanServer = AgentCommon.getMBeanServer();
                if (mbeanServer != null) {
                    mbeanServer.unregisterMBean(oName);
                    isStarted = true;
                }
            } catch (Exception ex) {
                ex.printStackTrace();
            }
        }
        isStarted = false;
    }

    @Override
    public boolean requiresLayout() {
        return true;
    }

    private void fireNotification(final Notification notif) {
        try {
            for (ListenerInfo li : notificationListeners) {
                li.listener.handleNotification(notif, li.handback);
            }
        } catch (Exception ex) {
            ex.printStackTrace();
        }
    }

    public void addNotificationListener(final NotificationListener listener, final NotificationFilter filter, final Object handback)
        throws IllegalArgumentException {
        if (listener == null) {
            throw new IllegalArgumentException("Listener can't be null");
        }
        notificationListeners.add(new ListenerInfo(listener, filter, handback));
    }

    public MBeanNotificationInfo[] getNotificationInfo() {
        return new MBeanNotificationInfo[] {new MBeanNotificationInfo(new String[] {notificationType}, Notification.class
            .getName(), "Log")};
    }

    public void removeNotificationListener(final NotificationListener listener) throws ListenerNotFoundException {
        ListenerInfo wildcard = new WildcardListenerInfo(listener);
        boolean removed = notificationListeners.removeAll(Collections.singleton(wildcard));
        if (!removed) {
            throw new ListenerNotFoundException("Listener not registered");
        }
    }

    private static class ListenerInfo {
        NotificationListener listener;

        NotificationFilter filter;

        Object handback;

        ListenerInfo(final NotificationListener listener, final NotificationFilter filter, final Object handback) {
            this.listener = listener;
            this.filter = filter;
            this.handback = handback;
        }

        @Override
        public boolean equals(final Object o) {
            if (!(o instanceof ListenerInfo)) {
                return false;
            }
            ListenerInfo li = (ListenerInfo) o;
            if (li instanceof WildcardListenerInfo) {
                return (li.listener == listener);
            } else {
                return (li.listener == listener && li.filter == filter && li.handback == handback);
            }
        }
    }

    private static class WildcardListenerInfo extends ListenerInfo {
        WildcardListenerInfo(final NotificationListener listener) {
            super(listener, null, null);
        }

        @Override
        public boolean equals(final Object o) {
            assert (!(o instanceof WildcardListenerInfo));
            return o.equals(this);
        }
    }
}
