/**
 * JASMINe VMMapi: JASMINe Virtual Machine Management API
 * Copyright (C) 2009 France Telecom R&D
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: Domain.java 6168 2010-03-23 16:44:46Z dangtran $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.vmm.agent.domain;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.management.ObjectName;

import org.apache.log4j.Logger;
import org.ow2.jasmine.vmm.api.DomainMXBean;
import org.ow2.jasmine.vmm.api.InsufficientResourcesException;
import org.ow2.jasmine.vmm.api.InvalidVMConfigException;
import org.ow2.jasmine.vmm.api.ServerPoolMXBean;
import org.ow2.jasmine.vmm.api.VMConfigSpec;
import org.ow2.jasmine.vmm.api.VMMException;
import org.ow2.jasmine.vmm.api.VirtualMachineMXBean;

/**
 * The Domain class is the implementation of the DomainMXBean interface. It
 * represents a domain managed entity acting as container of sub-domains and/or
 * server pools.
 */
public final class Domain extends ManagedResource implements DomainMXBean {
    static private Logger logger = Logger.getLogger(Domain.class);

    private String name;

    private final List<DomainMXBean> subDomains = new ArrayList<DomainMXBean>();

    private final List<ServerPoolMXBean> serverPools = new ArrayList<ServerPoolMXBean>();

    private final Map<String, String> attributes = new HashMap<String, String>();

    /**
     * Constucts a new domain with a given name and JMX object name
     * 
     * @param name user-friendly nale of the domain
     * @param objectName JMX object name of the domain
     */
    public Domain(final String name, final ObjectName objectName, final Map<String, String> attributes) {
        super(objectName);
        this.attributes.putAll(attributes);
        this.name = name;
    }

    /*
     * (non-Javadoc)
     * @see org.ow2.jasmine.vmm.api.DomainMXBean#getName()
     */
    public String getName() {
        return this.name;
    }

    /*
     * (non-Javadoc)
     * @see org.ow2.jasmine.vmm.api.DomainMXBean#getSubDomains()
     */
    public List<DomainMXBean> getSubDomains() {
        return this.subDomains;
    }

    /**
     * @param d
     */
    public void addSubDomain(final Domain d) {
        this.subDomains.add(d);
    }

    /*
     * (non-Javadoc)
     * @see org.ow2.jasmine.vmm.api.DomainMXBean#getAttribute(java.lang.String)
     */
    public String getAttribute(final String key) {
        return this.attributes.get(key);
    }

    /**
     * Add a server pool to the domain
     * 
     * @param pool server pool to add
     */
    public void addServerPool(final ServerPool pool) {
        this.serverPools.add(pool);
    }

    /*
     * (non-Javadoc)
     * @see org.ow2.jasmine.vmm.api.DomainMXBean#getServerPools()
     */
    public List<ServerPoolMXBean> getServerPools() {
        return this.serverPools;
    }

    private boolean areConstraintsSatisfied(final Map<String, String> constraints) {
        if (constraints == null) {
            return true;
        }
        String locationConstraint = constraints.get("location");
        if (locationConstraint == null || locationConstraint.equals("")) {
            return true;
        }
        String domainLocation = this.attributes.get("location");
        if (domainLocation == null) {
            return true;
        }
        if (domainLocation.equals(locationConstraint)) {
            return true;
        }
        return false;
    }

    /*
     * (non-Javadoc)
     * @see
     * org.ow2.jasmine.vmm.api.DomainMXBean#provisionVM(org.ow2.jasmine.vmm.
     * api.VMConfigSpec, java.util.Map, boolean)
     */
    public VirtualMachineMXBean provisionVM(final VMConfigSpec vmSpec, final Map<String, String> constraints, final boolean sync)
        throws InsufficientResourcesException, InvalidVMConfigException, VMMException {
        VirtualMachineMXBean vm = null;
        Domain.logger.info("Domain " + this.name + ": provisioning VM(name=" + vmSpec.getName() + ",imageID="
            + vmSpec.getVmImageUUID() + ")...");
        List<VMMException> raisedExceptions = new ArrayList<VMMException>();
        // first we try to provision the VM using one
        // of the server pools of the domain if any

        if (this.areConstraintsSatisfied(constraints)) {
            for (ServerPoolMXBean serverPool : this.serverPools) {
                try {
                    vm = serverPool.provisionVM(vmSpec, constraints, sync);
                    return vm;
                } catch (VMMException ex) {
                    raisedExceptions.add(ex);
                }
            }
        } else {
            throw new InsufficientResourcesException("Cannot meet constraints");
        }

        // next we recursively provision the VM on sub-domains
        for (DomainMXBean subDomain : this.subDomains) {
            try {
                vm = subDomain.provisionVM(vmSpec, constraints, sync);
                return vm;
            } catch (VMMException ex) {
                raisedExceptions.add(ex);
            }
        }
        for (VMMException ex : raisedExceptions) {
            if (ex instanceof InsufficientResourcesException) {
                throw (InsufficientResourcesException) ex;
            }
        }
        for (VMMException ex : raisedExceptions) {
            if (ex instanceof InvalidVMConfigException) {
                throw (InvalidVMConfigException) ex;
            }
        }
        for (VMMException ex : raisedExceptions) {
            throw ex;
        }
        throw new InsufficientResourcesException();
    }

}
