/**
 * JASMINe VMMapi: JASMINe Virtual Machine Management API
 * Copyright (C) 2009 France Telecom R&D
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: DomainBuilder.java 6099 2010-02-23 14:02:43Z dangtran $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.vmm.agent.domain;

import java.io.File;
import java.io.FileInputStream;
import java.util.HashMap;
import java.util.Map;
import java.util.Stack;

import javax.management.ObjectName;
import javax.xml.parsers.SAXParser;
import javax.xml.parsers.SAXParserFactory;

import org.apache.log4j.Logger;
import org.ow2.jasmine.vmm.agent.driver.Driver;
import org.ow2.jasmine.vmm.agent.driver.DriverFactory;
import org.ow2.jasmine.vmm.agent.jmx.MBeanObjectNamer;
import org.ow2.jasmine.vmm.agent.main.AgentCommon;
import org.ow2.jasmine.vmm.api.VMMException;
import org.xml.sax.Attributes;
import org.xml.sax.SAXException;
import org.xml.sax.helpers.DefaultHandler;

/**
 * The class DomainBuilder constructs Domain objects from XML specifications.
 * TODO check the syntax of the XML file with a schema
 */
public final class DomainBuilder {
    static Logger logger = Logger.getLogger(DomainBuilder.class);

    static class DomainXMLHandler extends DefaultHandler {
        Stack<Domain> domainStack = new Stack<Domain>();

        Domain rootDomain, currentDomain;

        ServerPool currentServerPool;

        StringBuffer currentPath = new StringBuffer();

        @Override
        public void startElement(final String uri, final String localName, final String qName, final Attributes attrs) {
            if (qName.equals("domain")) {
                String name = attrs.getValue("name");
                if (name == null) {
                    DomainBuilder.logger.error("Missing domain name, skipping domain definition");
                    this.currentDomain = null;
                    return;
                }
                this.currentPath.append("/" + name);

                ObjectName domainObjectName = MBeanObjectNamer.makeDomainName(this.currentPath.toString());
                Domain domain = new Domain(name, domainObjectName, DomainBuilder.attributes2Map(attrs));
                try {
                    AgentCommon.getMBeanServer().registerMBean(domain, domainObjectName);
                } catch (Exception ex) {
                    DomainBuilder.logger.error("Skipping domain " + name, ex);
                    this.currentDomain = null;
                    return;
                }
                DomainBuilder.logger.info("Added Domain " + domainObjectName);
                if (this.domainStack.isEmpty()) {
                    this.rootDomain = domain;
                } else {
                    this.domainStack.peek().addSubDomain(domain);
                }
                this.domainStack.push(domain);
                this.currentDomain = domain;
            } else if (qName.equals("serverPool")) {
                if (this.currentDomain != null) {
                    String name = attrs.getValue("name");
                    String driverName = attrs.getValue("driver");
                    if (name == null) {
                        DomainBuilder.logger.error("skipped server pool because of missing name");
                        return;
                    }
                    if (driverName == null) {
                        DomainBuilder.logger.error("skipped server pool because of missing driver name");
                        return;
                    }
                    ObjectName serverPoolObjectName = null;
                    serverPoolObjectName = MBeanObjectNamer.makeServerPoolName(this.currentPath.toString() + "/" + name);
                    Driver driver = DriverFactory.getInstance().newDriver(driverName);
                    if (driver != null) {
                        try {
                            this.currentServerPool = driver.newServerPool(name, serverPoolObjectName, DomainBuilder
                                .attributes2Map(attrs));
                        } catch (IllegalArgumentException ex) {
                            DomainBuilder.logger.error("Skipping ServerPool " + name + ":" + ex.getMessage());
                            this.currentServerPool = null;
                            return;
                        } catch (VMMException ex) {
                            DomainBuilder.logger.error("Skipping ServerPool " + name + ":" + ex.getMessage());
                            this.currentServerPool = null;
                            return;
                        }
                        try {
                            AgentCommon.getMBeanServer().registerMBean(this.currentServerPool, serverPoolObjectName);
                        } catch (Exception ex) {
                            DomainBuilder.logger.error("Skipping ServerPool " + name, ex);
                            this.currentServerPool = null;
                            return;
                        }
                        DomainBuilder.logger.info("Added ServerPool " + serverPoolObjectName);
                        this.currentDomain.addServerPool(this.currentServerPool);
                    } else {
                        DomainBuilder.logger.error("Driver " + driverName + " not found");
                        this.currentServerPool = null;
                    }
                }
            } else if (qName.equals("host")) {
                String hostName = attrs.getValue("name");
                if (hostName == null) {
                    DomainBuilder.logger.error("Missing host name, skipping host definition");
                    return;
                }
                if (this.currentServerPool != null) {
                    try {
                        this.currentServerPool.newHost(hostName, DomainBuilder.attributes2Map(attrs));
                    } catch (IllegalArgumentException ex) {
                        DomainBuilder.logger.error("Skipping host " + hostName + ":" + ex.getMessage());
                    }
                }
            }
        }

        @Override
        public void endElement(final java.lang.String uri, final java.lang.String localName, final java.lang.String qName)
            throws SAXException {
            if (qName.equals("domain")) {
                if (this.currentDomain != null) {
                    this.domainStack.pop();
                    this.currentPath.delete(this.currentPath.lastIndexOf("/"), this.currentPath.length());
                }
                if (this.domainStack.isEmpty()) {
                    this.currentDomain = null;
                } else {
                    this.currentDomain = this.domainStack.peek();
                }
            } else if (qName.equals("serverPool")) {
                this.currentServerPool = null;
            }
        }

    }

    static private Map<String, String> attributes2Map(final Attributes attrs) {
        Map<String, String> map = new HashMap<String, String>();
        for (int i = 0; i < attrs.getLength(); i++) {
            String name = attrs.getLocalName(i);
            String val = attrs.getValue(i);
            map.put(name, val);
        }
        return map;
    }

    /**
     * Constructs a new Domain object from an XML description
     * 
     * @param file the file containing of the XML description
     * @return the domain object or null if the domain could not be created.
     */
    public static Domain createManagedResourcesFromXML(final File file) {
        try {
            DomainXMLHandler handler = new DomainXMLHandler();
            SAXParser parser = SAXParserFactory.newInstance().newSAXParser();
            DomainBuilder.logger.info("Reading resource description file " + file.getAbsolutePath());
            parser.parse(new FileInputStream(file), handler);
            return handler.rootDomain;
        } catch (Exception ex) {
            DomainBuilder.logger.error("While parsing resource description file:", ex);
            return null;
        }
    }

}
