/**
 * JASMINe VMMapi: JASMINe Virtual Machine Management API
 * Copyright (C) 2009 France Telecom R&D
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: DummyServerPool.java 6104 2010-02-23 14:22:07Z dangtran $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.vmm.agent.driver.dummy;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import javax.management.ObjectName;

import org.apache.log4j.Logger;
import org.ow2.jasmine.vmm.agent.domain.ManagedResource;
import org.ow2.jasmine.vmm.agent.domain.ServerPool;
import org.ow2.jasmine.vmm.agent.jmx.MBeanObjectNamer;
import org.ow2.jasmine.vmm.agent.main.AgentCommon;
import org.ow2.jasmine.vmm.api.HostMXBean;
import org.ow2.jasmine.vmm.api.VMMException;
import org.ow2.jasmine.vmm.api.VirtualMachineImageMXBean;
import org.ow2.jasmine.vmm.api.VirtualMachineImageStoreMXBean;

/**
 * Dummy driver ServerPool MXBean implementation
 */
class DummyServerPool extends ServerPool {
    static protected Logger logger = Logger.getLogger(DummyServerPool.class);

    private static VirtualMachineImageStoreMXBean imageStore;

    public DummyServerPool(final String name, final ObjectName objectName) {
        super(name, objectName);
        if (DummyServerPool.imageStore == null) {
            ObjectName storeObjectName = MBeanObjectNamer.makeVMImageStoreName(name);
            DummyServerPool.imageStore = new DummyVirtualMachineImageStore(storeObjectName);
            try {
                AgentCommon.getMBeanServer().registerMBean(DummyServerPool.imageStore, storeObjectName);
            } catch (Exception ex) {
                DummyServerPool.logger.error("Cannot register Dummy VMImageStore", ex);
            }
        }
    }

    @Override
    public HostMXBean newHost(final String hostName, final Map<String, String> props) {
        ObjectName objectName = MBeanObjectNamer.makeHostName(this.getPath() + "/" + hostName, hostName);
        DummyHost host = new DummyHost(this, hostName, objectName);
        try {
            AgentCommon.getMBeanServer().registerMBean(host, objectName);
        } catch (Exception ex) {
            DummyServerPool.logger.error("Unable to register MBean " + objectName, ex);
            return null;
        }
        this.addHost(host);
        return host;
    }

    @Override
    public VirtualMachineImageStoreMXBean getVMImageStore() {
        return DummyServerPool.imageStore;
    }

    static private class DummyVirtualMachineImage extends ManagedResource implements VirtualMachineImageMXBean {
        private String uuid;

        private String name;

        private String metaData;

        public DummyVirtualMachineImage(final ObjectName objectName, final String uuid, final String name) {
            super(objectName);
            this.metaData = this.metaData;
            this.name = name;
            this.uuid = uuid;
        }

        public String getUUID() {
            return this.uuid;
        }

        public String getName() {
            return this.name;
        }

        public Map<String, String> getMetaData() {
            return null;
        }

        public String getDescription() {
            return "a dummy image";
        }

    }

    static class DummyVirtualMachineImageStore extends ManagedResource implements VirtualMachineImageStoreMXBean {
        private static long INIT_STORAGE_CAPACITY_MB = 1024 * 128;

        private ArrayList<DummyVirtualMachineImage> images;

        private long freeSpaceMB;

        public DummyVirtualMachineImageStore(final ObjectName objectName) {
            super(objectName);
            this.images = new ArrayList<DummyVirtualMachineImage>();
            this.addImage("vmiTest", "Used by unit tests");
            this.addImage("vmi222", "Debian Lenny");
            this.addImage("vmi333", "Fedora 10");
            this.freeSpaceMB = DummyVirtualMachineImageStore.INIT_STORAGE_CAPACITY_MB;
        }

        private void addImage(final String uuid, final String name) {
            try {
                ObjectName on = MBeanObjectNamer.makeVMImageName(uuid);
                DummyVirtualMachineImage image = new DummyVirtualMachineImage(on, uuid, name);
                AgentCommon.getMBeanServer().registerMBean(image, on);
                this.images.add(image);
            } catch (Exception ex) {
                DummyServerPool.logger.error("Failed to register VMImage with uuid=" + uuid, ex);
            }
        }

        public long getCapacityMB() {
            return DummyVirtualMachineImageStore.INIT_STORAGE_CAPACITY_MB;
        }

        public long getFreeSpaceMB() {
            return this.freeSpaceMB;
        }

        void updateFreeSpace(final long deltaMB) {
            this.freeSpaceMB += deltaMB;
        }

        public String getName() {
            return "ImageStore(default)";
        }

        public VirtualMachineImageMXBean lookUpByUUID(final String uuid) {
            for (DummyVirtualMachineImage image : this.images) {
                if (image.getUUID().equals(uuid)) {
                    return image;
                }
            }
            return null;
        }

        public List<VirtualMachineImageMXBean> listVMImageTemplates() {
            ArrayList<VirtualMachineImageMXBean> result = new ArrayList<VirtualMachineImageMXBean>(this.images);
            return result;
        }

        public void removeVMImageTemplate(final VirtualMachineImageMXBean image) throws VMMException {
            for (VirtualMachineImageMXBean im : this.images) {
                if (im.getUUID().equals(image.getUUID())) {
                    this.images.remove(image);
                }
            }
        }

    }

}
