/**
 * JASMINe VMMapi: JASMINe Virtual Machine Management API
 * Copyright (C) 2009 France Telecom R&D
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: DummyVirtualMachine.java 6104 2010-02-23 14:22:07Z dangtran $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.vmm.agent.driver.dummy;

import java.net.InetAddress;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Random;

import javax.management.MBeanNotificationInfo;
import javax.management.Notification;
import javax.management.ObjectName;

import org.apache.log4j.Logger;
import org.ow2.jasmine.vmm.agent.domain.ManagedResource;
import org.ow2.jasmine.vmm.api.BadVMPowerStateException;
import org.ow2.jasmine.vmm.api.HostMXBean;
import org.ow2.jasmine.vmm.api.IllegalOperationException;
import org.ow2.jasmine.vmm.api.InsufficientResourcesException;
import org.ow2.jasmine.vmm.api.NotificationTypes;
import org.ow2.jasmine.vmm.api.ResourceUsage;
import org.ow2.jasmine.vmm.api.VMCustomizationSpec;
import org.ow2.jasmine.vmm.api.VMMException;
import org.ow2.jasmine.vmm.api.VirtualMachineImageMXBean;
import org.ow2.jasmine.vmm.api.VirtualMachineMXBean;

/**
 * Dummy driver VirtualMachine MXBean implementation
 */
class DummyVirtualMachine extends ManagedResource implements VirtualMachineMXBean {
    static Logger logger = Logger.getLogger(DummyVirtualMachine.class);

    private String name;

    private DummyHost host;

    private String uuid;

    private String ipAddress;

    private int numVCPU;

    private int schedulingCap, schedulingWeight;

    private int diskSizeMB;

    private long memorySizeMB, memoryUsedMB;

    private PowerState status = PowerState.HALTED;

    private Date startTime;

    private boolean cpuAffinity[][];

    public DummyVirtualMachine(final String name, final String uuid, final ObjectName objectName, final DummyHost host) {
        super(objectName);
        this.name = name;
        this.host = host;
        this.uuid = uuid;
        this.memorySizeMB = 4096 * 2;

        this.startTime = new Date(System.currentTimeMillis());

        this.currentResourceUsage = new ResourceUsage();
        List<ResourceUsage.DiskStats> diskStats = new ArrayList<ResourceUsage.DiskStats>();
        diskStats.add(new ResourceUsage.DiskStats());
        diskStats.get(0).setDeviceName("disk");
        List<ResourceUsage.NetworkStats> netStats = new ArrayList<ResourceUsage.NetworkStats>();
        netStats.add(new ResourceUsage.NetworkStats());
        netStats.get(0).setDeviceName("net");
        this.currentResourceUsage.setDiskStats(diskStats);
        this.currentResourceUsage.setNetworkStats(netStats);

        try {
            this.ipAddress = InetAddress.getByName("198.7.9.66").getHostAddress();
        } catch (Exception ex) {
            DummyVirtualMachine.logger.error("Failed to get IP address", ex);
        }
    }

    public Date getStartTime() {
        return this.startTime;
    }

    public long getUpTimeSeconds() {
        return (System.currentTimeMillis() - this.startTime.getTime()) / 1000;
    }

    public int getNumVCPUs() {
        return this.numVCPU;
    }

    public void setNumVCPUs(final int numVCPU) {
        this.numVCPU = numVCPU;
    }

    public int getDiskSizeMB() {
        return this.diskSizeMB;
    }

    public void setDiskSizeMB(final int diskSizeMB) {
        this.diskSizeMB = diskSizeMB;
    }

    public long getMemorySizeMB() {
        return this.memorySizeMB;
    }

    public long getMemoryUsedMB() {
        return this.memoryUsedMB;
    }

    public void setMemorySizeMB(final long memorySizeMB) {
        this.memorySizeMB = memorySizeMB;
        this.memoryUsedMB = memorySizeMB / 2;
    }

    Map<String, String> userData = new HashMap<String, String>();

    public void addUserData(final String key, final String value) {
        this.userData.put(key, value);
    }

    public String getUserData(final String key) {
        return this.userData.get(key);
    }

    public String getConsole() {
        return null;
    }

    public void onVMStatusChanged() {
        DummyVirtualMachine.logger.info("VM " + this.getNameLabel() + " state=" + this.getState());
        this.emitNotification(NotificationTypes.VM_STATE_CHANGE, this.getState().toString(), null);
    }

    public boolean canLiveMigrateToHost(final HostMXBean targetHost) throws VMMException {
        return targetHost != this.host && this.getMemorySizeMB() <= targetHost.getFreeMemoryMB();
    }

    public void destroy() {
        try {
            Thread.sleep(3000);
        } catch (Exception ex) {
        }
        this.host.onDestroyVM(this);
    }

    public float[] getLoadPerVCPU() {
        return new float[] {8, 10};
    }

    public float getCPULoad() {
        if (this.status == PowerState.RUNNING) {
            return this.currentResourceUsage.getCpuLoad();
        } else {
            return 0;
        }
    }

    public HostMXBean getHostMBean() {
        return this.host;
    }

    public String getGuestIpAddress() {
        return this.ipAddress;
    }

    public String getMacAddress() {
        return "8E:10:65:AB:CE:26";
    }

    public int getMemory() {
        return 0;
    }

    public String getNameLabel() {
        return this.name;
    }

    public int getSchedulingCap() {
        return this.schedulingCap;
    }

    public int getSchedulingWeight() {
        return this.schedulingWeight;
    }

    public PowerState getState() {
        return this.status;
    }

    public String getUuid() {
        return this.uuid;
    }

    public void reboot() {
        DummyVirtualMachine.logger.info("Rebooting VM " + this.name);
    }

    public void resume() {
        if (this.status == PowerState.SUSPENDED) {
            this.status = PowerState.RUNNING;
            this.onVMStatusChanged();
            DummyVirtualMachine.logger.info("VM " + this.name + " resumed");
        }
    }

    public void setMacAddress(final String mac) {
    }

    public void setSchedulingCap(final int schedulingCap) {
        this.schedulingCap = schedulingCap;
    }

    public void setSchedulingWeight(final int schedulingWeight) {
        this.schedulingWeight = schedulingWeight;
    }

    @Override
    public boolean[][] getCPUAffinity() {
        if (this.cpuAffinity == null) {
            this.cpuAffinity = new boolean[this.numVCPU][];
            for (int i = 0; i < this.numVCPU; i++) {
                this.cpuAffinity[i] = new boolean[this.host.getNumCPU()];
                for (int j = 0; j < this.host.getNumCPU(); j++) {
                    this.cpuAffinity[i][j] = true;
                }
            }
        }
        return this.cpuAffinity;
    }

    @Override
    public void setCPUAffinity(final boolean[][] affinity) {
        boolean[][] cpuAffinity = this.getCPUAffinity();
        for (int i = 0; i < this.numVCPU; i++) {
            for (int j = 0; j < this.host.getNumCPU(); j++) {
                this.cpuAffinity[i][j] = affinity[i][j];
            }
        }
    }

    public void shutdown() {
        if (this.status == PowerState.RUNNING) {
            this.status = PowerState.HALTED;
            this.onVMStatusChanged();
            DummyVirtualMachine.logger.info("VM " + this.name + " shutdowned");
        }
    }

    public void start() {
        if (this.status == PowerState.HALTED) {
            this.status = PowerState.RUNNING;
            this.onVMStatusChanged();
            DummyVirtualMachine.logger.info("VM " + this.name + " started");
        }
    }

    public void suspend() throws BadVMPowerStateException {
        if (this.status == PowerState.RUNNING) {
            this.status = PowerState.SUSPENDED;
            this.onVMStatusChanged();
            DummyVirtualMachine.logger.info("VM " + this.name + " suspended");
        }
    }

    public void pause() throws BadVMPowerStateException {
        if (this.status == PowerState.RUNNING) {
            this.status = PowerState.PAUSED;
            this.onVMStatusChanged();
            DummyVirtualMachine.logger.info("VM " + this.name + " paused");
        }
    }

    public void unpause() throws BadVMPowerStateException {
        if (this.status == PowerState.PAUSED) {
            this.status = PowerState.RUNNING;
            this.onVMStatusChanged();
            DummyVirtualMachine.logger.info("VM " + this.name + " unpaused");
        }
    }

    int count = 0;

    public void migrate(HostMXBean targetHost, final boolean live) throws IllegalOperationException, VMMException {
        String targetHostName = targetHost.getHostName();
        boolean foundHostInServerPool = false;
        // TODO need to found out how to check equality between MXBean whether
        // proxy or local
        for (HostMXBean h : this.host.getServerPool().getManagedHosts()) {
            if (h.getHostName().equals(targetHost.getHostName())) {
                foundHostInServerPool = true;
                targetHost = h;
                break;
            }
        }

        if (targetHost.getFreeMemoryMB() < this.getMemorySizeMB()) {
            throw new VMMException("Not enough memory on target host");
        }

        if (!foundHostInServerPool) {
            throw new IllegalOperationException("Source and target hosts belong to different server pools");
        }

        this.emitNotification(NotificationTypes.VM_MIGRATION_START, targetHostName, this.uuid);
        try {
            Thread.sleep(1000);
        } catch (Exception ex) {
        }

        DummyHost target = (DummyHost) targetHost;
        this.host.postMigrateVM(this, target);
        this.host = target;

        this.emitNotification(NotificationTypes.VM_MIGRATION, targetHostName, this.uuid);
    }

    public VirtualMachineMXBean cloneVM(final String name, final VMCustomizationSpec custSpec, final boolean sync)
        throws InsufficientResourcesException, VMMException {
        return this.host.cloneVM(this, name, custSpec, sync);
    }

    private ResourceUsage currentResourceUsage;

    Random rand = new Random(System.currentTimeMillis());

    float currentCpuLoad = 0;

    public ResourceUsage getResourceUsage() {
        if (this.status == PowerState.RUNNING) {
            this.currentResourceUsage.setCpuLoad(this.currentCpuLoad);// this.rand.nextFloat()
            // *
            // 0.5f);
            this.currentResourceUsage.setMemoryUsedKBytes(this.getMemoryUsedMB() * 1024);
            this.currentResourceUsage.setSamplingTime(new Date(System.currentTimeMillis()));
            this.currentResourceUsage.getDiskStats().get(0).setDiskReadKBytePerSec(this.rand.nextInt(10));
            this.currentResourceUsage.getDiskStats().get(0).setDiskWrittenKBytesPerSec(this.rand.nextInt(10));
            this.currentResourceUsage.getNetworkStats().get(0).setNetReceivedKbitPerSec(this.rand.nextInt(10));
            this.currentResourceUsage.getNetworkStats().get(0).setNetTransmittedKbitPerSec(this.rand.nextInt(10));
        } else {
            this.currentResourceUsage.setCpuLoad(0);
            this.currentResourceUsage.setMemoryUsedKBytes(0);
            this.currentResourceUsage.setSamplingTime(new Date(System.currentTimeMillis()));
            this.currentResourceUsage.getDiskStats().get(0).setDiskReadKBytePerSec(0);
            this.currentResourceUsage.getDiskStats().get(0).setDiskWrittenKBytesPerSec(0);
            this.currentResourceUsage.getNetworkStats().get(0).setNetReceivedKbitPerSec(0);
            this.currentResourceUsage.getNetworkStats().get(0).setNetTransmittedKbitPerSec(0);
        }
        return this.currentResourceUsage;
    }

    // @Override
    @Override
    public MBeanNotificationInfo[] getNotificationInfo() {
        return new MBeanNotificationInfo[] {new MBeanNotificationInfo(new String[] {NotificationTypes.VM_STATE_CHANGE,
            NotificationTypes.VM_MIGRATION, NotificationTypes.LOG, NotificationTypes.ERROR}, Notification.class.getName(),
            "VM event")};
    }

    public VirtualMachineImageMXBean makeTemplate(final String name, final String description,
        final Map<String, String> metadata) throws InsufficientResourcesException, IllegalOperationException,
        BadVMPowerStateException, VMMException {
        throw new UnsupportedOperationException();
    }
}
