package org.ow2.jasmine.vmm.agent.driver.hyperv;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.Timer;
import java.util.TimerTask;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

import javax.management.InstanceAlreadyExistsException;
import javax.management.InstanceNotFoundException;
import javax.management.MBeanNotificationInfo;
import javax.management.MBeanRegistrationException;
import javax.management.NotCompliantMBeanException;
import javax.management.Notification;
import javax.management.NotificationEmitter;
import javax.management.ObjectName;

import org.apache.log4j.Logger;
import org.ow2.jasmine.vmm.agent.domain.ManagedResource;
import org.ow2.jasmine.vmm.agent.driver.hyperv.HyperVConnection.VM;
import org.ow2.jasmine.vmm.agent.driver.hyperv.HyperVVMImageStore.HypervVirtualMachineImage;
import org.ow2.jasmine.vmm.agent.driver.util.ResourceUsageHelper;
import org.ow2.jasmine.vmm.agent.jmx.MBeanObjectNamer;
import org.ow2.jasmine.vmm.agent.main.AgentCommon;
import org.ow2.jasmine.vmm.api.HostMXBean;
import org.ow2.jasmine.vmm.api.InvalidVMConfigException;
import org.ow2.jasmine.vmm.api.NotificationTypes;
import org.ow2.jasmine.vmm.api.ResourceUsage;
import org.ow2.jasmine.vmm.api.ServerPoolMXBean;
import org.ow2.jasmine.vmm.api.VMConfigSpec;
import org.ow2.jasmine.vmm.api.VMMException;
import org.ow2.jasmine.vmm.api.VirtualMachineImageStoreMXBean;
import org.ow2.jasmine.vmm.api.VirtualMachineMXBean;
import org.ow2.jasmine.vmm.api.VirtualMachineMXBean.PowerState;

public class HyperVHost extends ManagedResource implements HostMXBean, NotificationEmitter {
    static Logger logger = Logger.getLogger(HyperVHost.class);

    // the thread pool size determines the maximum number of
    // simultaneous VM creation on the same host
    private static final int MAX_PARALELL_VM_CREATION_PER_HOST = 3;

    // one extra thread for the event reader
    private static final int THREADPOOL_SIZE = 1 + HyperVHost.MAX_PARALELL_VM_CREATION_PER_HOST;

    private static final int DEFAULT_MONITORING_PERIOD_MS = 1000;

    private static ExecutorService executorService = Executors.newFixedThreadPool(HyperVHost.THREADPOOL_SIZE);

    private HashMap<String, HyperVVirtualMachine> vmList = new HashMap<String, HyperVVirtualMachine>();

    private HyperVServerPool serverPool;

    private HyperVConnection connection;

    private Timer perfTimer;

    private String hostName, vmFolderPath, vmTemplateFolderPath, disk;

    private int numCPU = -1;

    public static class HypervParams {

        private final String hypervUser;

        private final String hypervPassword;

        public HypervParams(final String hypervUser, final String hypervPassword) {
            this.hypervUser = hypervUser;
            this.hypervPassword = hypervPassword;
        }

        public String getHypervUser() {
            return this.hypervUser;
        }

        public String getHypervPassword() {
            return this.hypervPassword;
        }

    }

    public static HyperVHost newHost(final HyperVServerPool resourcePool, final ObjectName objectName, final String hostName,
        final String vmFolderPath, final String vmTemplateFolderPath, final HypervParams hypervParams, final String disk) {

        HyperVConnection connection = new HyperVConnection(hostName, hypervParams.getHypervUser(), hypervParams
            .getHypervPassword());

        HyperVHost hypervHost = new HyperVHost(resourcePool, connection, objectName, hostName, vmFolderPath,
            vmTemplateFolderPath, disk);

        return hypervHost;
    }

    private HyperVHost(final HyperVServerPool serverPool, final HyperVConnection connection, final ObjectName objectName,
        final String hostName, final String vmFolderPath, final String vmTemplateFolderPath, final String disk) {
        super(objectName);
        this.serverPool = serverPool;
        this.connection = connection;
        this.hostName = hostName;
        this.vmFolderPath = vmFolderPath;
        this.vmTemplateFolderPath = vmTemplateFolderPath;
        this.disk = disk;
        // this.startPerfMonitor();

    }

    public boolean isConnectionLost() {
        return !this.connection.isConnected();
    }

    public HyperVConnection getHypervConnection() {
        return this.connection;
    }

    public String getVMFolderpath() {
        return this.vmFolderPath;
    }

    public String getVMTemplateFolderpath() {
        return this.vmTemplateFolderPath;
    }

    private void addVM(final HyperVVirtualMachine vm) {
        this.vmList.put(vm.getNameLabel(), vm);
    }

    private void removeVM(final HyperVVirtualMachine vm) {
        this.vmList.remove(vm.getNameLabel());
    }

    public ServerPoolMXBean getServerPool() {
        return this.serverPool;
    }

    // TODO
    public void postMigrateVM(final HyperVVirtualMachine vm, final HyperVHost newHost) {
        // this.removeVM(vm);
        // newHost.addVM(vm);
        throw new UnsupportedOperationException("not implemented");
    }

    synchronized void onVMCreate(final HyperVVirtualMachine vm) throws InstanceAlreadyExistsException,
        MBeanRegistrationException, NotCompliantMBeanException {
        AgentCommon.getMBeanServer().registerMBean(vm, vm.getObjectName());
        HyperVHost.logger.info("Added HypervVirtualMachineMBean " + vm.getObjectName());
        this.addVM(vm);
        this.emitNotification(NotificationTypes.VM_ADD, "Created", vm.getObjectName());

    }

    synchronized void onVMDestroy(final HyperVVirtualMachine vm) throws InstanceNotFoundException, MBeanRegistrationException {

        this.removeVM(vm);
        this.emitNotification(NotificationTypes.VM_DEL, "Destroyed", vm.getObjectName());
        AgentCommon.getMBeanServer().unregisterMBean(vm.getObjectName());
    }

    synchronized void onPowerStateChangedVM(final String vmName, final VirtualMachineMXBean.PowerState state, final long time) {
        HyperVVirtualMachine vm = this.vmList.get(vmName);
        if (vm != null) {
            vm.onVMStateChanged(state);
        }
    }

    private Set<PerfMetric> currentMonitoredMetrics;

    private long currentMonitoringPeriod = HyperVHost.DEFAULT_MONITORING_PERIOD_MS;

    public void configurePerfMonitor(final Set<PerfMetric> metricsOfInterest, long periodMillis) {
        this.stopPerfMonitor();
        if (periodMillis < 1000) {
            periodMillis = 1000;
        }
        if (metricsOfInterest.contains(PerfMetric.VM_CPU_LOAD)) {
            this.currentMonitoringPeriod = periodMillis;
            this.currentMonitoredMetrics = metricsOfInterest;
            this.startPerfMonitor();
        }
    }

    private synchronized void startPerfMonitor() {
        if (this.perfTimer == null) {
            this.perfTimer = new Timer();
            this.perfTimer.schedule(new PerfTimerTask(), 0, this.currentMonitoringPeriod);
            HyperVHost.logger.info("Host " + this.hostName + ": started periodic performance reporting task");
        }
    }

    private synchronized void stopPerfMonitor() {
        if (this.perfTimer != null) {
            this.perfTimer.cancel();
            HyperVHost.logger.info("Host " + this.hostName + ": stopped periodic performance reporting task");
        }
        this.perfTimer = null;
    }

    @Override
    public ObjectName getObjectName() {
        return this.objectName;
    }

    HashMap<String, String> hypervisorInfo;

    // TODO
    public Map<String, String> getHypervisorInfo() {
        if (this.hypervisorInfo == null) {
            this.hypervisorInfo = new HashMap<String, String>();
            this.hypervisorInfo.put("vendor", "Microsoft");
            this.hypervisorInfo.put("name", "Hyper-V");
        }
        return this.hypervisorInfo;
    }

    HashMap<String, String> cpuInfo;

    public Map<String, String> getCPUInfo() {
        if (this.cpuInfo == null) {

            try {
                this.cpuInfo = this.connection.getHostCPUInfo();
                HyperVHost.logger.debug("GetCPUInfo of host : " + this.cpuInfo);
            } catch (Exception ex) {
                HyperVHost.logger.debug("Host " + this.hostName, ex);
            }

        }
        return this.cpuInfo;
    }

    public int getNumCPU() {
        if (this.numCPU == -1) {

            try {
                this.numCPU = this.connection.getHostNumCPU();
                HyperVHost.logger.debug("GetNumCPU of host : " + this.numCPU);
            } catch (Exception ex) {
                HyperVHost.logger.error("Failed to get NumCPU of host " + this.hostName, ex);
            }
        }

        return this.numCPU;
    }

    public String getHostName() {
        return this.hostName;
    }

    public VirtualMachineImageStoreMXBean getVMImageStore() {
        return this.serverPool.getVMImageStore();
    }

    public long getFreeMemoryMB() {
        long freeMem = 0;

        try {
            freeMem = this.connection.getHostFreePhysicalMemoryMB();
            HyperVHost.logger.debug("GetFreeMemoryMB of host : " + freeMem);
        } catch (Exception ex) {
            HyperVHost.logger.error("Failed to get FreeMemoryMB of host " + this.hostName, ex);
        }

        return freeMem;
    }

    public synchronized void sync(final HyperVConnection connection) throws VMMException {
        Set<HyperVConnection.VM> vmSet;

        try {
            vmSet = connection.getAllVM();
        } catch (Exception ex) {
            throw new VMMException(ex.getMessage());
        }

        for (HyperVConnection.VM vm : vmSet) {

            if (!this.vmList.containsKey(vm.getNameLabel())) {
                ObjectName name = MBeanObjectNamer.makeVirtualMachineName(this.serverPool.getPath() + "/" + vm.getNameLabel(),
                    vm.getUuid());

                HyperVVirtualMachine hypervVM = new HyperVVirtualMachine(name, this, this.connection, vm, null);

                if (hypervVM == null) {
                    continue;
                }

                try {
                    this.onVMCreate(hypervVM);
                } catch (Exception ex) {
                    HyperVHost.logger.error("Failed to register VM MBean", ex);
                    continue;
                }

            }
        }

        // get rid of deleted vm
        HashMap<String, HyperVVirtualMachine> vmListTemp = new HashMap<String, HyperVVirtualMachine>(this.vmList);
        for (String nameLabel : vmListTemp.keySet()) {
            boolean match = false;
            for (HyperVConnection.VM vm : vmSet) {

                if (nameLabel.equals(vm.getNameLabel())) {
                    match = true;
                    break;
                }
            }
            if (!match) {
                try {
                    this.onVMDestroy(vmListTemp.get(nameLabel));
                } catch (Exception ex) {
                    HyperVHost.logger.error("Failed to unregister VM MBean", ex);
                }
            }

        }
    }

    public List<VirtualMachineMXBean> getResidentVMs() {
        List<VirtualMachineMXBean> result = new ArrayList<VirtualMachineMXBean>(this.vmList.values());
        return result;
    }

    public VirtualMachineMXBean lookUpByNameLabel(final String nameLabel) {
        if (this.vmList.containsKey(nameLabel)) {
            return this.vmList.get(nameLabel);
        }

        return null;
    }

    public VirtualMachineMXBean createVM(final VMConfigSpec vmSpecIn, final boolean sync) throws InvalidVMConfigException,
        VMMException {
        final VMConfigSpec vmSpec = new VMConfigSpec(vmSpecIn);
        HyperVHost.logger.info("Creating new VM [name=" + vmSpec.getName() + ",memorySizeMB=" + vmSpec.getMemorySizeMB()
            + ",diskSize=" + vmSpec.getDiskSizeMB() + ",numVCPU=" + vmSpec.getNumVCPU() + "]");

        final HypervVirtualMachineImage template = (HypervVirtualMachineImage) this.serverPool.getVMImageStore().lookUpByUUID(
            vmSpec.getVmImageUUID());
        if (template == null) {
            throw new InvalidVMConfigException("Invalid VM Image UUID");
        }
        // XXX we make the vm label unique by appending the current time
        final String vmName = vmSpec.getName() + "-" + System.currentTimeMillis();

        if (!sync) {

            HyperVHost.executorService.execute(new Runnable() {
                public void run() {
                    try {

                        HyperVHost.this.copyFile(vmSpec, vmName, template);
                        HyperVHost.this.createVM2(vmSpec, vmName, template);
                    } catch (Exception ex) {
                    }
                }
            });
            return null;
        } else {
            HyperVHost.this.copyFile(vmSpec, vmName, template);
            return this.createVM2(vmSpec, vmName, template);
        }

    }

    private VirtualMachineMXBean createVM2(final VMConfigSpec vmSpec, final String vmName,
        final HypervVirtualMachineImage template) throws InvalidVMConfigException, VMMException {

        synchronized (this) {
            boolean success = false;
            try {

                success = this.connection.createVM(vmName, (this.getVMFolderpath().concat(vmName.concat(".vhd"))).replaceAll(
                    "\\\\\\\\", "\\\\"));

            } catch (Exception ex) {
                HyperVHost.logger.error("CreateVM failed: ", ex);
                throw new VMMException("VM creation failed: VMI connection failure", ex);
            }
            if (!success) {
                HyperVHost.logger.error("Command createVM failed");
                throw new VMMException("VM creation failed");
            }

            VM newVM = null;
            try {
                Set<HyperVConnection.VM> vms = this.connection.getAllVM();
                for (HyperVConnection.VM vm : vms) {
                    if (vm.getNameLabel().equals(vmName)) {
                        newVM = vm;
                    }
                }
            } catch (Exception ex) {
                HyperVHost.logger.error("Failed to get VM : ", ex);
                throw new VMMException("VM creation failure: Failed to get VM");
            }

            ObjectName name = null;
            HyperVVirtualMachine hypervVM = null;

            try {
                name = MBeanObjectNamer.makeVirtualMachineName(this.serverPool.getPath() + "/" + vmName, newVM.getUuid());
                hypervVM = new HyperVVirtualMachine(name, this, this.connection, newVM, null);
                this.addVM(hypervVM);
                this.emitNotification(NotificationTypes.VM_ADD, "Created", hypervVM.getObjectName());
                AgentCommon.getMBeanServer().registerMBean(hypervVM, name);
            } catch (Exception ex) {
                HyperVHost.logger.debug("Host " + this.hostName, ex);
                throw new VMMException(ex);
            }
            try {
                hypervVM.setNumVCPUs(vmSpec.getNumVCPU());
                hypervVM.setMemorySizeMB(vmSpec.getMemorySizeMB());

            } catch (Exception ex) {
                HyperVHost.logger.error("Failed modify VM : ", ex);
                throw new VMMException("VM creation failure: Failed modify VM");
            }

            return hypervVM;
        }

    }

    private void copyFile(final VMConfigSpec vmSpec, final String vmName, final HypervVirtualMachineImage template)
        throws VMMException {
        boolean success = false;
        try {

            success = this.connection.hostCopyFile(template.getFileName().concat(".vhd"), this.getVMFolderpath().concat(
                vmName.concat(".vhd")));

        } catch (Exception ex) {
            HyperVHost.logger.error("CopyFile failed ", ex);
            throw new VMMException("VM creation failed: WMI connection failure", ex);
        }
        if (!success) {
            HyperVHost.logger.error("Command copyFile failed");
            throw new VMMException("VM creation failed");
        }
    }

    public long getTotalMemoryMB() {
        long totalMem = 0;

        try {
            totalMem = this.connection.getHostTotalMemoryMB();
            HyperVHost.logger.debug("GetTotalMemoryMB of host : " + totalMem);
        } catch (Exception ex) {
            HyperVHost.logger.error("Failed to get FreeTotalMemoryMB of host " + this.hostName, ex);
        }

        return totalMem;
    }

    public float[] getLoadPerCPU() {
        float[] loadPerCPU = new float[0];

        try {
            loadPerCPU = this.connection.getHostCPULoad();
            HyperVHost.logger.debug("GetLoadPerCPU of host : ");
            for (float load : loadPerCPU) {
                HyperVHost.logger.debug(load);
            }
        } catch (Exception ex) {
            HyperVHost.logger.error("Failed to get LoadPerCPU of host " + this.hostName, ex);
        }

        return loadPerCPU;
    }

    public float getCPULoad() {
        float cpuLoad = 0;

        try {
            float[] loadPerCPU = this.connection.getHostCPULoad();
            float acc = 0;
            for (float load : loadPerCPU) {
                acc += load;
            }
            cpuLoad = acc / loadPerCPU.length;
            HyperVHost.logger.debug("GetCPULoad of host : " + cpuLoad);
        } catch (Exception ex) {
            HyperVHost.logger.error("Failed to get CPULoad of host " + this.hostName, ex);
        }

        return cpuLoad;
    }

    @Override
    public Map<String, ResourceUsage> getVMResourceUsage() {
        HashMap<String, ResourceUsage> result = new HashMap<String, ResourceUsage>();
        for (HyperVVirtualMachine vm : this.vmList.values()) {
            result.put(vm.getNameLabel(), vm.getResourceUsage());
        }
        return result;
    }

    @Override
    public MBeanNotificationInfo[] getNotificationInfo() {
        return new MBeanNotificationInfo[] {new MBeanNotificationInfo(new String[] {NotificationTypes.VM_STATE_CHANGE,
            NotificationTypes.VM_ADD, NotificationTypes.VM_DEL, NotificationTypes.ERROR, NotificationTypes.LOG},
            Notification.class.getName(), "Host event")};
    }

    public Map<String, Float> getVMCPULoads() {
        HashMap<String, Float> cpuLoads = new HashMap<String, Float>();

        try {
            cpuLoads = this.connection.getVMCPULoads();
        } catch (Exception ex) {
            HyperVHost.logger.error("Host " + this.hostName + ", Failed to get VMCPULoads", ex);
        }
        return cpuLoads;

    }

    private class PerfTimerTask extends TimerTask {
        @Override
        public void run() {
            // send notification
            Map<String, Object> notifUserData = new HashMap<String, Object>();
            for (HyperVVirtualMachine vm : HyperVHost.this.vmList.values()) {
                if (vm.getState() == PowerState.RUNNING && vm.getResourceUsage() != null) {
                    notifUserData.put(vm.getNameLabel(), ResourceUsageHelper.serialize(vm.getResourceUsage()));
                }
            }
            HyperVHost.this.emitNotification(NotificationTypes.PERF_REPORT, "Resource Usage", notifUserData);
        }
    }
}
