package org.ow2.jasmine.vmm.agent.driver.hyperv;

import java.util.Map;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

import javax.management.ObjectName;

import org.ow2.jasmine.vmm.agent.domain.ServerPool;
import org.ow2.jasmine.vmm.agent.jmx.MBeanObjectNamer;
import org.ow2.jasmine.vmm.agent.main.AgentCommon;
import org.ow2.jasmine.vmm.api.HostMXBean;
import org.ow2.jasmine.vmm.api.VirtualMachineImageStoreMXBean;

public class HyperVServerPool extends ServerPool {

    public static int EVENT_COLLECTOR_PERIOD;

    public static boolean LEGACY_NETWORK_ADAPTER;

    public static final String LEGACY_NETWORK_NAME = "Legacy Network Adapter";

    public static final String SYNTHETIC_NETWORK_NAME = "VMBus Network Adapter";

    private static final int THREADPOOL_SIZE = 20;

    private static ExecutorService executorService = Executors.newFixedThreadPool(HyperVServerPool.THREADPOOL_SIZE);

    private HyperVEventCollector eventCollector;

    private String hypervUser;

    private String hypervPassword;

    private VirtualMachineImageStoreMXBean imageStore;

    private String vmFolder, vmTemplateFolder, disk;

    public HyperVServerPool(final String name, final ObjectName objectName, final Map<String, String> attributes)
        throws IllegalArgumentException {
        super(name, objectName);
        this.hypervUser = attributes.get("user");
        this.hypervPassword = attributes.get("password");
        String vmFolderPath = attributes.get("vmFolderPath");
        if (vmFolderPath == null) {
            throw new IllegalArgumentException("Missing parameter: vmFolderPath");
        }
        this.vmFolder = (vmFolderPath.concat("\\")).replaceAll("\\\\", "\\\\\\\\");
        String vmTemplateFolderPath = attributes.get("vmTemplateFolderPath");
        if (vmTemplateFolderPath == null) {
            throw new IllegalArgumentException("Missing parameter: vmTemplateFolderPath");
        }
        this.vmTemplateFolder = (vmTemplateFolderPath.concat("\\")).replaceAll("\\\\", "\\\\\\\\");
        String isLegacyNetAdapter = attributes.get("legacyNetworkAdapter");
        HyperVServerPool.LEGACY_NETWORK_ADAPTER = isLegacyNetAdapter != null ? Boolean.parseBoolean(isLegacyNetAdapter) : true;
        String period = attributes.get("syncPeriodMillis");
        try {
            HyperVServerPool.EVENT_COLLECTOR_PERIOD = period != null ? Integer.parseInt(period) : 10 * 1000;
        } catch (NumberFormatException ex) {
            throw new IllegalArgumentException("Wrong value for parameter syncPeriodMillis");
        }
        this.disk = this.vmFolder.substring(0, 2);
        this.eventCollector = new HyperVEventCollector(this);
    }

    public ExecutorService getExecutorService() {
        return HyperVServerPool.executorService;
    }

    @Override
    public HostMXBean newHost(final String hostName, final Map<String, String> attributes) {
        try {
            for (HostMXBean host : this.hosts) {
                if (host.getHostName().equals(hostName)) {
                    return host;
                }
            }
            String user = attributes.get("user");
            if (user == null) {
                user = this.hypervUser;
            }
            if (user == null) {
                throw new IllegalArgumentException("Host: " + hostName + " Missing parameter: user");
            }
            String password = attributes.get("password");
            if (password == null) {
                password = this.hypervPassword;
            }
            if (password == null) {
                throw new IllegalArgumentException("Host: " + hostName + " Missing parameter: password");
            }
            ObjectName mbeanObjectName = MBeanObjectNamer.makeHostName(this.getPath() + "/" + hostName, hostName);

            HyperVHost host = HyperVHost.newHost(this, mbeanObjectName, hostName, this.vmFolder, this.vmTemplateFolder,
                new HyperVHost.HypervParams(user, password), this.disk);
            if (host != null) {
                AgentCommon.getMBeanServer().registerMBean(host, mbeanObjectName);

                this.hosts.add(host);

                ServerPool.logger.info("In server pool " + this.name + ": added Host MBean " + mbeanObjectName);
                if (this.imageStore == null) {
                    String vmImageStoreName = hostName + "-VMIStore";
                    this.imageStore = new HyperVVMImageStore(MBeanObjectNamer.makeVMImageStoreName(vmImageStoreName),
                        this.vmTemplateFolder, this.disk, host.getHypervConnection());
                    AgentCommon.getMBeanServer().registerMBean(this.imageStore, this.imageStore.getObjectName());
                }
            }

            this.eventCollector.add(host, new HyperVConnection(hostName, user, password));

            return host;

        } catch (Exception ex) {
            ex.printStackTrace();
            ServerPool.logger.error("Failed to add HyperVHost", ex);
            return null;
        }
    }

    @Override
    public VirtualMachineImageStoreMXBean getVMImageStore() {
        return this.imageStore;
    }

    public synchronized void stopEventCollector() {
        this.eventCollector.stop();
    }

}
