/**
 * JASMINe VMM Agent : JASMINe Virtual Machine Management Agent
 * Copyright (C) 2010 Bull SAS
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id$
 * --------------------------------------------------------------------------
 */

package org.ow2.jasmine.vmm.agent.driver.libvirt;

import java.io.ByteArrayInputStream;
import java.util.ArrayList;
import java.util.List;

import javax.xml.namespace.QName;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.xpath.XPath;
import javax.xml.xpath.XPathConstants;
import javax.xml.xpath.XPathExpression;
import javax.xml.xpath.XPathExpressionException;
import javax.xml.xpath.XPathFactory;

import org.apache.log4j.Logger;
import org.w3c.dom.Document;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

/**
 * LibvirtConfigFileUtils is a XML Parser dedicated to Libvirt Driver, To get
 * stats of Network Interfaces or/and Disk Devices, we need to know the name of
 * the device. The only way to get the name is to parse VM XML config file.
 */
public class LibvirtConfigFileUtils {
    static Logger logger = Logger.getLogger(LibvirtConfigFileUtils.class);

    public static Object getConfigItem(final String xmlConfig, final String xpathExpression, final QName returnType) {
        Document xmlDocument;
        XPath xPath;

        try {
            xmlDocument = DocumentBuilderFactory.newInstance().newDocumentBuilder().parse(
                new ByteArrayInputStream(xmlConfig.getBytes("UTF-8")));
            xPath = XPathFactory.newInstance().newXPath();
        } catch (Exception ex) {
            LibvirtConfigFileUtils.logger.error(ex);
            return null;
        }

        try {
            XPathExpression xPathExpression = xPath.compile(xpathExpression);
            return xPathExpression.evaluate(xmlDocument, returnType);
        } catch (XPathExpressionException ex) {
            LibvirtConfigFileUtils.logger.error(ex);
            return null;
        }
    }

    /**
     * Get the name of every disk devices plugged on the VM.
     * 
     * @param xmlConfig The VM XML config file
     * @return A list of Disk Device name
     */
    public static List<String> getDiskDevices(final String xmlConfig) {
        ArrayList<String> result = new ArrayList<String>();
        NodeList nl = (NodeList) LibvirtConfigFileUtils.getConfigItem(xmlConfig, "/domain/devices/disk/target",
            XPathConstants.NODESET);
        for (int i = 0; i < nl.getLength(); i++) {
            Node n = nl.item(i);
            result.add(n.getAttributes().getNamedItem("dev").getNodeValue());
        }
        return result;
    }

    /**
     * Get the name of every disk file plugged on the VM.
     * 
     * @param xmlConfig The VM XML config file
     * @return A list of Disk file name
     */
    public static List<String> getDiskFiles(final String xmlConfig) {
        ArrayList<String> result = new ArrayList<String>();
        NodeList nl = (NodeList) LibvirtConfigFileUtils.getConfigItem(xmlConfig, "/domain/devices/disk/source",
            XPathConstants.NODESET);
        for (int i = 0; i < nl.getLength(); i++) {
            Node n = nl.item(i);
            result.add(n.getAttributes().getNamedItem("file").getNodeValue());
        }
        return result;
    }

    /**
     * Get the name of every Network Interfaces plugged on the VM.
     * 
     * @param xmlConfig The VM XML config file
     * @return A list of Network Device name
     */
    public static List<String> getNetDevices(final String xmlConfig) {
        ArrayList<String> result = new ArrayList<String>();
        NodeList nl = (NodeList) LibvirtConfigFileUtils.getConfigItem(xmlConfig, "/domain/devices/interface/target",
            XPathConstants.NODESET);
        for (int i = 0; i < nl.getLength(); i++) {
            Node n = nl.item(i);
            result.add(n.getAttributes().getNamedItem("dev").getNodeValue());
        }
        return result;
    }

}
