package org.ow2.jasmine.vmm.agent.driver.libvirt;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

import javax.management.MBeanNotificationInfo;
import javax.management.Notification;
import javax.management.NotificationEmitter;
import javax.management.ObjectName;

import org.apache.log4j.Logger;
import org.libvirt.Connect;
import org.libvirt.Domain;
import org.libvirt.LibvirtException;
import org.libvirt.NodeInfo;
import org.ow2.jasmine.vmm.agent.domain.ManagedResource;
import org.ow2.jasmine.vmm.agent.driver.libvirt.ImageCatalog.VirtualMachineImage;
import org.ow2.jasmine.vmm.agent.driver.util.RemoteExec;
import org.ow2.jasmine.vmm.agent.jmx.MBeanObjectNamer;
import org.ow2.jasmine.vmm.agent.main.AgentCommon;
import org.ow2.jasmine.vmm.api.HostMXBean;
import org.ow2.jasmine.vmm.api.InsufficientResourcesException;
import org.ow2.jasmine.vmm.api.InvalidVMConfigException;
import org.ow2.jasmine.vmm.api.NotificationTypes;
import org.ow2.jasmine.vmm.api.ResourceUsage;
import org.ow2.jasmine.vmm.api.ServerPoolMXBean;
import org.ow2.jasmine.vmm.api.VMConfigSpec;
import org.ow2.jasmine.vmm.api.VMCustomizationSpec;
import org.ow2.jasmine.vmm.api.VMMException;
import org.ow2.jasmine.vmm.api.VirtualMachineImageStoreMXBean;
import org.ow2.jasmine.vmm.api.VirtualMachineMXBean;
import org.ow2.jasmine.vmm.api.VnicIPSettings;
import org.ow2.jasmine.vmm.api.VnicIPSettings.IpAssignmentMode;

public class LibvirtHost extends ManagedResource implements HostMXBean, NotificationEmitter {
    static Logger logger = Logger.getLogger(LibvirtHost.class);

    // the thread pool size determines the maximum number of
    // simultaneous VM creation on the same host
    private static final int MAX_PARALEL_VM_CREATION_PER_HOST = 3;

    private static final int THREADPOOL_SIZE = LibvirtHost.MAX_PARALEL_VM_CREATION_PER_HOST;

    private static ExecutorService executorService = Executors.newFixedThreadPool(LibvirtHost.THREADPOOL_SIZE);

    private List<LibvirtVirtualMachine> vmList = new CopyOnWriteArrayList<LibvirtVirtualMachine>();

    LibvirtServerPool serverPool;

    private LibvirtPerfCollector perfCollectionTask;

    private String hostName;

    private Connect connection;

    private int numCPU = -1;

    private long cpuFrequencyMhz = -1;

    private RemoteExec.SshAuthInfo authInfo;

    private HostSynchronizer hostSynchronizer;

    public static LibvirtHost newHost(final LibvirtServerPool serverPool, final ObjectName objectName, final String hostName,
        final String uri, final RemoteExec.SshAuthInfo authInfo) {

        try {
            LibvirtHost.logger.debug("Connecting to libvirt with URI " + uri);
            Connect connection = new Connect(uri);
            return new LibvirtHost(serverPool, connection, objectName, hostName, authInfo);
        } catch (LibvirtException ex) {
            LibvirtHost.logger.error("Failed to establish libvirt connection with uri " + uri, ex);
            return null;
        }
    }

    private LibvirtHost(final LibvirtServerPool serverPool, final Connect connection, final ObjectName objectName,
        final String hostName, final RemoteExec.SshAuthInfo authInfo) {
        super(objectName);
        this.serverPool = serverPool;
        this.connection = connection;
        this.hostName = hostName;
        this.authInfo = authInfo;
        this.hostSynchronizer = new HostSynchronizer(serverPool.getSyncPeriodMillis());
        this.hostSynchronizer.start();
        this.startPerfMonitor();
    }

    public boolean isConnectionLost() {
        return true; // TODO
    }

    public RemoteExec.SshAuthInfo getSshAuthInfo() {
        return this.authInfo;
    }

    Connect getLibvirtAPIConnection() {
        return this.connection;
    }

    List<LibvirtVirtualMachine> getVMs() {
        return this.vmList;
    }

    LibvirtVirtualMachine lookUpVMByName(final String name) {
        for (LibvirtVirtualMachine vm : this.vmList) {
            if (vm.getNameLabel().equals(name)) {
                return vm;
            }
        }
        return null;
    }

    private void addVM(final LibvirtVirtualMachine vm) {
        this.vmList.add(vm);
    }

    private void removeVM(final LibvirtVirtualMachine vm) {
        this.vmList.remove(vm);
    }

    public ServerPoolMXBean getServerPool() {
        return this.serverPool;
    }

    public void postMigrateVM(final LibvirtVirtualMachine vm, final LibvirtHost newHost) {
        this.removeVM(vm);
        newHost.addVM(vm);
    }

    public void onVMDestroy(final LibvirtVirtualMachine vm) {
        this.removeVM(vm);
        this.emitNotification(NotificationTypes.VM_DEL, "Destroyed", vm.getObjectName());
    }

    private Set<PerfMetric> currentMonitoredMetrics;

    private long currentMonitoringPeriod;

    public void configurePerfMonitor(final Set<PerfMetric> metricsOfInterest, long periodMillis) {
        this.stopPerfMonitor();
        if (periodMillis < 1000) {
            periodMillis = 1000;
        }
        if (metricsOfInterest.contains(PerfMetric.VM_CPU_LOAD)) {
            this.currentMonitoringPeriod = periodMillis;
            this.currentMonitoredMetrics = metricsOfInterest;
            this.startPerfMonitor();
        }
    }

    String getIPAddress(String macAddress) {
        macAddress = macAddress.toUpperCase();

        LibvirtHost.logger.debug("Determining IP address from MAC address " + macAddress + " ...");

        RemoteExec.Result result = null;
        String command = "getIPfromMAC.sh " + macAddress;

        LibvirtHost.logger.debug("Launching command: " + command);

        try {
            result = RemoteExec.commandAsRoot(this.hostName, this.authInfo, command);
        } catch (RemoteExec.SshException ex) {
            LibvirtHost.logger.debug("SSH failure", ex);
            return null;
        }

        if (result.exitCode != 0) {
            LibvirtHost.logger.error("Cannot get IP from MAC " + result.output);
            return null;
        }

        String ipAddress = result.output;
        if (ipAddress != null && ipAddress.length() > 0 && ipAddress.endsWith("\n")) {
            ipAddress = ipAddress.substring(0, ipAddress.length() - 1);
        }
        LibvirtHost.logger.debug("Mac-to-IP " + macAddress + " -> " + ipAddress);
        return ipAddress;
    }

    private synchronized void startPerfMonitor() {
        if (this.perfCollectionTask == null) {
            this.perfCollectionTask = new LibvirtPerfCollector(this);
            this.perfCollectionTask.start();
        }
    }

    private synchronized void stopPerfMonitor() {
        if (this.perfCollectionTask != null) {
            this.perfCollectionTask.stop();
            this.perfCollectionTask = null;
        }
    }

    @Override
    public ObjectName getObjectName() {
        return this.objectName;
    }

    private HashMap<String, String> hypervisorInfo;

    public Map<String, String> getHypervisorInfo() {
        if (this.hypervisorInfo == null) {
            this.hypervisorInfo = new HashMap<String, String>();
            try {
                this.hypervisorInfo.put("name", this.connection.getType());
                long v = this.connection.getVersion();
                long major = v / 1000000;
                long minor = (v - major * 1000000) / 1000;
                this.hypervisorInfo.put("version", "" + major + "." + minor);
            } catch (LibvirtException ex) {
                LibvirtHost.logger.error("Failed to get hypervisor info", ex);
            }
        }
        return this.hypervisorInfo;
    }

    private HashMap<String, String> cpuInfo;

    public Map<String, String> getCPUInfo() {
        if (this.cpuInfo == null) {
            this.cpuInfo = new HashMap<String, String>();
            NodeInfo nodeInfo;
            try {
                nodeInfo = this.connection.nodeInfo();
            } catch (LibvirtException e) {
                e.printStackTrace();
                return this.cpuInfo;
            }
            this.cpuInfo.put("model", new String(nodeInfo.model));
            this.cpuInfo.put("speedMHz", Integer.toString(nodeInfo.mhz));
            this.numCPU = nodeInfo.cpus;
        }
        return this.cpuInfo;
    }

    public int getNumCPU() {
        if (this.numCPU == -1) {
            this.getCPUInfo();
        }
        return this.numCPU;
    }

    public String getHostName() {
        return this.hostName;
    }

    public VirtualMachineImageStoreMXBean getVMImageStore() {
        return this.serverPool.getVMImageStore();
    }

    public long getFreeMemoryMB() {
        long freeMemKB = -1;
        try {
            freeMemKB = this.connection.getFreeMemory();
        } catch (LibvirtException ex) {
            LibvirtHost.logger.error(ex);
        }
        return freeMemKB / (1024 * 1024);
    }

    public void sync() {
        this.hostSynchronizer.sync();
    }

    private synchronized void doSync() {
        try {
            String vmNames[] = this.connection.listDefinedDomains();
            for (String vmName : vmNames) {
                Domain domain = this.connection.domainLookupByName(vmName);
                String uuid = domain.getUUIDString();
                ObjectName name = MBeanObjectNamer.makeVirtualMachineName(this.serverPool.getPath() + "/" + vmName, uuid);
                if (AgentCommon.getMBeanServer().isRegistered(name)) {
                    LibvirtVirtualMachine vm = this.lookUpVMByName(vmName);
                    if (vm != null) {
                        vm.setSynchronized(true);
                    }
                    continue;
                }
                LibvirtVirtualMachine vm = new LibvirtVirtualMachine(name, this, domain, null);
                this.addVM(vm);
                AgentCommon.getMBeanServer().registerMBean(vm, name);
                this.emitNotification(NotificationTypes.VM_ADD, "Created", name);
            }

            int domIDs[] = this.connection.listDomains();
            for (int domID : domIDs) {
                Domain domain = this.connection.domainLookupByID(domID);
                String uuid = domain.getUUIDString();
                String vmName = domain.getName();
                if (vmName.equals("Domain-0")) {
                    uuid = this.hostName; // XXX
                }
                ObjectName name = MBeanObjectNamer.makeVirtualMachineName(this.serverPool.getPath() + "/" + vmName, uuid);
                if (AgentCommon.getMBeanServer().isRegistered(name)) {
                    LibvirtVirtualMachine vm = this.lookUpVMByName(vmName);
                    if (vm != null) {
                        vm.setSynchronized(true);
                    }
                    continue;
                }
                LibvirtVirtualMachine vm = new LibvirtVirtualMachine(name, this, domain, null);
                this.addVM(vm);
                AgentCommon.getMBeanServer().registerMBean(vm, name);
                this.emitNotification(NotificationTypes.VM_ADD, "Created", name);
            }

            for (LibvirtVirtualMachine vm : this.vmList) {
                try {
                    if (vm.isMigrating()) {
                        continue;
                    }
                    if (!vm.isSynchronized()) {
                        this.removeVM(vm);
                        this.emitNotification(NotificationTypes.VM_DEL, "Destroyed", vm.getObjectName());
                        AgentCommon.getMBeanServer().unregisterMBean(vm.getObjectName());
                    } else {
                        vm.getState();
                    }
                } finally {
                    vm.setSynchronized(false);
                }
            }

        } catch (Exception ex) {
            LibvirtHost.logger.error(ex);
        }
    }

    public List<VirtualMachineMXBean> getResidentVMs() {
        List<VirtualMachineMXBean> result = new ArrayList<VirtualMachineMXBean>(this.vmList);
        return result;
    }

    public VirtualMachineMXBean createVM(final VMConfigSpec vmSpecIn, final boolean sync) throws InvalidVMConfigException,
        VMMException {
        final VMConfigSpec vmSpec = new VMConfigSpec(vmSpecIn);
        LibvirtHost.logger.info("Creating new VM [name=" + vmSpec.getName() + ",memorySizeMB=" + vmSpec.getMemorySizeMB()
            + ",diskSize=" + vmSpec.getDiskSizeMB() + ",numVCPU=" + vmSpec.getNumVCPU() + "]");

        final VirtualMachineImage template = (VirtualMachineImage) this.serverPool.getVMImageStore().lookUpByUUID(
            vmSpec.getVmImageUUID());
        if (template == null) {
            throw new InvalidVMConfigException("Invalid VM Image UUID");
        }

        final String vmName = vmSpec.getName();

        if (!sync) {
            LibvirtHost.executorService.execute(new Runnable() {
                public void run() {
                    try {
                        LibvirtHost.this.createVM2(vmSpec, vmName, template);
                    } catch (Exception ex) {
                    }
                }
            });
            return null;
        } else {
            return this.createVM2(vmSpec, vmName, template);
        }
    }

    private VirtualMachineMXBean createVM2(final VMConfigSpec vmSpec, final String vmName, final VirtualMachineImage template)
        throws InvalidVMConfigException, VMMException {

        template.newVM(vmSpec, vmName, this);

        synchronized (this) {
            String vmUUID = null;
            ObjectName name = null;
            LibvirtVirtualMachine vm = null;
            Domain domain = null;
            try {
                domain = this.connection.domainLookupByName(vmName);
                vmUUID = domain.getUUIDString();
                name = MBeanObjectNamer.makeVirtualMachineName(this.serverPool.getPath() + "/" + vmName, vmUUID);
                HashMap<String, String> map = new HashMap<String, String>();
                map.put("imageID", vmSpec.getVmImageUUID());
                domain = this.connection.domainLookupByName(vmName);
                vm = new LibvirtVirtualMachine(name, this, domain, map);
                this.addVM(vm);

                AgentCommon.getMBeanServer().registerMBean(vm, name);
            } catch (Exception ex) {
                LibvirtHost.logger.debug("Host " + this.hostName, ex);
                throw new VMMException(ex);
            }

            this.emitNotification(NotificationTypes.VM_ADD, "Created", name);

            return vm;
        }
    }

    public VirtualMachineMXBean cloneVM(final String targetVmName, final String clonedVmName,
        final VMCustomizationSpec custSpec, final boolean sync) throws InsufficientResourcesException, VMMException {
        String command = "cloneVM --src " + clonedVmName + " --name " + targetVmName + " --force ";
        if (custSpec != null) {
            if (custSpec.getGuestOsHostName() != null && !custSpec.getGuestOsHostName().equals("")) {
                command += " --hostname " + custSpec.getGuestOsHostName();
            }
            // XXX configure only first NIC
            if (custSpec.getVnicIpSettingsList() != null) {
                VnicIPSettings nic0Settings = custSpec.getVnicIpSettingsList().get(0);
                if (nic0Settings.getIpAssignmentMode() == IpAssignmentMode.DHCP) {
                    command += " --net eth0/dhcp";
                } else if (nic0Settings.getIpAssignmentMode() == IpAssignmentMode.FIXED) {
                    command += " --net eth0/static/" + nic0Settings.getIpAddress() + "/" + nic0Settings.getSubnetMask() + "/"
                        + nic0Settings.getGateway();
                }
            }
        }
        RemoteExec.Result result;
        try {
            result = RemoteExec.commandAsRoot(this.getHostName(), this.getSshAuthInfo(), command);
        } catch (RemoteExec.SshException ex) {
            throw new VMMException("VM cloning failed: SSH connection failure", ex);
        }
        if (result.exitCode != 0) {
            throw new VMMException("VM cloning failed: " + result.error);
        }

        Domain newVM = null;
        String vmUUID = "";

        try {
            newVM = this.connection.domainLookupByName(clonedVmName);
            vmUUID = newVM.getUUIDString();
        } catch (LibvirtException ex) {
            LibvirtHost.logger.error("Failed to look up VM " + clonedVmName);
            throw new VMMException("VM cloning failure");
        }

        synchronized (this) {
            ObjectName name = null;
            LibvirtVirtualMachine vm = null;
            try {
                name = MBeanObjectNamer.makeVirtualMachineName(this.serverPool.getPath() + "/" + clonedVmName, vmUUID);
                HashMap<String, String> map = new HashMap<String, String>();
                vm = new LibvirtVirtualMachine(name, this, newVM, map);
                this.addVM(vm);

                AgentCommon.getMBeanServer().registerMBean(vm, name);
            } catch (Exception ex) {
                LibvirtHost.logger.debug("Host " + this.hostName, ex);
                throw new VMMException(ex);
            }

            this.emitNotification(NotificationTypes.VM_ADD, "Created", name);

            return vm;
        }
    }

    public long getTotalMemoryMB() {
        NodeInfo nodeInfo;
        try {
            nodeInfo = this.connection.nodeInfo();
        } catch (LibvirtException e) {
            e.printStackTrace();
            return -1;
        }
        return nodeInfo.memory / 1024;
    }

    // XX unsupported
    public float[] getLoadPerCPU() {
        float[] result = new float[0];
        return result;
    }

    public float getCPULoad() {
        float aggregatedCPULoad = 0;
        for (LibvirtVirtualMachine vm : this.vmList) {
            aggregatedCPULoad += vm.getResourceUsage().getCpuLoad();
        }
        return aggregatedCPULoad;
    }

    @Override
    public MBeanNotificationInfo[] getNotificationInfo() {
        return new MBeanNotificationInfo[] {new MBeanNotificationInfo(new String[] {NotificationTypes.VM_STATE_CHANGE,
            NotificationTypes.VM_ADD, NotificationTypes.VM_DEL, NotificationTypes.ERROR, NotificationTypes.LOG},
            Notification.class.getName(), "Host event")};
    }

    public Map<String, ResourceUsage> getVMResourceUsage() {
        HashMap<String, ResourceUsage> result = new HashMap<String, ResourceUsage>();
        for (LibvirtVirtualMachine vm : this.vmList) {
            result.put(vm.getNameLabel(), vm.getResourceUsage());
        }
        return result;
    }

    public Map<String, Float> getVMCPULoads() {
        HashMap<String, Float> cpuLoads = new HashMap<String, Float>();
        for (LibvirtVirtualMachine vm : this.vmList) {
            cpuLoads.put(vm.getNameLabel(), vm.getResourceUsage().getCpuLoad());
        }
        return cpuLoads;
    }

    private class HostSynchronizer implements Runnable {
        private long periodMillis;

        private volatile boolean stopRequested;

        private Thread thread;

        HostSynchronizer(final long periodMillis) {
            this.periodMillis = periodMillis;
        }

        public void requestStop() {
            this.stopRequested = true;
        }

        void start() {
            this.thread = new Thread(this);
            this.thread.start();
        }

        synchronized void sync() {
            this.notify();
        }

        @Override
        public void run() {
            while (!this.stopRequested) {
                LibvirtHost.this.doSync();
                synchronized (this) {
                    try {
                        this.wait(this.periodMillis);
                    } catch (InterruptedException ex) {
                    }
                }
            }

        }
    }

}
