/**
 * JASMINe VMM Agent : JASMINe Virtual Machine Management Agent
 * Copyright (C) 2010 Bull SAS
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id$
 * --------------------------------------------------------------------------
 */

package org.ow2.jasmine.vmm.agent.driver.libvirt;

import java.util.Date;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

import org.apache.log4j.Logger;
import org.libvirt.Connect;
import org.libvirt.Domain;
import org.libvirt.DomainBlockStats;
import org.libvirt.DomainInterfaceStats;
import org.libvirt.LibvirtException;
import org.ow2.jasmine.vmm.agent.driver.util.ResourceUsageHelper;
import org.ow2.jasmine.vmm.api.NotificationTypes;
import org.ow2.jasmine.vmm.api.ResourceUsage;
import org.ow2.jasmine.vmm.api.VMMException;
import org.ow2.jasmine.vmm.api.VirtualMachineMXBean.PowerState;

/**
 * LibvirtPerfCollector is a thread launch on every Libvirt Host, It use Jna
 * Libvirt to get metrics. It collect metrics, stock them in a
 * LibvirtDomainMetrics tab and, after, update VM ResourceUsage.
 */
public class LibvirtPerfCollector implements Runnable {

    static Logger logger = Logger.getLogger(LibvirtPerfCollector.class);

    /**
     * Interval (in seconds) between time Sample
     */
    static int interval = 3;

    /**
     * Thread state
     */
    private boolean isRunning;

    private Thread thread;

    /**
     * Link to the LibvirtHost who is monitored by the thread
     */
    LibvirtHost host;

    /**
     * Link to the LibvirtApiConnection of the monitored host
     */
    Connect conn;

    /**
     * Tab of LibvirtDomainMetrics, the metrics are stocked in this tab before
     * being convert into RessourceUsage
     */
    LibvirtDomainMetrics[] libvirtDomainMetricsTab;

    int[] listIdVM = null;

    /**
     * Name of VM where metrics are collected (to get VmName when an Exception
     * is caught
     */
    String VmName;

    public LibvirtPerfCollector(final LibvirtHost host) {
        this.host = host;
        this.conn = host.getLibvirtAPIConnection();
        this.thread = new Thread(this);
        this.init();
        LibvirtPerfCollector.logger.info("Libvirt PerfCollector launch on host : " + host.getHostName());

    }

    /**
     * Start the thread
     */
    public void start() {
        this.isRunning = true;
        this.thread.start();

    }

    /**
     * Stop the thread
     */
    public void stop() {
        this.isRunning = false;
    }

    /**
     * Method called initialy and everytime a new VM is detected, it init the
     * libVirtMetrics Tab with the id and name of the active VM.
     */
    private void init() {
        try {
            this.listIdVM = this.conn.listDomains();
            this.libvirtDomainMetricsTab = new LibvirtDomainMetrics[this.listIdVM.length];
            for (int i = 0; i < this.libvirtDomainMetricsTab.length; i++) {
                Domain d = this.conn.domainLookupByID(this.listIdVM[i]);
                this.VmName = d.getName();
                LibvirtDomainMetrics temp = new LibvirtDomainMetrics(i, d.getName());
                this.libvirtDomainMetricsTab[i] = temp;
            }

        } catch (LibvirtException e) {
            LibvirtPerfCollector.logger.error("Error during listing of the available VM on host : " + this.host.getHostName());
            e.printStackTrace();
        }
    }

    @Override
    public void run() {
        while (this.isRunning) {

            this.readAllVMCPUload(1);
            this.readAllVMDiskIO(1);
            this.readAllVMNetIO(1);

            try {
                Thread.sleep(LibvirtPerfCollector.interval * 1000);
            } catch (InterruptedException e1) {
            }

            this.readAllVMCPUload(2);
            this.readAllVMMemoryUsage();
            this.readAllVMDiskIO(2);
            this.readAllVMNetIO(2);

            this.updateAllVM();

            try {
                Thread.sleep(5000);
            } catch (InterruptedException e) {
                e.printStackTrace();
            }

            try {
                if (this.listIdVM.equals(this.conn.listDomains())) {
                    // Do nothing
                } else {
                    this.init();
                }
            } catch (LibvirtException e) {
                // TODO Auto-generated catch block
                e.printStackTrace();
            }

            // send notification
            Map<String, Object> notifUserData = new HashMap<String, Object>();
            for (LibvirtVirtualMachine vm : this.host.getVMs()) {
                try {
                    if (vm.getState() == PowerState.RUNNING && vm.getResourceUsage() != null) {
                        notifUserData.put(vm.getNameLabel(), ResourceUsageHelper.serialize(vm.getResourceUsage()));
                    }
                } catch (VMMException e) {
                }
            }
            this.host.emitNotification(NotificationTypes.PERF_REPORT, "Resource Usage", notifUserData);
        }
    }

    /**
     * Get every VM CPU Load It make a first pass, where every VM CPU time is
     * collected, wait some seconds (defined by the Interval variable) and do a
     * second pass. CpuLoad in % = ((First Sample - Second Sample) / 1000000000
     * * Numbers of VCPU on Hypervisor * Interval between sample (in second))
     */
    private void readAllVMCPUload(final int passNumber) {
        try {
            if (passNumber == 1) {
                for (LibvirtDomainMetrics element : this.libvirtDomainMetricsTab) {
                    Domain d = this.conn.domainLookupByName(element.getName());
                    this.VmName = d.getName();
                    element.setCpuTime(d.getInfo().cpuTime);
                    element.cpuSamplingTimeMillis = System.currentTimeMillis();
                }
            } else {
                for (LibvirtDomainMetrics element : this.libvirtDomainMetricsTab) {
                    Domain d = this.conn.domainLookupByName(element.getName());
                    this.VmName = d.getName();

                    long diffEchantCPU = (element.getCpuTime() - d.getInfo().cpuTime);
                    double deltaTime = (System.currentTimeMillis() - element.cpuSamplingTimeMillis) * 1000000;
                    double denom = (this.host.getNumCPU() * deltaTime);
                    double result1 = -(diffEchantCPU / denom);
                    element.setCpuLoad((float) result1);
                }
            }
        } catch (LibvirtException e) {
            LibvirtPerfCollector.logger.error("LibvirtException during Cpu Load collect on VM : " + this.VmName);
            e.printStackTrace();
        }

    }

    /**
     * Get every VM Memory Usage
     */
    private void readAllVMMemoryUsage() {
        try {
            for (LibvirtDomainMetrics element : this.libvirtDomainMetricsTab) {
                Domain d = this.conn.domainLookupByName(element.getName());
                this.VmName = d.getName();
                element.setMemUsed(d.getInfo().memory);
            }
        } catch (LibvirtException e) {
            LibvirtPerfCollector.logger.error("LibvirtException during Memory Usage collect on VM : " + this.VmName);
            e.printStackTrace();
        }
    }

    /**
     * Get every VM Network I/O
     */
    private void readAllVMDiskIO(final int passNumber) {

        try {
            for (LibvirtDomainMetrics element : this.libvirtDomainMetricsTab) {
                Domain d = this.conn.domainLookupByName(element.getName());
                this.VmName = d.getName();
                String xmlDesc = d.getXMLDesc(0);
                // XXX we skip HVM domains because of a bug in libvirt 0.6.x
                if (xmlDesc.indexOf("hvm") != -1) {
                    continue;
                }

                List<ResourceUsage.DiskStats> list = element.getDiskStatsList();
                if (list == null) {
                    list = new LinkedList<ResourceUsage.DiskStats>();
                    element.setDiskStatsList(list);
                }
                List<String> disks = LibvirtConfigFileUtils.getDiskDevices(xmlDesc);

                element.diskStatsSamplingTimeMillis = System.currentTimeMillis();
                for (String disk : disks) {
                    ResourceUsage.DiskStats diskStats = null;
                    for (ResourceUsage.DiskStats ds : list) {
                        if (ds.getDeviceName().equals(disk)) {
                            diskStats = ds;
                            break;
                        }
                    }
                    if (diskStats == null) {
                        diskStats = new ResourceUsage.DiskStats();
                        list.add(diskStats);
                    }
                    diskStats.setDeviceName(disk);
                    DomainBlockStats stats = d.blockStats(disk);
                    if (passNumber == 1) {
                        diskStats.setDiskReadKBytePerSec(stats.rd_bytes);
                        diskStats.setDiskWrittenKBytesPerSec(stats.wr_bytes);
                    } else {
                        long deltaTimeMillis = (System.currentTimeMillis() - element.diskStatsSamplingTimeMillis);
                        diskStats.setDiskReadKBytePerSec(1000 * (stats.rd_bytes - diskStats.getDiskReadKBytePerSec())
                            / (1024 * deltaTimeMillis));
                        diskStats.setDiskWrittenKBytesPerSec(1000 * (stats.wr_bytes - diskStats.getDiskWrittenKBytesPerSec())
                            / (1024 * deltaTimeMillis));
                    }
                }

            }
        } catch (LibvirtException e) {
            LibvirtPerfCollector.logger.debug("LibvirtException during Disk I/O collect on VM : " + this.VmName);
        }

    }

    /**
     * Get every VM Network I/O
     */
    private void readAllVMNetIO(final int passNumber) {
        try {
            for (LibvirtDomainMetrics element : this.libvirtDomainMetricsTab) {
                Domain d = this.conn.domainLookupByName(element.getName());
                this.VmName = d.getName();
                String xmlDesc = d.getXMLDesc(0);

                List<ResourceUsage.NetworkStats> list = element.getNetworksStatsList();
                if (list == null) {
                    list = new LinkedList<ResourceUsage.NetworkStats>();
                    element.setNetworksStatsList(list);
                }
                List<String> ethList = LibvirtConfigFileUtils.getNetDevices(xmlDesc);

                element.netStatsSamplingTimeMillis = System.currentTimeMillis();
                for (String eth : ethList) {
                    ResourceUsage.NetworkStats netStats = null;
                    for (ResourceUsage.NetworkStats ns : list) {
                        if (ns.getDeviceName().equals(eth)) {
                            netStats = ns;
                            break;
                        }
                    }
                    if (netStats == null) {
                        netStats = new ResourceUsage.NetworkStats();
                        list.add(netStats);
                    }
                    netStats.setDeviceName(eth);
                    DomainInterfaceStats stats = d.interfaceStats(eth);
                    if (passNumber == 1) {
                        netStats.setNetReceivedKbitPerSec(stats.rx_bytes);
                        netStats.setNetTransmittedKbitPerSec(stats.tx_bytes);
                    } else {
                        long deltaTimeMillis = (System.currentTimeMillis() - element.netStatsSamplingTimeMillis);
                        netStats.setNetReceivedKbitPerSec(1000 * (stats.rx_bytes - netStats.getNetReceivedKbitPerSec())
                            / (128 * deltaTimeMillis));
                        netStats.setNetTransmittedKbitPerSec(1000 * (stats.tx_bytes - netStats.getNetTransmittedKbitPerSec())
                            / (128 * deltaTimeMillis));
                    }
                }
            }
        } catch (LibvirtException e) {
            LibvirtPerfCollector.logger.debug("LibvirtException during Network I/O collect on VM : " + this.VmName);
            e.printStackTrace();
        }
    }

    /**
     * Update ResourceUsage of each VM with the metrics collected.
     */
    private void updateAllVM() {
        Date samplingTime = new Date(System.currentTimeMillis());
        for (LibvirtVirtualMachine vm : this.host.getVMs()) {
            for (LibvirtDomainMetrics element : this.libvirtDomainMetricsTab) {
                if (element.getName().equals(vm.getNameLabel())) {
                    ResourceUsage vmUsage = new ResourceUsage();
                    vmUsage.setSamplingTime(samplingTime);
                    vmUsage.setCpuLoad(element.getCpuLoad());
                    vmUsage.setMemoryUsedKBytes(element.getMemUsed());
                    vmUsage.setDiskStats(element.getDiskStatsList());
                    vmUsage.setNetworkStats(element.getNetworksStatsList());
                    vm.updateCurrentResourceUsage(vmUsage);
                }
            }
        }
    }

    /**
     * This class is used to stock Metrics and informations about VM before
     * conversion into ResourceUsage
     */
    private static class LibvirtDomainMetrics {

        /**
         * VM Id
         */
        private int id;

        /**
         * VM Name
         */
        private String name;

        /**
         * VM Cpu Load
         */
        private float cpuLoad = -1;

        private long cpuTime;

        long cpuSamplingTimeMillis;

        /**
         * Memory used by the VM
         */
        private long memUsed = -1;

        long diskStatsSamplingTimeMillis;

        /**
         * List of ResourceUsage for Disk Devices, one for each disk
         */
        private List<ResourceUsage.DiskStats> DiskStatsList = null;

        long netStatsSamplingTimeMillis;

        /**
         * List of ResourceUsage for Network Devices, one for each network
         * interface.
         */
        private List<ResourceUsage.NetworkStats> NetworksStatsList = null;

        public LibvirtDomainMetrics(final int id, final String name) {
            this.id = id;
            this.name = name;
        }

        public int getId() {
            return this.id;
        }

        public String getName() {
            return this.name;
        }

        public float getCpuLoad() {
            return this.cpuLoad;
        }

        public long getMemUsed() {
            return this.memUsed;
        }

        public List<ResourceUsage.DiskStats> getDiskStatsList() {
            return this.DiskStatsList;
        }

        public List<ResourceUsage.NetworkStats> getNetworksStatsList() {
            return this.NetworksStatsList;
        }

        public synchronized long getCpuTime() {
            return this.cpuTime;
        }

        public synchronized void setCpuTime(final long cpuTime) {
            this.cpuTime = cpuTime;
        }

        public void setCpuLoad(final float cpuLoad) {
            this.cpuLoad = cpuLoad;
        }

        public void setMemUsed(final long memUsed) {
            this.memUsed = memUsed;
        }

        public void setDiskStatsList(final List<ResourceUsage.DiskStats> diskStats) {
            this.DiskStatsList = diskStats;
        }

        public void setNetworksStatsList(final List<ResourceUsage.NetworkStats> networkStats) {
            this.NetworksStatsList = networkStats;
        }
    }

}
