package org.ow2.jasmine.vmm.agent.driver.libvirt;

import java.util.Map;

import javax.management.ObjectName;

import org.apache.log4j.Logger;
import org.ow2.jasmine.vmm.agent.domain.ServerPool;
import org.ow2.jasmine.vmm.agent.driver.util.RemoteExec;
import org.ow2.jasmine.vmm.agent.jmx.MBeanObjectNamer;
import org.ow2.jasmine.vmm.agent.main.AgentCommon;
import org.ow2.jasmine.vmm.api.HostMXBean;
import org.ow2.jasmine.vmm.api.VirtualMachineImageStoreMXBean;

public class LibvirtServerPool extends ServerPool {
    static Logger logger = Logger.getLogger(LibvirtServerPool.class);

    static long DEFAULT_HOST_SYNC_PERIOD_MILLIS = 10000;

    private RemoteExec.SshAuthInfo defaultAuthInfo;

    private String vmImageStoreDir, diskStoreDir;

    private String vmImageStoreHostname;

    private VirtualMachineImageStoreMXBean imageStore;

    private long syncPeriodMillis = LibvirtServerPool.DEFAULT_HOST_SYNC_PERIOD_MILLIS;

    public LibvirtServerPool(final String name, final ObjectName objectName, final Map<String, String> attributes)
        throws IllegalArgumentException {
        super(name, objectName);
        String sshPassword = attributes.get("sshRootPassword");
        String sshPrivateKeyFile = attributes.get("sshPrivateKeyFile");
        if (sshPassword != null || sshPrivateKeyFile != null) {
            this.defaultAuthInfo = new RemoteExec.SshAuthInfo(sshPassword, sshPrivateKeyFile);
        }
        this.vmImageStoreDir = attributes.get("sharedImageStore");
        if (this.vmImageStoreDir == null) {
            throw new IllegalArgumentException("Missing parameter: vmImageStoreDir");
        }
        if (this.vmImageStoreDir != null) {
            this.vmImageStoreHostname = attributes.get("imageStoreHost");
        }
        this.diskStoreDir = attributes.get("sharedDiskStore");
        if (this.diskStoreDir == null) {
            throw new IllegalArgumentException("Missing parameter: sharedDiskStore");
        }
        String period = attributes.get("syncPeriodMillis");
        if (period != null) {
            try {
                this.syncPeriodMillis = Integer.parseInt(period);
            } catch (NumberFormatException ex) {
                throw new IllegalArgumentException("Wrong value for parameter syncPeriodMillis");
            }
        }
    }

    @Override
    public HostMXBean newHost(final String hostName, final Map<String, String> props) {
        try {
            for (HostMXBean host : this.hosts) {
                if (host.getHostName().equals(hostName)) {
                    return host;
                }
            }
            ObjectName mbeanObjectName = MBeanObjectNamer.makeHostName(this.getPath() + "/" + hostName, hostName);
            RemoteExec.SshAuthInfo hostAuthInfo;
            if (props.get("sshRootPassword") != null) {
                hostAuthInfo = new RemoteExec.SshAuthInfo(props.get("sshRootPassword"), null);
            } else if (props.get("sshPrivateKeyFile") != null) {
                hostAuthInfo = new RemoteExec.SshAuthInfo(null, props.get("sshPrivateKeyFile"));
            } else {
                hostAuthInfo = this.defaultAuthInfo;
            }
            if (hostAuthInfo == null) {
                throw new IllegalArgumentException("Missing SSH authentification info");
            }

            String url = props.get("url");
            if (url == null) {
                throw new IllegalArgumentException("Missing parameter: url");
            }

            LibvirtHost host = LibvirtHost.newHost(this, mbeanObjectName, hostName, url, hostAuthInfo);
            if (host != null) {
                AgentCommon.getMBeanServer().registerMBean(host, mbeanObjectName);
                this.hosts.add(host);
                LibvirtServerPool.logger.info("In server pool " + this.name + ": added Host MBean " + mbeanObjectName);
                if (this.imageStore == null) {
                    ObjectName defaultDataStoreName = MBeanObjectNamer.makeVMImageStoreName(hostName + ".default");
                    this.imageStore = new ImageCatalog(defaultDataStoreName, this.vmImageStoreDir, hostName,
                        this.defaultAuthInfo);
                    AgentCommon.getMBeanServer().registerMBean(this.imageStore, defaultDataStoreName);
                }
            }

            return host;
        } catch (Exception ex) {
            LibvirtServerPool.logger.error("Failed to add LibvirtHost", ex);
            return null;
        }
    }

    @Override
    public VirtualMachineImageStoreMXBean getVMImageStore() {
        return this.imageStore;
    }

    public String getSharedDiskStoreDir() {
        return this.diskStoreDir;
    }

    public String getVmImageStoreHostname() {
        return this.vmImageStoreHostname;
    }

    public synchronized long getSyncPeriodMillis() {
        return this.syncPeriodMillis;
    }

}
