package org.ow2.jasmine.vmm.agent.driver.libvirt;

import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;

import javax.management.MBeanNotificationInfo;
import javax.management.Notification;
import javax.management.ObjectName;
import javax.xml.xpath.XPathConstants;

import org.apache.log4j.Logger;
import org.libvirt.Domain;
import org.libvirt.DomainInfo;
import org.libvirt.LibvirtException;
import org.libvirt.SchedParameter;
import org.libvirt.SchedUintParameter;
import org.libvirt.VcpuInfo;
import org.libvirt.DomainInfo.DomainState;
import org.ow2.jasmine.vmm.agent.domain.ManagedResource;
import org.ow2.jasmine.vmm.agent.driver.util.RemoteExec;
import org.ow2.jasmine.vmm.api.BadVMPowerStateException;
import org.ow2.jasmine.vmm.api.HostMXBean;
import org.ow2.jasmine.vmm.api.IllegalOperationException;
import org.ow2.jasmine.vmm.api.InsufficientResourcesException;
import org.ow2.jasmine.vmm.api.NotificationTypes;
import org.ow2.jasmine.vmm.api.ResourceUsage;
import org.ow2.jasmine.vmm.api.VMCustomizationSpec;
import org.ow2.jasmine.vmm.api.VMMException;
import org.ow2.jasmine.vmm.api.VirtualMachineImageMXBean;
import org.ow2.jasmine.vmm.api.VirtualMachineMXBean;

/**
 * Libvirt driver Virtual Machine MXBean implementation
 */
public class LibvirtVirtualMachine extends ManagedResource implements VirtualMachineMXBean {
    static Logger logger = Logger.getLogger(LibvirtVirtualMachine.class);

    private LibvirtHost host;

    private Domain domain;

    private String uuid;

    private String name;

    private String ipAddress;

    private String macAddress;

    private PowerState cachedPowerState = PowerState.UNKNOWN;

    boolean isSynchronized;

    private boolean isMigrating;

    public LibvirtVirtualMachine(final ObjectName objectName, final LibvirtHost host, final Domain domain,
        final Map<String, String> userData) throws VMMException {
        super(objectName);
        this.isSynchronized = true;
        try {
            this.host = host;
            this.domain = domain;
            this.uuid = domain.getUUIDString();
            this.name = domain.getName();
        } catch (LibvirtException ex) {
            throw LibvirtVirtualMachine.translateLibvirtException(ex);
        }
    }

    public boolean canLiveMigrateToHost(final HostMXBean targetHost) throws VMMException {
        LibvirtHost host = null;
        for (HostMXBean h : this.host.getServerPool().getManagedHosts()) {
            if (h.getObjectName().equals(targetHost.getObjectName())) {
                host = (LibvirtHost) h;
                break;
            }
        }
        return host != null && host != this.host && this.getMemorySizeMB() < host.getFreeMemoryMB();
    }

    public synchronized boolean isSynchronized() {
        return this.isSynchronized;
    }

    public synchronized void setSynchronized(final boolean isSynchronized) {
        this.isSynchronized = isSynchronized;
    }

    public synchronized boolean isMigrating() {
        return this.isMigrating;
    }

    public synchronized void setMigrating(final boolean isMigrating) {
        this.isMigrating = isMigrating;
    }

    public VirtualMachineMXBean cloneVM(final String clonedVmName, final VMCustomizationSpec custSpec, final boolean sync)
        throws InsufficientResourcesException, VMMException {
        return this.host.cloneVM(this.getNameLabel(), clonedVmName, custSpec, sync);
    }

    public String getUuid() {
        return this.uuid;
    }

    public HostMXBean getHostMBean() {
        return this.host;
    }

    public long getDomID() {
        try {
            return this.domain.getID();
        } catch (LibvirtException ex) {
            LibvirtVirtualMachine.logger.error(ex);
            return -1;
        }
    }

    public List<String> getDiskFiles() throws VMMException {
        try {
            String xmlDesc = this.domain.getXMLDesc(0);
            return LibvirtConfigFileUtils.getDiskFiles(xmlDesc);
        } catch (LibvirtException ex) {
            throw LibvirtVirtualMachine.translateLibvirtException(ex);
        }
    }

    public Date getStartTime() {
        return null;
    }

    public long getUpTimeSeconds() {
        return -1;
    }

    private ResourceUsage currentResourceUsage;

    void updateCurrentResourceUsage(final ResourceUsage usage) {
        this.currentResourceUsage = usage;
    }

    public ResourceUsage getResourceUsage() {
        return this.currentResourceUsage;
    }

    public float getCPULoad() {
        return this.currentResourceUsage.getCpuLoad();
    }

    public float[] getLoadPerVCPU() throws VMMException {
        try {
            VcpuInfo vcpuInfos1[] = this.domain.getVcpusInfo();
            try {
                Thread.sleep(500);
            } catch (Exception ex) {
            }
            VcpuInfo vcpuInfos2[] = this.domain.getVcpusInfo();
            float vcpuLoad[] = new float[vcpuInfos1.length];
            for (int i = 0; i < vcpuInfos1.length; i++) {
                vcpuLoad[i] = (float) ((vcpuInfos2[i].cpuTime - vcpuInfos1[i].cpuTime) / Math.pow(10, 9));
            }
            return vcpuLoad;
        } catch (LibvirtException ex) {
            throw LibvirtVirtualMachine.translateLibvirtException(ex);
        }
    }

    public long getMemorySizeMB() throws VMMException {
        try {
            return this.domain.getInfo().maxMem / 1024;
        } catch (LibvirtException ex) {
            throw new VMMException(ex);
        }
    }

    public void setMemorySizeMB(final long size) throws VMMException {
        try {
            this.domain.setMaxMemory(size * 1024);
            this.domain.setMemory(size * 1024);
        } catch (LibvirtException ex) {
            throw LibvirtVirtualMachine.translateLibvirtException(ex);
        }
    }

    public long getMemoryUsedMB() throws VMMException {
        try {
            return this.domain.getInfo().memory / 1024;
        } catch (LibvirtException ex) {
            throw LibvirtVirtualMachine.translateLibvirtException(ex);
        }
    }

    public String getNameLabel() {
        return this.name;
    }

    // XX xen-specific
    public void addUserData(final String key, final String value) throws VMMException {
        try {
            RemoteExec.Result result = null;
            String command = "xenstore-write /domain/" + this.uuid + "/" + key + " " + value + " 2>&1";
            LibvirtVirtualMachine.logger.debug("VM " + this.name + " addUserData: " + command);
            result = RemoteExec.commandAsRoot(this.host.getHostName(), this.host.getSshAuthInfo(), command);
            if (result.exitCode != 0) {
                LibvirtVirtualMachine.logger.debug("VM " + this.name + " addUserData failed: " + result.output);
                LibvirtVirtualMachine.logger.error(command + " returns " + result.exitCode);
            }
        } catch (Exception ex) {
            throw new VMMException(ex.getMessage());
        }
    }

    /*
     * (non-Javadoc)
     * @see
     * org.ow2.jasmine.vmm.api.VirtualMachineMXBean#getUserData(java.lang.String
     * )
     */
    public String getUserData(final String key) throws VMMException {
        // TODO this method has a major drawback: it fails
        // in case of live migration
        try {
            RemoteExec.Result result = null;
            String command = "xenstore-read /domain/" + this.uuid + "/" + key;
            result = RemoteExec.commandAsRoot(this.host.getHostName(), this.host.getSshAuthInfo(), command);
            if (result.exitCode != 0) {
                throw new VMMException("Cannot get user data :" + result.error);
            }
            return result.output;
        } catch (Exception ex) {
            throw new VMMException(ex.getMessage());
        }
    }

    public String getConsole() throws VMMException {
        try {
            String xmlDesc = this.domain.getXMLDesc(0);
            String vncPortString = (String) LibvirtConfigFileUtils.getConfigItem(xmlDesc,
                "/domain/devices/graphics[@type='vnc']/@port", XPathConstants.STRING);
            if (vncPortString == null || vncPortString.equals("")) {
                return null;
            }
            String console = "vnc://" + this.host.getHostName() + ":" + Integer.parseInt(vncPortString);
            LibvirtVirtualMachine.logger.debug("VM " + this.name + " console=" + console);
            return console;
        } catch (LibvirtException ex) {
            throw LibvirtVirtualMachine.translateLibvirtException(ex);
        }
    }

    /*
     * (non-Javadoc)
     * @see org.ow2.jasmine.vmm.api.VirtualMachineMXBean#getNumVCPUs()
     */
    public int getNumVCPUs() throws VMMException {
        DomainInfo info;
        try {
            info = this.domain.getInfo();
        } catch (LibvirtException ex) {
            throw LibvirtVirtualMachine.translateLibvirtException(ex);
        }
        return info.nrVirtCpu;
    }

    @Override
    public boolean[][] getCPUAffinity() throws VMMException {
        try {
            int numVCPUs = this.domain.getInfo().nrVirtCpu;
            int numPCPUs = this.domain.getConnect().nodeInfo().cpus;
            int numMaxPCPUs = this.domain.getConnect().nodeInfo().maxCpus();
            int[] cpuMap = this.domain.getVcpusCpuMaps();
            boolean[][] result = new boolean[numVCPUs][];
            for (int i = 0; i < numVCPUs; i++) {
                result[i] = new boolean[numPCPUs];
            }
            int maxVCPUS = this.domain.getMaxVcpus();
            int cpuMapLength = this.domain.cpuMapLength(numMaxPCPUs);
            for (int i = 0; i < numVCPUs; i++) {
                for (int j = 0; j < numPCPUs; j++) {
                    result[i][j] = (cpuMap[i * cpuMapLength + (j / 8)] & (1 << (j % 8))) != 0;
                }
            }

            return result;
        } catch (LibvirtException ex) {
            throw LibvirtVirtualMachine.translateLibvirtException(ex);
        }
    }

    @Override
    public void setCPUAffinity(final boolean[][] affinity) throws VMMException {
        try {
            if (affinity == null) {
                throw new IllegalArgumentException();
            }
            int mapLength = (this.domain.getConnect().nodeInfo().cpus + 7) / 8;
            for (int vcpuIndex = 0; vcpuIndex < affinity.length; vcpuIndex++) {
                int[] cpuMap = new int[mapLength];
                if (affinity[vcpuIndex] == null) {
                    throw new IllegalArgumentException();
                }
                for (int pcpuIndex = 0; pcpuIndex < affinity[vcpuIndex].length; pcpuIndex++) {
                    if (affinity[vcpuIndex][pcpuIndex]) {
                        cpuMap[pcpuIndex / 8] |= (1 << (pcpuIndex % 8));
                    }
                }
                this.domain.pinVcpu(vcpuIndex, cpuMap);
            }
        } catch (LibvirtException ex) {
            throw LibvirtVirtualMachine.translateLibvirtException(ex);
        }
    }

    /*
     * (non-Javadoc)
     * @see org.ow2.jasmine.vmm.api.VirtualMachineMXBean#getSchedulingCap()
     */
    public int getSchedulingCap() throws VMMException {
        try {
            SchedParameter[] schedParams = this.domain.getSchedulerParameters();
            for (SchedParameter param : schedParams) {
                if (param.field.equals("cap")) {
                    return ((SchedUintParameter) param).value;
                }
            }
            return -1;
        } catch (LibvirtException ex) {
            throw LibvirtVirtualMachine.translateLibvirtException(ex);
        }
    }

    /*
     * (non-Javadoc)
     * @see org.ow2.jasmine.vmm.api.VirtualMachineMXBean#getSchedulingWeight()
     */
    public int getSchedulingWeight() throws VMMException {
        try {
            SchedParameter[] schedParams = this.domain.getSchedulerParameters();
            for (SchedParameter param : schedParams) {
                if (param.field.equals("weight")) {
                    return ((SchedUintParameter) param).value;
                }
            }
            return -1;
        } catch (LibvirtException ex) {
            throw LibvirtVirtualMachine.translateLibvirtException(ex);
        }
    }

    /*
     * (non-Javadoc)
     * @see org.ow2.jasmine.vmm.api.VirtualMachineMXBean#getState()
     */
    public PowerState getState() throws VMMException {
        DomainState state;
        try {
            state = this.domain.getInfo().state;
        } catch (LibvirtException ex) {
            throw LibvirtVirtualMachine.translateLibvirtException(ex);
        }
        PowerState newState = PowerState.UNKNOWN;
        switch (state) {
        case VIR_DOMAIN_BLOCKED:
        case VIR_DOMAIN_RUNNING:
        case VIR_DOMAIN_SHUTDOWN:
            newState = PowerState.RUNNING;
            break;
        case VIR_DOMAIN_SHUTOFF:
            newState = PowerState.HALTED;
            break;
        case VIR_DOMAIN_PAUSED:
            newState = PowerState.PAUSED;
            break;
        case VIR_DOMAIN_CRASHED:
        case VIR_DOMAIN_NOSTATE:
            newState = PowerState.UNKNOWN;
            break;
        }
        if (newState != this.cachedPowerState) {
            this.emitNotification(NotificationTypes.VM_STATE_CHANGE, newState.toString(), null);
            LibvirtVirtualMachine.logger.debug("VM " + this.getNameLabel() + " new power state: " + newState.toString());
            this.cachedPowerState = newState;
        }
        return this.cachedPowerState;
    }

    /*
     * (non-Javadoc)
     * @see org.ow2.jasmine.vmm.api.VirtualMachineMXBean#getGuestIpAddress()
     */
    public String getGuestIpAddress() throws VMMException {
        try {
            if (this.getState() != PowerState.RUNNING) {
                return "unknown";
            }
        } catch (Exception ex) {
        }
        if (this.ipAddress == null) {
            // wait until the guest OS has booted to acquire its IP address
            int nbAttempts = 10;
            while (nbAttempts-- > 0) {
                this.ipAddress = this.host.getIPAddress(this.getMacAddress());
                if (this.ipAddress != null && !this.ipAddress.equals("")) {
                    break;
                }
                try {
                    TimeUnit.SECONDS.sleep(10);
                } catch (InterruptedException ex) {
                }
            }
        }
        return this.ipAddress == null ? "unknown" : this.ipAddress;
    }

    /*
     * (non-Javadoc)
     * @see org.ow2.jasmine.vmm.api.VirtualMachineMXBean#getMacAddress()
     */
    public String getMacAddress() throws VMMException {
        if (this.macAddress == null) {
            String xmlDesc;
            try {
                xmlDesc = this.domain.getXMLDesc(0);
            } catch (LibvirtException ex) {
                throw LibvirtVirtualMachine.translateLibvirtException(ex);
            }
            int i = xmlDesc.indexOf("mac address");
            if (i == -1) {
                return null;
            }
            int startIndex = i + "mac address".length() + 2;
            this.macAddress = xmlDesc.substring(startIndex, startIndex + 17);
        }
        return this.macAddress;
    }

    public void setNumVCPUs(final int numVCPUs) throws VMMException {
        try {
            this.domain.setVcpus(numVCPUs);
        } catch (LibvirtException ex) {
            throw LibvirtVirtualMachine.translateLibvirtException(ex);
        }
    }

    /*
     * (non-Javadoc)
     * @see org.ow2.jasmine.vmm.api.VirtualMachineMXBean#setSchedulingCap(int)
     */
    public void setSchedulingCap(final int schedulingCap) throws VMMException {
        SchedUintParameter cap = new SchedUintParameter(schedulingCap);
        cap.field = "cap";
        try {
            this.domain.setSchedulerParameters(new SchedParameter[] {cap});
        } catch (LibvirtException ex) {
            throw LibvirtVirtualMachine.translateLibvirtException(ex);
        }
    }

    /*
     * (non-Javadoc)
     * @see
     * org.ow2.jasmine.vmm.api.VirtualMachineMXBean#setSchedulingWeight(int)
     */
    public void setSchedulingWeight(final int schedulingWeight) throws VMMException {
        SchedUintParameter cap = new SchedUintParameter(schedulingWeight);
        cap.field = "weight";
        try {
            this.domain.setSchedulerParameters(new SchedParameter[] {cap});
        } catch (LibvirtException ex) {
            throw LibvirtVirtualMachine.translateLibvirtException(ex);
        }
    }

    /*
     * (non-Javadoc)
     * @see
     * org.ow2.jasmine.vmm.api.VirtualMachineMXBean#migrate(org.ow2.jasmine.
     * vmm.api.HostMXBean, boolean)
     */
    public void migrate(HostMXBean targetHost, final boolean live) throws IllegalOperationException, VMMException {
        this.setMigrating(true);
        try {
            String targetHostName = targetHost.getHostName();
            LibvirtVirtualMachine.logger.info("Attempting live migration of VM " + this.name + " to host " + targetHostName);

            boolean foundHostInServerPool = false;

            for (HostMXBean h : this.host.getServerPool().getManagedHosts()) {
                if (h.getHostName().equals(targetHost.getHostName())) {
                    foundHostInServerPool = true;
                    targetHost = h;
                    break;
                }
            }

            if (!foundHostInServerPool) {
                throw new IllegalOperationException("Source and target hosts belong to different server pools");
            }

            LibvirtHost targetLibvirtHost = (LibvirtHost) targetHost;

            LibvirtVirtualMachine.logger.info("VM " + this.name + " live migration to host " + targetHostName + "...");
            this.emitNotification(NotificationTypes.VM_MIGRATION_START, targetHostName, null);

            Domain newDomain = null;
            try {
                newDomain = this.domain.migrate(targetLibvirtHost.getLibvirtAPIConnection(), live ? 1 : 0, null, null, 0);
            } catch (LibvirtException ex) {
                LibvirtVirtualMachine.logger.error("Migration failure", ex);
                throw new VMMException("Migration failure: " + ex.getMessage());
            }

            try {
                String xmlDesc = this.domain.getXMLDesc(0);
                this.domain.undefine();
                newDomain.getConnect().domainDefineXML(xmlDesc);
            } catch (LibvirtException ex) {
                LibvirtVirtualMachine.logger.error(ex);
            }

            LibvirtVirtualMachine.logger.info("VM " + this.name + " live migration done");

            this.host.postMigrateVM(this, targetLibvirtHost);
            this.host = targetLibvirtHost;
            this.domain = newDomain;

            this.emitNotification(NotificationTypes.VM_MIGRATION, targetHostName, this.uuid);
        } finally {
            this.setMigrating(false);
        }
    }

    /*
     * (non-Javadoc)
     * @see org.ow2.jasmine.vmm.api.VirtualMachineMXBean#shutdown()
     */
    public void shutdown() throws VMMException {
        try {
            this.domain.shutdown();
        } catch (LibvirtException ex) {
            throw LibvirtVirtualMachine.translateLibvirtException(ex);
        }
    }

    /*
     * (non-Javadoc)
     * @see org.ow2.jasmine.vmm.api.VirtualMachineMXBean#start()
     */
    public void start() throws VMMException {
        try {
            this.domain.create();
            this.host.sync();
        } catch (LibvirtException ex) {
            throw LibvirtVirtualMachine.translateLibvirtException(ex);
        }
    }

    /*
     * (non-Javadoc)
     * @see org.ow2.jasmine.vmm.api.VirtualMachineMXBean#reboot()
     */
    public void reboot() throws VMMException {
        try {
            this.domain.reboot(0);
        } catch (LibvirtException ex) {
            throw LibvirtVirtualMachine.translateLibvirtException(ex);
        }
    }

    /*
     * (non-Javadoc)
     * @see org.ow2.jasmine.vmm.api.VirtualMachineMXBean#suspend()
     */
    public void suspend() throws VMMException {
        try {
            this.domain.save(" /tmp/" + this.getNameLabel() + ".save");
            this.host.sync();
        } catch (LibvirtException ex) {
            throw LibvirtVirtualMachine.translateLibvirtException(ex);
        }
    }

    /*
     * (non-Javadoc)
     * @see org.ow2.jasmine.vmm.api.VirtualMachineMXBean#resume()
     */
    public void resume() throws VMMException {
        try {
            this.domain.getConnect().restore("/tmp/" + this.getNameLabel() + ".save");
            this.host.sync();
        } catch (LibvirtException ex) {
            throw LibvirtVirtualMachine.translateLibvirtException(ex);
        }
    }

    public void pause() throws VMMException {
        try {
            this.domain.suspend();
            this.host.sync();
            // this.onVMStateChanged();
        } catch (LibvirtException ex) {
            throw LibvirtVirtualMachine.translateLibvirtException(ex);
        }
    }

    public void unpause() throws VMMException {
        try {
            this.domain.resume();
            this.host.sync();
            // this.onVMStateChanged();
        } catch (LibvirtException ex) {
            throw LibvirtVirtualMachine.translateLibvirtException(ex);
        }
    }

    /*
     * (non-Javadoc)
     * @see org.ow2.jasmine.vmm.api.VirtualMachineMXBean#destroy()
     */
    public void destroy() throws VMMException {
        List<String> diskFiles;
        try {
            String xmlDesc = this.domain.getXMLDesc(0);
            diskFiles = LibvirtConfigFileUtils.getDiskFiles(xmlDesc);
            try {
                if (this.getState() == PowerState.RUNNING) {
                    this.domain.destroy();
                }
            } catch (Exception ex) {
            }
            this.domain.undefine();
        } catch (LibvirtException ex) {
            throw LibvirtVirtualMachine.translateLibvirtException(ex);
        }
        for (String file : diskFiles) {
            RemoteExec.Result result;
            String command = "rm -f " + file;
            try {
                LibvirtVirtualMachine.logger.debug("Detroying disk: " + command);
                result = RemoteExec.commandAsRoot(this.host.getHostName(), this.host.getSshAuthInfo(), command);
            } catch (RemoteExec.SshException ex) {
                throw new VMMException("VM cloning failed: SSH connection failure", ex);
            }
            if (result.exitCode != 0) {
                throw new VMMException("VM cloning failed: " + result.error);
            }
        }

        this.host.sync();
    }

    @Override
    public MBeanNotificationInfo[] getNotificationInfo() {
        return new MBeanNotificationInfo[] {new MBeanNotificationInfo(new String[] {NotificationTypes.VM_STATE_CHANGE,
            NotificationTypes.VM_MIGRATION_ABORT, NotificationTypes.VM_MIGRATION_START, NotificationTypes.VM_MIGRATION,
            NotificationTypes.LOG, NotificationTypes.ERROR}, Notification.class.getName(), "VM event")};
    }

    @Override
    public VirtualMachineImageMXBean makeTemplate(final String name, final String description,
        final Map<String, String> metadata) throws InsufficientResourcesException, IllegalOperationException,
        BadVMPowerStateException, VMMException {
        return ((ImageCatalog) this.host.getVMImageStore()).newVMImageTemplate(this, name, description, metadata);
    }

    static VMMException translateLibvirtException(final LibvirtException ex) {
        LibvirtVirtualMachine.logger.error("Libvirt exception:" + ex.toString(), ex);
        return new VMMException(ex.toString());
    }
}