package org.ow2.jasmine.vmm.agent.driver.vmware;

import org.apache.log4j.Logger;
import org.ow2.jasmine.vmm.api.VMConfigSpec;
import org.ow2.jasmine.vmm.api.VnicIPSettings;
import org.ow2.jasmine.vmm.api.VnicIPSettings.IpAssignmentMode;

import com.vmware.vim.CustomizationAdapterMapping;
import com.vmware.vim.CustomizationDhcpIpGenerator;
import com.vmware.vim.CustomizationFixedIp;
import com.vmware.vim.CustomizationFixedName;
import com.vmware.vim.CustomizationGlobalIPSettings;
import com.vmware.vim.CustomizationGuiUnattended;
import com.vmware.vim.CustomizationIPSettings;
import com.vmware.vim.CustomizationIdentification;
import com.vmware.vim.CustomizationLicenseDataMode;
import com.vmware.vim.CustomizationLicenseFilePrintData;
import com.vmware.vim.CustomizationLinuxPrep;
import com.vmware.vim.CustomizationPassword;
import com.vmware.vim.CustomizationSpec;
import com.vmware.vim.CustomizationSysprep;
import com.vmware.vim.CustomizationUserData;
import com.vmware.vim.CustomizationWinOptions;

public class VMwareHelper {
    static Logger logger = Logger.getLogger(VMwareHelper.class);

    static private String WINDOWS_SYSPREP_PASSWORD_KEY = "windows.sysprep.password";

    static private String WINDOWS_SYSPREP_LICENSE_TYPE_KEY = "windows.sysprep.license_type";

    static private String WINDOWS_SYSPREP_LICENSE_TYPE_PER_SEAT_VALUE = "per_seat";

    static private String WINDOWS_SYSPREP_LICENSE_TYPE_PER_SERVER_VALUE = "per_server";

    static private String WINDOWS_SYSPREP_WORKGROUP_KEY = "windows.sysprep.workgroup";

    static private String WINDOWS_SYSPREP_AUTO_LOGON_COUNT_KEY = "windows.sysprep.auto_logon_count";

    static private String WINDOWS_SYSPREP_TIMEZONE_KEY = "windows.sysprep.timezone";

    static private String WINDOWS_SYSPREP_USERDATA_FULLNAME_KEY = "windows.sysprep.userdata_fullname";

    static private String WINDOWS_SYSPREP_USERDATA_ORGNAME_KEY = "windows.sysprep.userdata_orgname";

    static private String WINDOWS_SYSPREP_USERDATA_PRODUCTID_KEY = "windows.sysprep.userdata_fullname";

    /**
     * Prepares a CustomizationSpec object to be used when cloning a VM. Most
     * supported customization parameters except network-related ones are used
     * to sysprep Windows OS
     * 
     * @param vmSpec VMConfigSpec object containing customization parameters
     * @param isWindows true if the VM to customize contains a Windows guest OS
     * @return a VMWare API CustomizationSpec object
     */
    public static CustomizationSpec buildCustomizationSpec(final VMConfigSpec vmSpec, final boolean isWindows) {
        VMwareHelper.logger.debug("Preparing customizationSpec for VM " + vmSpec.getName());
        CustomizationSpec custSpec = new CustomizationSpec();

        if (isWindows) {
            CustomizationSysprep custSysprep = new CustomizationSysprep();

            String password = vmSpec.getGuestOsCustomizationParams().get(VMwareHelper.WINDOWS_SYSPREP_PASSWORD_KEY);
            String autoLogonCount = vmSpec.getGuestOsCustomizationParams().get(
                VMwareHelper.WINDOWS_SYSPREP_AUTO_LOGON_COUNT_KEY);
            String timeZone = vmSpec.getGuestOsCustomizationParams().get(VMwareHelper.WINDOWS_SYSPREP_TIMEZONE_KEY);
            if (autoLogonCount != null && timeZone != null) {
                VMwareHost.logger.debug("Customizing VM " + vmSpec.getName() + " password=" + password + " autoLogonCount="
                    + autoLogonCount + " timeZone=" + timeZone);
                CustomizationGuiUnattended custGuiUnattended = new CustomizationGuiUnattended();
                if (password != null) {
                    CustomizationPassword custPassword = new CustomizationPassword();
                    custPassword.setPlainText(true);
                    custPassword.setValue(password);
                    custGuiUnattended.setPassword(custPassword);
                }
                custGuiUnattended.setAutoLogon(true);
                custGuiUnattended.setAutoLogonCount(Integer.parseInt(autoLogonCount));
                custGuiUnattended.setTimeZone(Integer.parseInt(timeZone));
                custSysprep.setGuiUnattended(custGuiUnattended);
            }

            String licenseType = vmSpec.getGuestOsCustomizationParams().get(VMwareHelper.WINDOWS_SYSPREP_LICENSE_TYPE_KEY);
            if (licenseType != null) {
                if (licenseType.equalsIgnoreCase(VMwareHelper.WINDOWS_SYSPREP_LICENSE_TYPE_PER_SEAT_VALUE)) {
                    CustomizationLicenseFilePrintData custLPD = new CustomizationLicenseFilePrintData();
                    custLPD.setAutoMode(CustomizationLicenseDataMode.perSeat);
                    custSysprep.setLicenseFilePrintData(custLPD);
                } else if (licenseType.equalsIgnoreCase(VMwareHelper.WINDOWS_SYSPREP_LICENSE_TYPE_PER_SERVER_VALUE)) {
                    CustomizationLicenseFilePrintData custLPD = new CustomizationLicenseFilePrintData();
                    custLPD.setAutoMode(CustomizationLicenseDataMode.perServer);
                    custSysprep.setLicenseFilePrintData(custLPD);
                } else {
                    VMwareHelper.logger.warn("unknown license type: " + licenseType);
                }
            }

            String workgroup = vmSpec.getGuestOsCustomizationParams().get(VMwareHelper.WINDOWS_SYSPREP_WORKGROUP_KEY);
            if (workgroup != null) {
                CustomizationIdentification custId = new CustomizationIdentification();
                custId.setJoinWorkgroup(workgroup);
                custSysprep.setIdentification(custId);
            }

            String fullName = vmSpec.getGuestOsCustomizationParams().get(VMwareHelper.WINDOWS_SYSPREP_USERDATA_FULLNAME_KEY);
            String orgName = vmSpec.getGuestOsCustomizationParams().get(VMwareHelper.WINDOWS_SYSPREP_USERDATA_ORGNAME_KEY);
            String productId = vmSpec.getGuestOsCustomizationParams().get(VMwareHelper.WINDOWS_SYSPREP_USERDATA_PRODUCTID_KEY);

            if (fullName != null && orgName != null && productId != null && vmSpec.getGuestOsHostName() != null) {
                CustomizationUserData custUserData = new CustomizationUserData();
                custUserData.setFullName(fullName);
                custUserData.setOrgName(orgName);
                custUserData.setProductId(productId);
                VMwareHost.logger.debug("Customizing VM " + vmSpec.getName() + " productId=" + productId);
                CustomizationFixedName custFixedName = new CustomizationFixedName();
                VMwareHost.logger.debug("Customizing VM " + vmSpec.getName() + " hostname=" + vmSpec.getGuestOsHostName());
                custFixedName.setName(vmSpec.getGuestOsHostName());
                custUserData.setComputerName(custFixedName);
                custSysprep.setUserData(custUserData);
            }

            custSpec.setIdentity(custSysprep);

            CustomizationWinOptions custWinOptions = new CustomizationWinOptions();
            custWinOptions.setChangeSID(true);
            custSpec.setOptions(custWinOptions);
        } else {
            // Linux guest
            CustomizationLinuxPrep custLinuxPrep = new CustomizationLinuxPrep();
            custLinuxPrep.setDomain(vmSpec.getGuestOsDomain());
            CustomizationFixedName fixedName = new CustomizationFixedName();
            fixedName.setName(vmSpec.getGuestOsHostName());
            custLinuxPrep.setHostName(fixedName);
            custSpec.setIdentity(custLinuxPrep);
        }

        // TODO support for multiple VNICs
        if (vmSpec.getVnicIpSettingsList() != null && !!vmSpec.getVnicIpSettingsList().isEmpty()) {
            CustomizationGlobalIPSettings custGlobalIPSettings = new CustomizationGlobalIPSettings();
            custGlobalIPSettings.setDnsServerList(vmSpec.getGuestOsDnsServerList().toArray(new String[0]));
            custGlobalIPSettings.setDnsSuffixList(new String[] {vmSpec.getGuestOsDomain()});
            custSpec.setGlobalIPSettings(new CustomizationGlobalIPSettings());

            VnicIPSettings vnicIPSettings = vmSpec.getVnicIpSettingsList().get(0);

            CustomizationAdapterMapping custAdapterMapping = new CustomizationAdapterMapping();
            CustomizationIPSettings custIPSettings = new CustomizationIPSettings();
            custIPSettings.setDnsDomain(vmSpec.getGuestOsDomain());
            VMwareHost.logger.debug("Customizing VM " + vmSpec.getName() + " DNS server="
                + vmSpec.getGuestOsDnsServerList().get(0));
            custIPSettings.setDnsServerList(vmSpec.getGuestOsDnsServerList().toArray(new String[0]));

            VMwareHost.logger.debug("Customizing VM " + vmSpec.getName() + " gateway=" + vnicIPSettings.getGateway());
            custIPSettings.setGateway(new String[] {vnicIPSettings.getGateway()});
            custIPSettings.setSubnetMask(vnicIPSettings.getSubnetMask());
            if (vnicIPSettings.getIpAssignmentMode() == IpAssignmentMode.FIXED) {
                CustomizationFixedIp custFixedIp = new CustomizationFixedIp();
                VMwareHost.logger.debug("Customizing VM " + vmSpec.getName() + " VNIC IP address="
                    + vnicIPSettings.getIpAddress());
                custFixedIp.setIpAddress(vnicIPSettings.getIpAddress());
                custIPSettings.setIp(custFixedIp);
            } else if (vnicIPSettings.getIpAssignmentMode() == IpAssignmentMode.DHCP) {
                CustomizationDhcpIpGenerator custDhcp = new CustomizationDhcpIpGenerator();
                custIPSettings.setIp(custDhcp);
            }
            custAdapterMapping.setAdapter(custIPSettings);

            custSpec.setNicSettingMap(new CustomizationAdapterMapping[] {custAdapterMapping});
        }

        return custSpec;
    }
}
