/**
 * JASMINe VMMapi: JASMINe Virtual Machine Management API
 * Copyright (C) 2009 France Telecom R&D
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: VMwareHost.java 6176 2010-03-23 17:08:23Z dangtran $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.vmm.agent.driver.vmware;

import java.math.BigInteger;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.Timer;

import javax.management.MBeanNotificationInfo;
import javax.management.Notification;
import javax.management.ObjectName;

import org.apache.log4j.Logger;
import org.ow2.jasmine.vmm.agent.domain.ManagedResource;
import org.ow2.jasmine.vmm.agent.driver.util.RemoteExec;
import org.ow2.jasmine.vmm.agent.driver.util.ResourceUsageHelper;
import org.ow2.jasmine.vmm.agent.driver.vmware.VMwareServerPool.VMwareVirtualMachineISOTemplate;
import org.ow2.jasmine.vmm.agent.jmx.MBeanObjectNamer;
import org.ow2.jasmine.vmm.agent.main.AgentCommon;
import org.ow2.jasmine.vmm.api.HostMXBean;
import org.ow2.jasmine.vmm.api.InsufficientResourcesException;
import org.ow2.jasmine.vmm.api.InvalidVMConfigException;
import org.ow2.jasmine.vmm.api.NotificationTypes;
import org.ow2.jasmine.vmm.api.ResourceUsage;
import org.ow2.jasmine.vmm.api.ServerPoolMXBean;
import org.ow2.jasmine.vmm.api.VMConfigSpec;
import org.ow2.jasmine.vmm.api.VMMException;
import org.ow2.jasmine.vmm.api.VirtualMachineImageMXBean;
import org.ow2.jasmine.vmm.api.VirtualMachineImageStoreMXBean;
import org.ow2.jasmine.vmm.api.VirtualMachineMXBean;
import org.ow2.jasmine.vmm.api.VnicIPSettings;
import org.ow2.jasmine.vmm.api.VnicIPSettings.IpAssignmentMode;
import org.w3c.dom.Element;

import com.vmware.vim.ArrayOfManagedObjectReference;
import com.vmware.vim.ArrayOfVirtualDevice;
import com.vmware.vim.ConfigTarget;
import com.vmware.vim.CustomizationSpec;
import com.vmware.vim.DynamicProperty;
import com.vmware.vim.HostCpuInfo;
import com.vmware.vim.HostCpuPackage;
import com.vmware.vim.HostHardwareInfo;
import com.vmware.vim.ManagedObjectReference;
import com.vmware.vim.ObjectContent;
import com.vmware.vim.VirtualCdrom;
import com.vmware.vim.VirtualCdromIsoBackingInfo;
import com.vmware.vim.VirtualDevice;
import com.vmware.vim.VirtualDeviceConfigSpec;
import com.vmware.vim.VirtualDeviceConfigSpecFileOperation;
import com.vmware.vim.VirtualDeviceConfigSpecOperation;
import com.vmware.vim.VirtualDisk;
import com.vmware.vim.VirtualDiskFlatVer2BackingInfo;
import com.vmware.vim.VirtualEthernetCard;
import com.vmware.vim.VirtualEthernetCardNetworkBackingInfo;
import com.vmware.vim.VirtualFloppy;
import com.vmware.vim.VirtualFloppyImageBackingInfo;
import com.vmware.vim.VirtualIDEController;
import com.vmware.vim.VirtualLsiLogicController;
import com.vmware.vim.VirtualMachineCloneSpec;
import com.vmware.vim.VirtualMachineConfigOption;
import com.vmware.vim.VirtualMachineConfigSpec;
import com.vmware.vim.VirtualMachineFileInfo;
import com.vmware.vim.VirtualMachineNetworkInfo;
import com.vmware.vim.VirtualMachineRelocateSpec;
import com.vmware.vim.VirtualPCNet32;
import com.vmware.vim.VirtualSCSISharing;

/**
 * VMware driver Host MXBean implementation
 */
class VMwareHost extends ManagedResource implements HostMXBean {
    static Logger logger = Logger.getLogger(VMwareHost.class);

    private VMwareServerPool pool;

    private VMwareServerPool.ConnectionPool connectionPool;

    private String hostName;

    private int numCPU = -1;

    private ManagedObjectReference mor;

    private HashMap<String, VMwareVirtualMachine> vmRefs = new HashMap<String, VMwareVirtualMachine>();

    private VMwarePerfMonitor perfMon = null;

    private HashMap<String, BigInteger> hostMetricValues = null;

    private HashMap<String, HashMap<String, BigInteger>> metricValuesPerVM = null;

    private String[][] counterNames = null;

    private ManagedObjectReference myResourcePool;

    private String defaultNetworkName;

    private VirtualDevice[] hostDefaultDevices;

    private RemoteExec.SshAuthInfo authInfo;

    public VMwareHost(final VMwareServerPool pool, final ManagedObjectReference mor, final ObjectName objectName,
        final String hostName, final RemoteExec.SshAuthInfo authInfo) {
        super(objectName);
        this.pool = pool;
        this.connectionPool = pool.getConnectionPool();
        this.mor = mor;
        this.objectName = objectName;
        this.hostName = hostName;
        this.authInfo = authInfo;
        this.counterNames = new String[][] {new String[] {"cpu", "usage", "average"}, // host
            // & vm
            new String[] {"mem", "usage", "average"}, // host & vm
            new String[] {"mem", "active", "average"}, // host & vm
            new String[] {"mem", "granted", "average"}, // host & vm
            new String[] {"mem", "state", "latest"}, // host only
            new String[] {"disk", "usage", "average"}, // host (& vm
            // ???)
            new String[] {"net", "transmitted", "average"}, // host & vm
            new String[] {"net", "received", "average"}, // host & vm
            new String[] {"sys", "uptime", "latest"} // host & vm
        };
        VMwareServiceConnection connection = this.connectionPool.getConnection();
        try {
            VirtualMachineConfigOption vmConfigOption = null;
            ConfigTarget configTarget = null;
            ManagedObjectReference hostParentCompResRef = (ManagedObjectReference) this.getDynamicProarray(connection, mor,
                "parent")[0].getVal();
            ManagedObjectReference envBrowseMor = (ManagedObjectReference) this.getDynamicProarray(connection,
                hostParentCompResRef, "environmentBrowser")[0].getVal();
            vmConfigOption = connection.getService().queryConfigOption(envBrowseMor, null, mor);
            configTarget = connection.getService().queryConfigTarget(envBrowseMor, mor);
            this.defaultNetworkName = pool.getNetworkName();
            if (this.defaultNetworkName == null && configTarget.getNetwork() != null) {
                for (VirtualMachineNetworkInfo netInfo : configTarget.getNetwork()) {
                    if (netInfo.getNetwork().isAccessible()) {
                        this.defaultNetworkName = netInfo.getNetwork().getName();
                        break;
                    }
                }
            }
            VMwareHost.logger.debug("Default network for host " + hostName + ": " + this.defaultNetworkName);
            if (vmConfigOption != null) {
                this.hostDefaultDevices = vmConfigOption.getDefaultDevice();
            }
        } catch (Exception ex) {
            VMwareHost.logger.error("Failed to retrieve host " + hostName + " network name and device set", ex);
        } finally {
            connection.release();
        }
    }

    public boolean isConnectionLost() {
        // XXX
        // heavyweight test
        VMwareServiceConnection connection = this.connectionPool.getConnection();
        try {
            return !connection.isAlive();
        } finally {
            connection.release();
        }

    }

    public ManagedObjectReference getMOR() {
        return this.mor;
    }

    public ServerPoolMXBean getServerPool() {
        return this.pool;
    }

    @Override
    public ObjectName getObjectName() {
        return this.objectName;
    }

    // HostMBean Interface

    public String getHostName() {
        return this.hostName;
    }

    public VMwareServerPool.ConnectionPool getConnectionPool() {
        return this.pool.getConnectionPool();
    }

    HashMap<String, String> hypervisorInfo = null;

    public Map<String, String> getHypervisorInfo() { // TODO
        if (this.hypervisorInfo == null) {
            this.hypervisorInfo = new HashMap<String, String>();
            this.hypervisorInfo.put("name", "ESX");
            this.hypervisorInfo.put("vendor", "VMware");
            this.hypervisorInfo.put("version", "3.0.2");
        }
        return this.hypervisorInfo;
    }

    public int getNumCPU() {
        if (this.numCPU == -1) {
            this.getCPUInfo();
        }
        return this.numCPU;
    }

    HashMap<String, String> cpuInfo = null;

    public Map<String, String> getCPUInfo() {
        if (this.cpuInfo == null) {
            this.cpuInfo = new HashMap<String, String>();
            VMwareServiceConnection connection = this.connectionPool.getConnection();
            try {
                DynamicProperty[] cpuArry = this.getDynamicProarray(connection, this.mor, "hardware");
                HostHardwareInfo hinfo = (HostHardwareInfo) cpuArry[0].getVal();
                HostCpuInfo tinfo = hinfo.getCpuInfo();
                HostCpuPackage cinfo = hinfo.getCpuPkg(0);

                this.cpuInfo.put("speedMHz", Long.toString(tinfo.getHz() / 1000000));
                this.numCPU = tinfo.getNumCpuCores();
                this.cpuInfo.put("vendor", cinfo.getVendor());
            } catch (Exception ex) {
                VMwareHost.logger.debug("Host " + this.hostName, ex);
            } finally {
                connection.release();
            }
        }
        return this.cpuInfo;
    }

    private VMwareVirtualMachine addVm(final ManagedObjectReference vmRef, final String vmName) {
        VMwareServiceConnection connection = this.connectionPool.getConnection();
        try {
            ObjectName vmMBeanObjectName = MBeanObjectNamer.makeVirtualMachineName(this.pool.getPath() + "/" + vmName,
                VMwareVirtualMachine.getUuid(connection, vmRef));
            VMwareVirtualMachine vm = new VMwareVirtualMachine(vmMBeanObjectName, this.pool, vmRef, this, vmName);
            AgentCommon.getMBeanServer().registerMBean(vm, vmMBeanObjectName);
            this.vmRefs.put(vmName, vm);
            VMwareHost.logger.info("Added VirtualMachine " + vmName + " " + vmMBeanObjectName);
            return vm;
        } catch (Exception ex) {
            VMwareHost.logger.error("addVm", ex);
            return null;
        } finally {
            connection.release();
        }

    }

    public List<VirtualMachineMXBean> getResidentVMs() {
        List<VirtualMachineMXBean> result = new ArrayList<VirtualMachineMXBean>();

        final VMwareServiceConnection connection = this.connectionPool.getConnection();
        try {
            ArrayOfManagedObjectReference vms = (ArrayOfManagedObjectReference) this.getDynamicProarray(connection, this.mor,
                "vm")[0].getVal();
            if (vms != null && vms.getManagedObjectReference() != null) {
                for (ManagedObjectReference vmMor : vms.getManagedObjectReference()) {
                    boolean isTemplate = (Boolean) this.getDynamicProarray(connection, vmMor, "summary.config.template")[0]
                        .getVal();
                    if (isTemplate) {
                        continue;
                    }
                    DynamicProperty[] properties = this.getDynamicProarray(connection, vmMor, "config.name");
                    if (properties == null || properties.length < 1) {
                        continue;
                    }
                    String vmName = (String) properties[0].getVal();
                    VMwareVirtualMachine vm = this.vmRefs.get(vmName);
                    if (vm == null) {
                        vm = this.addVm(vmMor, vmName);
                    }

                    result.add(vm);
                }
            }
        } catch (Exception ex) {
            VMwareHost.logger.error("Failed to retrieve resident VM of host " + this.hostName, ex);
        } finally {
            connection.release();
        }

        return result;
    }

    synchronized void onPowerStateChangedVM(final ManagedObjectReference vmRef, final String vmName,
        final VirtualMachineMXBean.PowerState state, final long time) {
        VMwareVirtualMachine vm = this.vmRefs.get(vmName);
        if (vm != null) {
            vm.onPowerStateChangedVM(state, time);
        }
    }

    public long getTotalMemoryMB() {
        long total = 0;
        final VMwareServiceConnection connection = this.connectionPool.getConnection();
        try {
            DynamicProperty[] cpuArry = this.getDynamicProarray(connection, this.mor, "hardware");
            total = ((HostHardwareInfo) cpuArry[0].getVal()).getMemorySize();
        } catch (Exception ex) {
            VMwareHost.logger.debug("Host " + this.hostName, ex);
        } finally {
            connection.release();
        }
        return total / (1024 * 1024);
    }

    public long getFreeMemoryMB() {
        if (this.hostMetricValues != null) {
            return this.hostMetricValues.get("mem.granted").longValue();
        } else {
            final VMwareServiceConnection connection = this.connectionPool.getConnection();
            try {
                int overallMemoryUsage = (Integer) this.getDynamicProarray(connection, this.mor,
                    "summary.quickStats.overallMemoryUsage")[0].getVal();
                return this.getTotalMemoryMB() - overallMemoryUsage;
            } catch (Exception ex) {
                VMwareHost.logger.error("Cannot retrieve attribute summary.quickStats.overallMemoryUsage", ex);
                return -1;
            } finally {
                connection.release();
            }
        }
    }

    public float[] getLoadPerCPU() {
        VMwareHost.logger.info("[VMWareHost.getLoadPerCPU] : Not Implemented");
        return null;
    }

    public float getCPULoad() { // [0,1] interval ?
        if (this.hostMetricValues != null) {
            return (this.hostMetricValues.get("cpu.usage").floatValue() / 10000);
        }
        return 0;
    }

    public VirtualMachineImageStoreMXBean getVMImageStore() {
        return this.pool.getVMImageStore();
    }

    private Timer perfTimer = null;

    private Set<PerfMetric> currentMonitoredMetrics;

    private long currentMonitoringPeriod;

    public void configurePerfMonitor(final Set<PerfMetric> metricsOfInterest, long periodMillis) {
        this.stopPerfMonitor();
        if (periodMillis < 20000) {
            periodMillis = 20000;
        }
        if (metricsOfInterest.contains(PerfMetric.VM_CPU_LOAD)) {
            this.currentMonitoringPeriod = periodMillis;
            this.currentMonitoredMetrics = metricsOfInterest;
            this.startPerfMonitor();
        }
    }

    public Map<String, ResourceUsage> getVMResourceUsage() {
        Map<String, ResourceUsage> result = new HashMap<String, ResourceUsage>();
        for (VMwareVirtualMachine vm : this.vmRefs.values()) {
            result.put(vm.getNameLabel(), vm.getResourceUsage());
        }
        return result;
    }

    public synchronized void startPerfMonitor() {
        if (this.perfTimer == null) {
            this.perfTimer = new Timer(true);
            this.perfMon = new VMwarePerfMonitor(this.connectionPool, this);
            this.perfMon.configure(this.counterNames);
            this.perfTimer.schedule(this.perfMon, 0, 20000);
        }

    }

    public synchronized void stopPerfMonitor() {
        if (this.perfTimer != null) {
            this.perfTimer.cancel();
            this.hostMetricValues = null;
            this.metricValuesPerVM = null;
            for (VMwareVirtualMachine vm : this.vmRefs.values()) {
                vm.updateMetrics(null);
            }
            this.perfTimer = null;
        }
    }

    public void updateMetrics(final HashMap<String, BigInteger> hM, final HashMap<String, HashMap<String, BigInteger>> vmM) {
        this.hostMetricValues = hM;
        this.metricValuesPerVM = vmM;

        HashMap<String, Object> vmUsage = new HashMap<String, Object>();

        for (VMwareVirtualMachine vm : this.vmRefs.values()) {
            String vmName = vm.getNameLabel();
            if (vmM.containsKey(vmName)) {
                vm.updateMetrics(vmM.get(vmName));
                vmUsage.put(vmName, ResourceUsageHelper.serialize(vm.getResourceUsage()));
            }
        }
        this.emitNotification(NotificationTypes.PERF_REPORT, "Resource Usage", vmUsage);
    }

    // TODO revisit this function when VCPU >1
    public Map<String, Float> getVMCPULoads() {
        HashMap<String, Float> cpuLoads = new HashMap<String, Float>();

        long hostMhz = Long.parseLong(this.getCPUInfo().get("speed"));
        final VMwareServiceConnection connection = this.connectionPool.getConnection();
        try {
            for (VMwareVirtualMachine vm : this.vmRefs.values()) {
                if (vm.getHostMBean() == this && vm.getState() == VirtualMachineMXBean.PowerState.RUNNING
                    && !(Boolean) this.getDynamicProarray(connection, vm.getRef(), "summary.config.template")[0].getVal()) {

                    int loadMhz = (Integer) this.getDynamicProarray(connection, vm.getRef(),
                        "summary.quickStats.overallCpuUsage")[0].getVal();
                    float loadPercentage = (float) loadMhz / hostMhz;
                    if (loadPercentage > 1) {
                        loadPercentage = 1;
                    }
                    cpuLoads.put(vm.getNameLabel(), loadPercentage);
                }
            }

        } catch (Exception ex) {
            VMwareHost.logger.error("Failed to retrieve CPU loads", ex);
        } finally {
            connection.release();
        }
        return cpuLoads;
    }

    public ManagedObjectReference getVCResourcePool(final VMwareServiceConnection connection) {
        if (this.myResourcePool == null) {
            DynamicProperty[] dProps;
            try {
                dProps = this.getDynamicProarray(connection, this.mor, "parent");
                ManagedObjectReference computeResource = (ManagedObjectReference) dProps[0].getVal();
                dProps = this.getDynamicProarray(connection, computeResource, "resourcePool");
                this.myResourcePool = (ManagedObjectReference) dProps[0].getVal();
                VMwareHost.logger.debug("Found resourcePool of host " + this.hostName);
            } catch (Exception e) {
                VMwareHost.logger.error("Failed to get Resource Pool of host " + this.hostName, e);
            }
        }
        return this.myResourcePool;
    }

    private ManagedObjectReference findVMTemplateMOR(final VMwareServiceConnection connection, final String name) {
        try {
            ArrayList<ManagedObjectReference> vms = connection.getDecendentMoRefs(((VMwareServerPool) this.getServerPool())
                .getVmTemplateFolderRef(), "VirtualMachine");

            for (ManagedObjectReference vmMor : vms) {
                DynamicProperty[] properties = this.getDynamicProarray(connection, vmMor, "config.name");
                if (properties == null || properties.length < 1) {
                    continue;
                }
                String vmName = (String) properties[0].getVal();
                VMwareHost.logger.debug("vmName=" + vmName);
                if (vmName.equals(name)) {
                    return vmMor;
                }
            }
        } catch (Exception ex) {
            VMwareHost.logger.error(ex);
        }
        return null;
    }

    private VirtualMachineMXBean createVMFromISO(final VMConfigSpec vmSpec, final VMwareVirtualMachineISOTemplate image,
        final boolean sync) throws InsufficientResourcesException, InvalidVMConfigException, VMMException {
        VMwareHost.logger.debug("createVMFromISO: name=" + vmSpec.getName());
        final VMwareServiceConnection connection = this.connectionPool.getConnection();
        if (connection == null) {
            VMwareHost.logger.debug("Out of connection");
            throw new VMMException("Too many VM creation requests, try again later");
        }

        String dataStoreVolume = "[" + this.pool.getDatastoreName() + "]";
        String floppyImagePathName = null;
        String uuid;
        ManagedObjectReference vmMor = null;

        StringBuffer command = new StringBuffer(image.getMakeFloppyShellScript());
        command.append(" -diskMB " + vmSpec.getDiskSizeMB());
        if (vmSpec.getGuestOsCustomizationParams() != null) {
            String sshPublicKey = vmSpec.getGuestOsCustomizationParams().get("sshPublicKey");
            if (sshPublicKey != null) {
                command.append(" -key \"" + sshPublicKey + "\"");
            }
        }
        VnicIPSettings guestIPSettings = vmSpec.getVnicIpSettingsList().get(0);
        if (guestIPSettings != null) {
            if (guestIPSettings.getIpAssignmentMode() == IpAssignmentMode.FIXED) {
                command.append(" -ip " + guestIPSettings.getIpAddress());
                command.append(" -domain " + vmSpec.getGuestOsDomain());
                command.append(" -gateway " + guestIPSettings.getGateway());
                command.append(" -hostname " + vmSpec.getGuestOsHostName());
                command.append(" -subnetmask " + guestIPSettings.getSubnetMask());
                if (vmSpec.getGuestOsDnsServerList() != null) {
                    for (String dnsServer : vmSpec.getGuestOsDnsServerList()) {
                        command.append(" -dnsserver " + dnsServer);
                    }
                }
            } else if (guestIPSettings.getIpAssignmentMode() == IpAssignmentMode.DHCP) {
                command.append(" -dhcp");
            }
        }

        VMwareHost.logger.debug("Launching makeKickStartFloppy script:\n" + command);

        try {
            RemoteExec.Result result = RemoteExec.commandAsRoot(this.getHostName(), this.authInfo, command.toString());
            if (result.exitCode != 0) {
                VMwareHost.logger.error("Failed to create kickstart floppy " + result.error);
                throw new VMMException("Kickstart floppy creation failure " + result.error);
            }
            floppyImagePathName = result.output;
            VMwareHost.logger.debug("Kickstart floppy image file: " + floppyImagePathName);
        } catch (Exception ex) {
            VMwareHost.logger.error("SSH connection failure", ex);
            ;
            throw new VMMException("SS connection failure", ex);
        }

        // the network name (portgroup in VMware parlance) to use
        String networkName = vmSpec.getGuestOsCustomizationParams().get("networkName");
        if (networkName == null) {
            networkName = this.defaultNetworkName;
        }

        synchronized (connection) {
            try {
                VirtualMachineConfigSpec vmConfigSpec = new VirtualMachineConfigSpec();
                VirtualMachineFileInfo fileInfo = new VirtualMachineFileInfo();
                fileInfo.setVmPathName(dataStoreVolume);

                VirtualDevice ideCtlr = null;

                for (VirtualDevice vd : this.hostDefaultDevices) {
                    if (vd instanceof VirtualIDEController) {
                        ideCtlr = vd;
                        break;
                    }
                }
                if (ideCtlr == null) {
                    VMwareHost.logger.error("Cannot find IDE controller");
                    throw new VMMException("Unable to retrieve IDE controller");
                }

                int diskCtlrKey = 1;

                VirtualDeviceConfigSpec scsiDeviceSpec = new VirtualDeviceConfigSpec();
                scsiDeviceSpec.setOperation(VirtualDeviceConfigSpecOperation.add);
                VirtualLsiLogicController scsiCtrl = new VirtualLsiLogicController();
                scsiCtrl.setKey(diskCtlrKey);
                scsiCtrl.setBusNumber(0);
                scsiCtrl.setSharedBus(VirtualSCSISharing.noSharing);
                scsiDeviceSpec.setDevice(scsiCtrl);

                VirtualDeviceConfigSpec diskSpec = new VirtualDeviceConfigSpec();
                diskSpec.setFileOperation(VirtualDeviceConfigSpecFileOperation.create);
                diskSpec.setOperation(VirtualDeviceConfigSpecOperation.add);
                VirtualDisk disk = new VirtualDisk();
                VirtualDiskFlatVer2BackingInfo diskfileBacking = new VirtualDiskFlatVer2BackingInfo();
                diskfileBacking.setFileName(dataStoreVolume);
                diskfileBacking.setDiskMode("persistent");
                disk.setKey(0);
                disk.setControllerKey(diskCtlrKey);
                disk.setUnitNumber(0);
                disk.setBacking(diskfileBacking);
                disk.setCapacityInKB(vmSpec.getDiskSizeMB() * 1024);
                diskSpec.setDevice(disk);

                VirtualDeviceConfigSpec cdromDeviceSpec = new VirtualDeviceConfigSpec();
                cdromDeviceSpec.setOperation(VirtualDeviceConfigSpecOperation.add);
                VirtualCdrom virtualCdrom = new VirtualCdrom();
                VirtualCdromIsoBackingInfo cdromBacking = new VirtualCdromIsoBackingInfo();
                cdromBacking.setDatastore(this.pool.getDatastoreRef());
                cdromBacking.setFileName(dataStoreVolume + image.getIsoFileName());
                virtualCdrom.setBacking(cdromBacking);
                virtualCdrom.setUnitNumber(0);
                virtualCdrom.setControllerKey(ideCtlr.getKey());
                cdromDeviceSpec.setDevice(virtualCdrom);

                VirtualDeviceConfigSpec floppySpec = new VirtualDeviceConfigSpec();
                floppySpec.setOperation(VirtualDeviceConfigSpecOperation.add);
                VirtualFloppy floppy = new VirtualFloppy();
                VirtualFloppyImageBackingInfo flpBacking = new VirtualFloppyImageBackingInfo();
                flpBacking.setFileName(dataStoreVolume + floppyImagePathName);
                floppy.setBacking(flpBacking);
                floppy.setKey(3);
                floppySpec.setDevice(floppy);

                VirtualDeviceConfigSpec nicSpec = new VirtualDeviceConfigSpec();
                nicSpec.setOperation(VirtualDeviceConfigSpecOperation.add);
                VirtualEthernetCard nic = new VirtualPCNet32();
                VirtualEthernetCardNetworkBackingInfo nicBacking = new VirtualEthernetCardNetworkBackingInfo();
                VMwareHost.logger.debug("Setting VM network to " + networkName);
                nicBacking.setDeviceName(networkName);
                nic.setBacking(nicBacking);
                nic.setAddressType("generated");
                nic.setKey(4);
                nicSpec.setDevice(nic);

                VirtualDeviceConfigSpec deviceConfigSpecs[] = new VirtualDeviceConfigSpec[5];
                deviceConfigSpecs[0] = scsiDeviceSpec;
                deviceConfigSpecs[1] = floppySpec;
                deviceConfigSpecs[2] = diskSpec;
                deviceConfigSpecs[3] = cdromDeviceSpec;
                deviceConfigSpecs[4] = nicSpec;

                vmConfigSpec.setDeviceChange(deviceConfigSpecs);
                vmConfigSpec.setMemoryMB(vmSpec.getMemorySizeMB());
                vmConfigSpec.setName(vmSpec.getName());
                vmConfigSpec.setNumCPUs(vmSpec.getNumVCPU());
                vmConfigSpec.setFiles(fileInfo);
                vmConfigSpec.setGuestId(image.getGuestId());

                VMwareHost.logger.debug("Creating VM...");
                String status = null;
                ManagedObjectReference task = null;
                try {
                    task = connection.getService().createVM_Task(this.pool.getVmFolderRef(), vmConfigSpec,
                        this.getVCResourcePool(connection), this.mor);
                    status = connection.waitForTask(task);
                } catch (Exception ex) {
                    VMwareHost.logger.error("CreateVM failure", ex);
                    throw new VMMException("VM creation failure");
                }
                VMwareHost.logger.debug("Status=" + status);
                if (status.equalsIgnoreCase("failure")) {
                    VMwareHost.logger.error("CreateVM failure");
                    throw new VMMException("VM creation failure");
                }

                try {
                    vmMor = (ManagedObjectReference) this.getDynamicProarray(connection, task, "info.result")[0].getVal();
                    uuid = VMwareVirtualMachine.getUuid(connection, vmMor);
                } catch (Exception ex) {
                    VMwareHost.logger.error("Cannot retrieve VM MOR");
                    // TODO destroy the VM in the unlikely event that it exists
                    throw new VMMException("VM creation failure");
                }

                VMwareHost.logger.debug("Powering on VM...");
                try {
                    task = connection.getService().powerOnVM_Task(vmMor, null);
                    status = connection.waitForTask(task);
                } catch (Exception ex) {
                    VMwareHost.logger.error("PowerOnVM failure");
                    // TODO destroy the VM
                    throw new VMMException("VM creation failure");
                }
                if (status.equalsIgnoreCase("failure")) {
                    VMwareHost.logger.error("PowerOnVM failure");
                    // TODO destroy the VM
                    throw new VMMException("VM creation failure");
                }

                // wait for VM to be ready

                VMwareHost.logger.debug("Waiting 5 minutes to allow the guest OS to kickstart...");
                try {
                    Thread.sleep(5 * 60 * 1000);
                    VMwareHost.logger.debug("Getting IP address...");
                    int tryNumber = 50;
                    String ipAddress = null;
                    while (tryNumber-- > 0) {
                        DynamicProperty[] props = this.getDynamicProarray(connection, vmMor, "guest.ipAddress");
                        if (props != null && props.length > 0) {
                            ipAddress = (String) props[0].getVal();
                            VMwareHost.logger.debug("IP address of VM is " + ipAddress);
                            break;
                        }
                        Thread.sleep(20 * 1000);
                        VMwareHost.logger.debug("Waiting for IP address");
                    }
                } catch (Exception ex) {
                    VMwareHost.logger.error("Failed to retrieve IP address", ex);
                }

                VMwareHost.logger.debug("Powering off VM in order to remove Cdrom & floppy devices...");
                try {
                    task = connection.getService().powerOffVM_Task(vmMor);
                    status = connection.waitForTask(task);
                    if (status.equalsIgnoreCase("failure")) {
                        VMwareHost.logger.error("PowerOffVM failure");
                        // XXXX
                    }
                } catch (Exception ex) {
                    VMwareHost.logger.error("Failed to power off VM", ex);
                }

                // remove the CDROM and floppy devices

                virtualCdrom = null;
                floppy = null;
                try {
                    VirtualDevice[] devices = ((ArrayOfVirtualDevice) this.getDynamicProarray(connection, vmMor,
                        "config.hardware.device")[0].getVal()).getVirtualDevice();

                    for (VirtualDevice device : devices) {
                        if (device instanceof VirtualCdrom) {
                            virtualCdrom = (VirtualCdrom) device;
                            VMwareHost.logger.debug("Found Cdrom device");
                        } else if (device instanceof VirtualFloppy) {
                            floppy = (VirtualFloppy) device;
                            VMwareHost.logger.debug("Found floppy device");
                        }
                    }
                } catch (Exception ex) {
                    VMwareHost.logger.error("Failed to retrieve VM CDROM and floppy devices", ex);
                }

                vmConfigSpec = new VirtualMachineConfigSpec();
                deviceConfigSpecs = new VirtualDeviceConfigSpec[2];
                deviceConfigSpecs[0] = new VirtualDeviceConfigSpec();
                deviceConfigSpecs[0].setOperation(VirtualDeviceConfigSpecOperation.remove);
                deviceConfigSpecs[0].setDevice(virtualCdrom);
                deviceConfigSpecs[1] = new VirtualDeviceConfigSpec();
                deviceConfigSpecs[1].setOperation(VirtualDeviceConfigSpecOperation.remove);
                deviceConfigSpecs[1].setDevice(floppy);
                vmConfigSpec.setDeviceChange(deviceConfigSpecs);

                VMwareHost.logger.debug("Reconfiguring VM...");
                try {
                    task = connection.getService().reconfigVM_Task(vmMor, vmConfigSpec);
                    status = connection.waitForTask(task);
                    if (status.equalsIgnoreCase("failure")) {
                        VMwareHost.logger.error("ReconfigureVM failure");
                    }
                    VMwareHost.logger.debug("Done");
                } catch (Exception ex) {
                    VMwareHost.logger.error("Failed to reconfigure VM", ex);
                }

                VMwareHost.logger.debug("VM ready");
            } finally {
                connection.release();
            }
        }

        ObjectName name = null;
        VMwareVirtualMachine newVM;
        try {
            name = MBeanObjectNamer.makeVirtualMachineName(this.pool.getPath() + "/" + vmSpec.getName(), uuid);
            newVM = new VMwareVirtualMachine(name, this.pool, vmMor, this, vmSpec.getName());
            this.vmRefs.put(vmSpec.getName(), newVM);
            AgentCommon.getMBeanServer().registerMBean(newVM, name);
        } catch (Exception ex) {
            VMwareHost.logger.error("Failed to register new VM MBean " + name, ex);
            return null;
        }

        this.emitNotification(NotificationTypes.VM_ADD, "Created", name);

        return newVM;
    }

    public VirtualMachineMXBean createVM(final VMConfigSpec vmSpec, final boolean sync) throws InsufficientResourcesException,
        InvalidVMConfigException, VMMException {
        VMwareHost.logger.info("Creating new VM [name=" + vmSpec.getName() + ",memorySizeMB=" + vmSpec.getMemorySizeMB()
            + ",diskSize=" + vmSpec.getDiskSizeMB() + ",numVCPU=" + vmSpec.getNumVCPU() + ", ID=" + vmSpec.getVmImageUUID()
            + "]");
        VirtualMachineImageMXBean image = null;
        if ((image = this.pool.getVMImageStore().lookUpByUUID(vmSpec.getVmImageUUID())) == null) {
            VMwareHost.logger.debug("Invalid VM Image UUID " + vmSpec.getVmImageUUID());
            throw new InvalidVMConfigException("Invalid VM Image UUID");
        }

        // the image is either a VMwareTemplate or an ISO
        // switch to a different VM creation method accordingly

        if (image instanceof VMwareVirtualMachineISOTemplate) {
            return this.createVMFromISO(vmSpec, (VMwareVirtualMachineISOTemplate) image, sync);
        }

        final VMwareServiceConnection connection = this.connectionPool.getConnection();
        if (connection == null) {
            VMwareHost.logger.debug("Out of connection");
            throw new VMMException("Too many VM creation requests, try again later");
        }

        ManagedObjectReference templateVMRef = this.findVMTemplateMOR(connection, vmSpec.getVmImageUUID());
        if (templateVMRef == null) {
            VMwareHost.logger.debug("Template " + vmSpec.getVmImageUUID() + " not found");
            return null;
        }

        String networkName = vmSpec.getGuestOsCustomizationParams().get("networkName");
        if (networkName == null) {
            networkName = this.defaultNetworkName;
        }

        try {
            VirtualMachineCloneSpec cloneSpec = new VirtualMachineCloneSpec();
            VirtualMachineRelocateSpec relocSpec = new VirtualMachineRelocateSpec();

            relocSpec.setPool(this.getVCResourcePool(connection));
            relocSpec.setDatastore(this.pool.getDatastoreRef());

            cloneSpec.setLocation(relocSpec);
            cloneSpec.setPowerOn(false);
            cloneSpec.setTemplate(false);
            String clonedName = vmSpec.getName();

            if (vmSpec.getVnicIpSettingsList() != null && !!vmSpec.getVnicIpSettingsList().isEmpty()) {
                String guestId = (String) this.getDynamicProarray(connection, templateVMRef, "config.guestId")[0].getVal();
                CustomizationSpec custSpec = VMwareHelper.buildCustomizationSpec(vmSpec, guestId.startsWith("win"));
                cloneSpec.setCustomization(custSpec);
            }

            // find ethernet device

            VirtualEthernetCard vnicDevice = null;
            ArrayOfVirtualDevice devices = (ArrayOfVirtualDevice) this.getDynamicProarray(connection, templateVMRef,
                "config.hardware.device")[0].getVal();
            for (VirtualDevice device : devices.getVirtualDevice()) {
                if (device instanceof VirtualEthernetCard) {
                    vnicDevice = (VirtualEthernetCard) device;
                    break;
                }
            }

            if (vnicDevice != null) {
                VirtualMachineConfigSpec vmConfigSpec = new VirtualMachineConfigSpec();
                VirtualDeviceConfigSpec deviceConfigSpec = new VirtualDeviceConfigSpec();
                VirtualEthernetCardNetworkBackingInfo nicBacking = new VirtualEthernetCardNetworkBackingInfo();
                VMwareHost.logger.debug("Setting VM network to " + networkName);
                nicBacking.setDeviceName(networkName);
                vnicDevice.setBacking(nicBacking);
                deviceConfigSpec.setDevice(vnicDevice);
                deviceConfigSpec.setOperation(VirtualDeviceConfigSpecOperation.edit);

                vmConfigSpec.setDeviceChange(new VirtualDeviceConfigSpec[] {deviceConfigSpec});
                cloneSpec.setConfig(vmConfigSpec);
            } else {
                VMwareHost.logger.error("Cannot find Ethernet Card device of template VM " + vmSpec.getVmImageUUID());
            }

            synchronized (connection) {
                final ManagedObjectReference cloneTask = connection.getService().cloneVM_Task(templateVMRef,
                    this.pool.getVmFolderRef(), clonedName, cloneSpec);
                if (!sync) {
                    this.pool.getExecutorService().execute(new Runnable() {
                        public void run() {
                            try {
                                VMwareHost.this.createVM2(vmSpec, connection, cloneTask);
                            } catch (Exception ex) {
                            }
                        }
                    });
                    return null;
                } else {
                    return this.createVM2(vmSpec, connection, cloneTask);
                }
            }

        } catch (Exception e) {
            connection.release();
            if (e instanceof org.apache.axis.AxisFault) {
                VMwareHost.logger.debug("AxisFault Exception...");
                org.apache.axis.AxisFault fault = (org.apache.axis.AxisFault) e;
                org.w3c.dom.Element[] errors = fault.getFaultDetails();
                for (Element error : errors) {
                    if (error.toString().indexOf("DuplicateName") != -1) {
                        VMwareHost.logger.debug("Virtual Machine with the same name already exists");
                        throw new InvalidVMConfigException("Virtual Machine with the same name already exists");
                    } else if (error.toString().indexOf("InvalidArgument") != -1) {
                        VMwareHost.logger.debug("Specification is invalid");
                        throw new InvalidVMConfigException();
                    } else if (error.toString().indexOf("InvalidName") != -1) {
                        VMwareHost.logger.debug("Virtual Machine name is empty or too long");
                        throw new InvalidVMConfigException("Virtual Machine name is empty or too long");
                    } else if (error.toString().indexOf("RuntimeFault") != -1) {
                        VMwareHost.logger.debug(error.toString());
                        throw new VMMException();
                    } else {
                        VMwareHost.logger.debug(error.toString());
                        throw new VMMException();
                    }
                }
            } else {
                VMwareHost.logger.error("CloneVMTask Exception...", e);
                throw new VMMException(e);
            }
        }
        return null;
    }

    private VirtualMachineMXBean createVM2(final VMConfigSpec vmSpec, final VMwareServiceConnection connection,
        final ManagedObjectReference cloneTask) throws InsufficientResourcesException, InvalidVMConfigException, VMMException {
        VMwareVirtualMachine newVM;
        ManagedObjectReference vmMor = null;
        ObjectName name = null;
        String uuid = null;

        try {
            VMwareHost.logger.debug("CloneVMTask launched, waiting for result...");
            String status = null;
            synchronized (connection) {
                status = connection.waitForTask(cloneTask);
            }
            VMwareHost.logger.debug("CloneVMTask, status=" + status);

            if (status.equalsIgnoreCase("success")) {
                VMwareHost.logger.debug("Virtual Machine cloned  successfully.");
                vmMor = (ManagedObjectReference) this.getDynamicProarray(connection, cloneTask, "info.result")[0].getVal();
                uuid = VMwareVirtualMachine.getUuid(connection, vmMor);
            } else {
                String msg = "Failure -: Virtual Machine cannot be cloned";
                VMwareHost.logger.error(msg);
                throw new VMMException(msg);
            }
            // before starting the VM set the memory and number of CPU
            // parameters
            this.setResourceCapacity(connection, vmMor, vmSpec);
        } catch (Exception e) {
            throw new VMMException(e);
        } finally {
            connection.release();
        }

        try {
            name = MBeanObjectNamer.makeVirtualMachineName(this.pool.getPath() + "/" + vmSpec.getName(), uuid);
            newVM = new VMwareVirtualMachine(name, this.pool, vmMor, this, vmSpec.getName());
            this.vmRefs.put(vmSpec.getName(), newVM);
            AgentCommon.getMBeanServer().registerMBean(newVM, name);
        } catch (Exception ex) {
            VMwareHost.logger.error("Failed to register new VM MBean " + name, ex);
            return null;
        }

        VMwareHost.logger.debug("[Host " + this.hostName + "] VM started");

        this.emitNotification(NotificationTypes.VM_ADD, "Created", name);

        return newVM;
    }

    void onVMDestroy(final VMwareVirtualMachine vm) {
        this.vmRefs.remove(vm.getNameLabel());
        this.emitNotification(NotificationTypes.VM_DEL, "Destroyed", vm.getObjectName());
    }

    private void setResourceCapacity(final VMwareServiceConnection connection, final ManagedObjectReference vmMor,
        final VMConfigSpec vmSpec) {
        VirtualMachineConfigSpec vmConfigSpec = new VirtualMachineConfigSpec();

        // extend disk if needed
        ArrayOfVirtualDevice devices = null;
        try {
            devices = (ArrayOfVirtualDevice) this.getDynamicProarray(connection, vmMor, "config.hardware.device")[0].getVal();
        } catch (Exception ex) {
            VMwareHost.logger.error("Failed to set disk size", ex);
        }
        if (devices != null) {
            for (VirtualDevice device : devices.getVirtualDevice()) {
                if (device instanceof VirtualDisk) {
                    VirtualDisk disk = (VirtualDisk) device;
                    if (disk.getCapacityInKB() < vmSpec.getDiskSizeMB() * 1024) {
                        disk.setCapacityInKB(vmSpec.getDiskSizeMB() * 1024);

                        VirtualDeviceConfigSpec deviceConfigSpec = new VirtualDeviceConfigSpec();
                        deviceConfigSpec.setDevice(disk);
                        deviceConfigSpec.setOperation(VirtualDeviceConfigSpecOperation.edit);

                        vmConfigSpec.setDeviceChange(new VirtualDeviceConfigSpec[] {deviceConfigSpec});
                    }

                    break;
                }
            }
        }

        try {
            vmConfigSpec.setMemoryMB(vmSpec.getMemorySizeMB());
            vmConfigSpec.setNumCPUs(vmSpec.getNumVCPU());
            synchronized (connection) {
                ManagedObjectReference tmor = connection.getService().reconfigVM_Task(vmMor, vmConfigSpec);

                String status = connection.waitForTask(tmor);

                if (status.equalsIgnoreCase("failure")) {
                    String msg = "Failure -: Virtual Machine cannot be reconfigured";
                    VMwareHost.logger.error(msg);
                } else {
                    VMwareHost.logger.debug("VM " + vmSpec.getName() + ": Memory capacity set to " + vmSpec.getMemorySizeMB()
                        + "MB, number of CPU set to " + vmSpec.getNumVCPU() + " disk size set to " + vmSpec.getDiskSizeMB()
                        + "MB");
                }
            }
        } catch (Exception e) {
            VMwareHost.logger.error("Failed to set VM memory and numCPU", e);
        }
    }

    private DynamicProperty[] getDynamicProarray(final VMwareServiceConnection connection, final ManagedObjectReference MOR,
        final String pName) throws Exception {
        ObjectContent[] objContent = connection.getObjectProperties(null, MOR, new String[] {pName});
        ObjectContent contentObj = objContent[0];
        DynamicProperty[] objArr = contentObj.getPropSet();
        return objArr;
    }

    @Override
    public MBeanNotificationInfo[] getNotificationInfo() {
        return new MBeanNotificationInfo[] {new MBeanNotificationInfo(new String[] {NotificationTypes.VM_STATE_CHANGE,
            NotificationTypes.VM_ADD, NotificationTypes.VM_DEL, NotificationTypes.ERROR, NotificationTypes.LOG},
            Notification.class.getName(), "Host event")};
    }

}
