/**
 * JASMINe VMMapi: JASMINe Virtual Machine Management API
 * Copyright (C) 2009 France Telecom R&D
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: XenServerPool.java 6193 2010-03-25 17:25:02Z dangtran $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.vmm.agent.driver.xenapi;

import java.util.Map;

import javax.management.ObjectName;

import org.ow2.jasmine.vmm.agent.domain.ServerPool;
import org.ow2.jasmine.vmm.agent.jmx.MBeanObjectNamer;
import org.ow2.jasmine.vmm.agent.main.AgentCommon;
import org.ow2.jasmine.vmm.api.HostMXBean;
import org.ow2.jasmine.vmm.api.VirtualMachineImageStoreMXBean;

/**
 * The XenServerPool is the XenAPI-driver-specific implementation of the
 * ServerPoolMXBean interface.
 * <p>
 * The XenServerPool accepts the following properties:
 * <ul>
 * <li>user : XenAPI default login
 * <li>password : XenAPI default password
 * <li>port : XenAPI default port
 * </ul>
 * These properties will be used to connect to each host member of the
 * XenServerPool.
 */
public class XenServerPool extends ServerPool {
    private int port;

    private String user;

    private String password;

    private VirtualMachineImageStoreMXBean sharedImageStore;

    private String sharedStorageRepository;

    public XenServerPool(final String name, final ObjectName objectName, final Map<String, String> attributes)
        throws IllegalArgumentException {
        super(name, objectName);
        String port = attributes.get("port");
        if (port == null) {
            this.port = 9363;
        } else {
            try {
                this.port = Integer.parseInt(port);
            } catch (NumberFormatException ex) {
                throw new IllegalArgumentException("Wrong value for port");
            }
        }
        this.user = attributes.get("user");
        this.password = attributes.get("password");

        this.sharedStorageRepository = attributes.get("sharedStorageRepository");
        if (this.sharedStorageRepository == null) {
            throw new IllegalArgumentException("Missing parameter: sharedStorageRepository");
        }
    }

    public String getSharedStorageRepository() {
        return this.sharedStorageRepository;
    }

    @Override
    public HostMXBean newHost(final String hostName, final Map<String, String> attributes) throws IllegalArgumentException {
        try {
            for (HostMXBean host : this.hosts) {
                if (host.getHostName().equals(hostName)) {
                    return host;
                }
            }
            ObjectName mbeanObjectName = MBeanObjectNamer.makeHostName(this.getPath() + "/" + hostName, hostName);

            XenHost host = XenHost.newHost(this, mbeanObjectName, hostName, this.user, this.password, this.port, attributes);
            if (host != null) {
                AgentCommon.getMBeanServer().registerMBean(host, mbeanObjectName);
                this.hosts.add(host);
                ServerPool.logger.info("In server pool " + this.name + ": added Host MBean " + mbeanObjectName);

                if (this.sharedImageStore == null && this.sharedStorageRepository != null) {
                    ObjectName sharedImageStoreName = MBeanObjectNamer.makeVMImageStoreName(this.sharedStorageRepository);
                    try {
                        this.sharedImageStore = new XenVMImageStore(sharedImageStoreName, this.sharedStorageRepository, host);
                        AgentCommon.getMBeanServer().registerMBean(this.sharedImageStore, sharedImageStoreName);
                    } catch (Exception ex) {
                        ServerPool.logger.error(ex);
                    }
                }
            }
            return host;
        } catch (Exception ex) {
            ServerPool.logger.error("Failed to add XenHost", ex);
            return null;
        }
    }

    @Override
    public VirtualMachineImageStoreMXBean getVMImageStore() {
        return this.sharedImageStore;
    }

}
