/**
 * JASMINe VMMapi: JASMINe Virtual Machine Management API
 * Copyright (C) 2009 France Telecom R&D
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: XenVirtualMachine.java 6098 2010-02-23 13:50:28Z dangtran $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.vmm.agent.driver.xenapi;

import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.GregorianCalendar;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;
import java.util.StringTokenizer;

import javax.management.MBeanNotificationInfo;
import javax.management.Notification;
import javax.management.ObjectName;

import org.apache.log4j.Logger;
import org.ow2.jasmine.vmm.agent.domain.ManagedResource;
import org.ow2.jasmine.vmm.api.BadVMPowerStateException;
import org.ow2.jasmine.vmm.api.HostMXBean;
import org.ow2.jasmine.vmm.api.IllegalOperationException;
import org.ow2.jasmine.vmm.api.InsufficientResourcesException;
import org.ow2.jasmine.vmm.api.NotificationTypes;
import org.ow2.jasmine.vmm.api.ResourceUsage;
import org.ow2.jasmine.vmm.api.VMCustomizationSpec;
import org.ow2.jasmine.vmm.api.VMMException;
import org.ow2.jasmine.vmm.api.VirtualMachineImageMXBean;
import org.ow2.jasmine.vmm.api.VirtualMachineMXBean;
import org.ow2.jasmine.vmm.api.ResourceUsage.DiskStats;
import org.ow2.jasmine.vmm.api.ResourceUsage.NetworkStats;

import com.xensource.xenapi.Connection;
import com.xensource.xenapi.Types;
import com.xensource.xenapi.VBD;
import com.xensource.xenapi.VBDMetrics;
import com.xensource.xenapi.VIF;
import com.xensource.xenapi.VIFMetrics;
import com.xensource.xenapi.VM;
import com.xensource.xenapi.VMGuestMetrics;

/**
 * XenAPI driver Virtual Machine MXBean implementation
 */
public class XenVirtualMachine extends ManagedResource implements VirtualMachineMXBean {
    static Logger logger = Logger.getLogger(XenVirtualMachine.class);

    private XenHost host;

    private Connection connection;

    private VM vm;

    private String uuid;

    private String name;

    private String macAddress;

    private String imageID;

    private Date startTime;

    private PowerState cachedPowerState;

    public XenVirtualMachine(final ObjectName objectName, final XenHost host, final Connection connection, final VM vm,
        final Map<String, String> userData) {
        super(objectName);
        try {
            this.host = host;
            this.connection = connection;
            this.vm = vm;
            synchronized (connection) {
                this.uuid = vm.getUuid(connection);
                if (!vm.getRecord(connection).isControlDomain) {
                    this.name = vm.getNameLabel(connection);
                } else {
                    this.name = "Domain-0";
                }

                if (userData != null) {
                    for (String key : userData.keySet()) {
                        this.addUserData(key, userData.get(key));
                    }
                    this.imageID = userData.get("imageID");
                }
                if (this.imageID == null) {
                    this.imageID = this.getUserData("imageID");
                }

            }
        } catch (Exception ex) {
            XenVirtualMachine.logger.error("Failed to init VM", ex);
        }
    }

    public boolean canLiveMigrateToHost(final HostMXBean targetHost) throws VMMException {
        XenHost xenHost = null;
        for (HostMXBean h : this.host.getServerPool().getManagedHosts()) {
            if (h.getObjectName().equals(targetHost.getObjectName())) {
                xenHost = (XenHost) h;
                break;
            }
        }
        return xenHost != null && xenHost != this.host && this.getMemorySizeMB() < xenHost.getFreeMemoryMB();
    }

    // XXX no support for VM customization
    public VirtualMachineMXBean cloneVM(final String clonedVmName, final VMCustomizationSpec custSpec, final boolean sync)
        throws InsufficientResourcesException, VMMException {
        return this.host.cloneVM(this.vm, clonedVmName, custSpec, sync);
    }

    public String getUuid() {
        return this.uuid;
    }

    public HostMXBean getHostMBean() {
        return this.host;
    }

    public long getDomID() throws VMMException {
        // cannot cache domID because it might change
        // during the VM lifespan (e.g. as a result of migration)
        synchronized (this.connection) {
            try {
                return this.vm.getDomid(this.connection).intValue();
            } catch (Exception ex) {
                throw XenVirtualMachine.translateXenAPIException(ex);
            }
        }
    }

    public Date getStartTime() throws VMMException {
        if (this.startTime == null) {
            synchronized (this.connection) {
                try {
                    this.startTime = this.vm.getMetrics(this.connection).getStartTime(this.connection);
                    Calendar calendar = new GregorianCalendar();
                    this.startTime = new Date(this.startTime.getTime() + calendar.get(Calendar.DST_OFFSET)
                        + calendar.get(Calendar.ZONE_OFFSET));
                } catch (Exception ex) {
                    throw XenVirtualMachine.translateXenAPIException(ex);
                }
            }
        }
        return this.startTime;
    }

    public long getUpTimeSeconds() throws VMMException {
        return (System.currentTimeMillis() - this.getStartTime().getTime()) / 1000;
    }

    public ResourceUsage getResourceUsage() throws VMMException {
        ResourceUsage usage = new ResourceUsage();
        try {
            usage.setCpuLoad(this.getCPULoad());
            usage.setMemoryUsedKBytes(this.getMemoryUsedMB());

            Set<VIF> vifs = this.vm.getVIFs(this.connection);
            ArrayList<NetworkStats> netStats = new ArrayList<NetworkStats>();
            for (VIF vif : vifs) {
                VIFMetrics.Record vifMetrics = vif.getMetrics(this.connection).getRecord(this.connection);
                NetworkStats netStat = new NetworkStats(vif.getDevice(this.connection), vifMetrics.ioReadKbs.longValue(),
                    vifMetrics.ioWriteKbs.longValue());
                netStats.add(netStat);
            }
            usage.setNetworkStats(netStats);

            Set<VBD> vbds = this.vm.getVBDs(this.connection);
            ArrayList<DiskStats> diskStats = new ArrayList<DiskStats>();
            for (VBD vbd : vbds) {
                VBDMetrics.Record vbdMetrics = vbd.getMetrics(this.connection).getRecord(this.connection);
                DiskStats diskStat = new DiskStats(vbd.getDevice(this.connection), vbdMetrics.ioReadKbs.longValue(),
                    vbdMetrics.ioWriteKbs.longValue());
                diskStats.add(diskStat);
            }
            usage.setDiskStats(diskStats);

        } catch (Exception ex) {
            throw XenVirtualMachine.translateXenAPIException(ex);
        }
        return usage;
    }

    public float getCPULoad() throws VMMException {
        float vcpuLoads[] = this.getLoadPerVCPU();
        float cpuLoad = 0;
        for (float vcpuLoad : vcpuLoads) {
            cpuLoad += vcpuLoad;
        }
        return cpuLoad / this.host.getNumCPU();
    }

    public float[] getLoadPerVCPU() throws VMMException {

        synchronized (this.connection) {
            try {
                Map<Long, Double> map = this.vm.getMetrics(this.connection).getVCPUsUtilisation(this.connection);
                float vals[] = new float[map.keySet().size()];
                for (Map.Entry<Long, Double> entry : map.entrySet()) {
                    vals[entry.getKey().intValue()] = entry.getValue().floatValue();
                }
                return vals;
            } catch (Exception ex) {
                throw XenVirtualMachine.translateXenAPIException(ex);
            }
        }
    }

    public long getMemorySizeMB() throws VMMException {
        synchronized (this.connection) {
            try {
                return this.vm.getMemoryDynamicMax(this.connection) / (1024 * 1024);
            } catch (Exception ex) {
                throw XenVirtualMachine.translateXenAPIException(ex);
            }
        }
    }

    public void setMemorySizeMB(final long size) throws VMMException {
        synchronized (this.connection) {
            try {
                this.vm.setMemoryDynamicMax(this.connection, size * 1014 * 1024);
                this.vm.setMemoryDynamicMin(this.connection, size * 1024 * 1024);
            } catch (Exception ex) {
                throw XenVirtualMachine.translateXenAPIException(ex);
            }
        }
    }

    public long getMemoryUsedMB() throws VMMException {
        synchronized (this.connection) {
            try {
                return this.vm.getMetrics(this.connection).getMemoryActual(this.connection).intValue() / (1024 * 1024);
            } catch (Exception ex) {
                throw XenVirtualMachine.translateXenAPIException(ex);
            }
        }
    }

    public String getNameLabel() {
        return this.name;
    }

    public void addUserData(final String key, final String value) throws VMMException {
        synchronized (this.connection) {
            try {
                this.vm.addToXenstoreData(this.connection, key, value);
            } catch (Exception ex) {
                throw XenVirtualMachine.translateXenAPIException(ex);
            }
        }
    }

    /*
     * (non-Javadoc)
     * @see
     * org.ow2.jasmine.vmm.api.VirtualMachineMXBean#getUserData(java.lang.String
     * )
     */
    public String getUserData(final String key) throws VMMException {
        synchronized (this.connection) {
            try {
                return this.vm.getXenstoreData(this.connection).get(key);
            } catch (Exception ex) {
                throw XenVirtualMachine.translateXenAPIException(ex);
            }
        }
    }

    // TODO
    public String getConsole() {
        return null;
    }

    /*
     * (non-Javadoc)
     * @see org.ow2.jasmine.vmm.api.VirtualMachineMXBean#getNumVCPUs()
     */
    public int getNumVCPUs() throws VMMException {
        synchronized (this.connection) {
            try {
                return this.vm.getMetrics(this.connection).getVCPUsNumber(this.connection).intValue();
            } catch (Exception ex) {
                throw XenVirtualMachine.translateXenAPIException(ex);
            }
        }
    }

    @Override
    public boolean[][] getCPUAffinity() throws VMMException {
        synchronized (this.connection) {
            boolean[][] result = new boolean[this.getNumVCPUs()][];
            for (int i = 0; i < result.length; i++) {
                result[i] = new boolean[this.host.getNumCPU()];
            }
            try {
                Map<String, String> map = this.vm.getVCPUsParams(this.connection);
                String mask = map.get("mask");
                // XXX we assume no pinning if mask undefined
                if (mask == null) {
                    for (int i = 0; i < result.length; i++) {
                        for (int j = 0; j < result[i].length; j++) {
                            result[i][j] = true;
                        }
                    }
                } else {
                    StringTokenizer st = new StringTokenizer(mask, ",");
                    ArrayList<Integer> pinCpus = new ArrayList<Integer>();
                    while (st.hasMoreTokens()) {
                        try {
                            pinCpus.add(Integer.parseInt(st.nextToken()));
                        } catch (NumberFormatException ex) {
                        }
                    }
                    for (int i = 0; i < this.getNumVCPUs(); i++) {
                        for (Integer j : pinCpus) {
                            result[i][j] = true;
                        }
                    }
                }
            } catch (Exception ex) {
                throw XenVirtualMachine.translateXenAPIException(ex);
            }
            return result;
        }
    }

    @Override
    public void setCPUAffinity(final boolean[][] affinity) throws VMMException {
        synchronized (this.connection) {
            try {
                Map<String, String> map = this.vm.getVCPUsParams(this.connection);
                // XX only set VCPU #0 affinity
                StringBuffer mask = new StringBuffer();
                int count = 0;
                for (int i = 0; i < affinity[0].length; i++) {
                    if (affinity[0][i]) {
                        if (count++ > 0) {
                            mask.append(',');
                        }
                        mask.append(i);
                    }
                }
                map.put("mask", mask.toString());
                this.vm.setVCPUsParams(this.connection, map);
            } catch (Exception ex) {
                throw XenVirtualMachine.translateXenAPIException(ex);
            }
        }
    }

    /*
     * (non-Javadoc)
     * @see org.ow2.jasmine.vmm.api.VirtualMachineMXBean#getSchedulingCap()
     */
    public int getSchedulingCap() throws VMMException {
        synchronized (this.connection) {
            try {
                Map<String, String> vcpuParams = this.vm.getVCPUsParams(this.connection);
                XenVirtualMachine.logger.debug("params=" + vcpuParams);
                if (vcpuParams.get("cap") != null) {
                    return Integer.parseInt(vcpuParams.get("cap"));
                } else {
                    // XXX XenServer seems to forget to set VCPUs params by
                    // default
                    return 0;
                }
            } catch (Exception ex) {
                throw XenVirtualMachine.translateXenAPIException(ex);
            }
        }
    }

    /*
     * (non-Javadoc)
     * @see org.ow2.jasmine.vmm.api.VirtualMachineMXBean#getSchedulingWeight()
     */
    public int getSchedulingWeight() throws VMMException {
        synchronized (this.connection) {
            try {
                Map<String, String> vcpuParams = this.vm.getVCPUsParams(this.connection);
                if (vcpuParams.get("weight") != null) {
                    return Integer.parseInt(vcpuParams.get("weight"));
                } else {
                    // XXX XenServer seems to forget to set VCPUs params by
                    // default
                    return 256;
                }
            } catch (Exception ex) {
                throw XenVirtualMachine.translateXenAPIException(ex);
            }
        }
    }

    /*
     * (non-Javadoc)
     * @see org.ow2.jasmine.vmm.api.VirtualMachineMXBean#getState()
     */
    public PowerState getState() throws VMMException {
        synchronized (this.connection) {
            try {
                Types.VmPowerState state = this.vm.getPowerState(this.connection);
                this.updatePowerState(state);
            } catch (Exception ex) {
                throw XenVirtualMachine.translateXenAPIException(ex);
            }
        }
        return this.cachedPowerState;
    }

    void updatePowerState(final Types.VmPowerState state) {
        PowerState newState = PowerState.UNKNOWN;
        if (state == Types.VmPowerState.HALTED) {
            newState = PowerState.HALTED;
        } else if (state == Types.VmPowerState.RUNNING) {
            newState = PowerState.RUNNING;
        } else if (state == Types.VmPowerState.PAUSED) {
            newState = PowerState.PAUSED;
        } else if (state == Types.VmPowerState.SUSPENDED) {
            newState = PowerState.SUSPENDED;
        } else if (state == Types.VmPowerState.PAUSED) {
            newState = PowerState.PAUSED;
        }
        if (newState != this.cachedPowerState) {
            this.emitNotification(NotificationTypes.VM_STATE_CHANGE, newState.toString(), null);
            this.cachedPowerState = newState;
        }
    }

    /*
     * (non-Javadoc)
     * @see org.ow2.jasmine.vmm.api.VirtualMachineMXBean#getGuestIpAddress()
     */
    public String getGuestIpAddress() throws VMMException {
        // XX only returns first NIC address
        // XX assume XenServer tools installed in guest OS
        synchronized (this.connection) {
            try {
                VMGuestMetrics guestMetrics = this.vm.getGuestMetrics(this.connection);
                Map<String, String> map = guestMetrics.getOsVersion(this.connection);
                map = guestMetrics.getNetworks(this.connection);
                if (map != null) {
                    return map.get("0/ip");
                }
            } catch (Exception ex) {
                throw XenVirtualMachine.translateXenAPIException(ex);
            }
        }
        return null;
    }

    /*
     * (non-Javadoc)
     * @see org.ow2.jasmine.vmm.api.VirtualMachineMXBean#getMacAddress()
     */
    public String getMacAddress() throws VMMException {
        try {
            if (this.getState() != PowerState.RUNNING) {
                return "";
            }
        } catch (Exception ex) {
        }
        if (this.macAddress == null) {
            synchronized (this.connection) {
                try {
                    Set<VIF> vifs = this.vm.getVIFs(this.connection);
                    for (VIF vif : vifs) {
                        String vifName = vif.getDevice(this.connection);
                        if (vifName.equals("0")) {
                            this.macAddress = vif.getMAC(this.connection);
                            XenVirtualMachine.logger.debug("VM " + this.name + " Mac address=" + this.macAddress);
                            break;
                        }
                    }
                } catch (Exception ex) {
                    XenVirtualMachine.logger.error("Failed to get MAC address of VM " + this.name, ex);
                    throw XenVirtualMachine.translateXenAPIException(ex);
                }
            }
        }
        return this.macAddress;
    }

    public void setNumVCPUs(final int numVCPUs) throws VMMException {
        synchronized (this.connection) {
            try {
                this.vm.setVCPUsNumberLive(this.connection, (long) numVCPUs);
            } catch (Exception ex) {
                throw XenVirtualMachine.translateXenAPIException(ex);
            }
        }
    }

    /*
     * (non-Javadoc)
     * @see org.ow2.jasmine.vmm.api.VirtualMachineMXBean#setSchedulingCap(int)
     */
    public void setSchedulingCap(final int schedulingCap) throws VMMException {
        synchronized (this.connection) {
            HashMap<String, String> map = new HashMap<String, String>();
            map.put("cap", String.valueOf(schedulingCap));
            try {
                this.vm.setVCPUsParams(this.connection, map);
            } catch (Exception ex) {
                throw XenVirtualMachine.translateXenAPIException(ex);
            }
        }
    }

    /*
     * (non-Javadoc)
     * @see
     * org.ow2.jasmine.vmm.api.VirtualMachineMXBean#setSchedulingWeight(int)
     */
    public void setSchedulingWeight(final int schedulingWeight) throws VMMException {
        synchronized (this.connection) {
            HashMap<String, String> map = new HashMap<String, String>();
            map.put("weight", String.valueOf(schedulingWeight));
            try {
                this.vm.setVCPUsParams(this.connection, map);
            } catch (Exception ex) {
                throw XenVirtualMachine.translateXenAPIException(ex);
            }
        }
    }

    /*
     * (non-Javadoc)
     * @see org.ow2.jasmine.vmm.api.VirtualMachineMXBean#suspend()
     */
    public void suspend() throws VMMException {
        synchronized (this.connection) {
            try {
                this.vm.suspend(this.connection);
            } catch (Exception ex) {
                throw XenVirtualMachine.translateXenAPIException(ex);
            }
        }
    }

    /*
     * (non-Javadoc)
     * @see
     * org.ow2.jasmine.vmm.api.VirtualMachineMXBean#migrate(org.ow2.jasmine.
     * vmm.api.HostMXBean, boolean)
     */
    public void migrate(HostMXBean targetHost, final boolean live) throws IllegalOperationException, VMMException {
        // TODO cold migration
        if (!live) {
            throw new IllegalOperationException("Not implemented");
        }
        String targetHostName = targetHost.getHostName();
        XenVirtualMachine.logger.info("Attempting live migration of VM " + this.name + " to host " + targetHostName);

        boolean foundHostInServerPool = false;

        for (HostMXBean h : this.host.getServerPool().getManagedHosts()) {
            if (h.getHostName().equals(targetHost.getHostName())) {
                foundHostInServerPool = true;
                targetHost = h;
                break;
            }
        }

        if (!foundHostInServerPool) {
            throw new IllegalOperationException("Source and target hosts belong to different server pools");
        }

        XenHost targetXenHost = (XenHost) targetHost;
        String imageID = this.getUserData("imageID");

        XenVirtualMachine.logger.info("VM " + this.name + " live migration to host " + targetHostName + "...");
        this.emitNotification(NotificationTypes.VM_MIGRATION_START, targetHostName, null);

        try {
            this.vm.poolMigrate(this.connection, targetXenHost.host, null);
        } catch (Exception ex) {
            throw XenVirtualMachine.translateXenAPIException(ex);
        }

        XenVirtualMachine.logger.info("VM " + this.name + " live migration done");

        this.host.postMigrateVM(this, targetXenHost);
        this.host = targetXenHost;
        this.addUserData("imageID", imageID);
        this.connection = targetXenHost.getXenAPIConnection();
        synchronized (this.connection) {
            int attempts = 3;
            while (true) {
                try {
                    this.vm = VM.getByUuid(this.connection, this.uuid);
                    break;
                } catch (Exception ex) {
                    attempts -= 1;
                    if (attempts > 0) {
                        try {
                            Thread.sleep(1000);
                        } catch (Exception e) {
                        }
                        continue;
                    } else {
                        XenVirtualMachine.logger.error("Cannot get VM XenAPI handle after migration: ", ex);
                        break;
                    }
                }
            }
        }

        this.emitNotification(NotificationTypes.VM_MIGRATION, targetHostName, this.uuid);
    }

    /*
     * (non-Javadoc)
     * @see org.ow2.jasmine.vmm.api.VirtualMachineMXBean#shutdown()
     */
    public void shutdown() throws VMMException {
        synchronized (this.connection) {
            try {
                this.vm.cleanShutdownAsync(this.connection);
                this.getState();
            } catch (Exception ex) {
                throw XenVirtualMachine.translateXenAPIException(ex);
            }
        }
    }

    /*
     * (non-Javadoc)
     * @see org.ow2.jasmine.vmm.api.VirtualMachineMXBean#start()
     */
    public void start() throws VMMException {
        synchronized (this.connection) {
            try {
                this.vm.startAsync(this.connection, false, false);
                this.getState();
            } catch (Exception ex) {
                throw XenVirtualMachine.translateXenAPIException(ex);
            }
        }
    }

    /*
     * (non-Javadoc)
     * @see org.ow2.jasmine.vmm.api.VirtualMachineMXBean#reboot()
     */
    public void reboot() throws VMMException {
        synchronized (this.connection) {
            try {
                this.vm.cleanRebootAsync(this.connection);
            } catch (Exception ex) {
                throw XenVirtualMachine.translateXenAPIException(ex);
            }
        }
    }

    /*
     * (non-Javadoc)
     * @see org.ow2.jasmine.vmm.api.VirtualMachineMXBean#resume()
     */
    public void resume() throws VMMException {
        synchronized (this.connection) {
            try {
                this.vm.resumeAsync(this.connection, false, false);
                this.getState();
            } catch (Exception ex) {
                throw XenVirtualMachine.translateXenAPIException(ex);
            }
        }
    }

    public void pause() throws VMMException {
        synchronized (this.connection) {
            try {
                this.vm.pauseAsync(this.connection);
                this.getState();
            } catch (Exception ex) {
                throw XenVirtualMachine.translateXenAPIException(ex);
            }
        }
    }

    public void unpause() throws VMMException {
        synchronized (this.connection) {
            try {
                this.vm.unpauseAsync(this.connection);
                this.getState();
            } catch (Exception ex) {
                throw XenVirtualMachine.translateXenAPIException(ex);
            }
        }
    }

    /*
     * (non-Javadoc)
     * @see org.ow2.jasmine.vmm.api.VirtualMachineMXBean#destroy()
     */
    public void destroy() throws VMMException {
        synchronized (this.connection) {
            try {
                if (this.getState() == PowerState.RUNNING) {
                    this.vm.hardShutdown(this.connection);
                }
                this.vm.destroy(this.connection);
                XenVirtualMachine.logger.info("VM " + this.name + " destroyed");
            } catch (Exception ex) {
                throw XenVirtualMachine.translateXenAPIException(ex);
            }
        }
    }

    @Override
    public MBeanNotificationInfo[] getNotificationInfo() {
        return new MBeanNotificationInfo[] {new MBeanNotificationInfo(new String[] {NotificationTypes.VM_STATE_CHANGE,
            NotificationTypes.VM_MIGRATION_ABORT, NotificationTypes.VM_MIGRATION_START, NotificationTypes.VM_MIGRATION,
            NotificationTypes.LOG, NotificationTypes.ERROR}, Notification.class.getName(), "VM event")};
    }

    @Override
    public VirtualMachineImageMXBean makeTemplate(final String name, final String description,
        final Map<String, String> metadata) throws InsufficientResourcesException, IllegalOperationException,
        BadVMPowerStateException, VMMException {
        // TODO
        throw new UnsupportedOperationException();
    }

    static VMMException translateXenAPIException(final Exception ex) {
        XenVirtualMachine.logger.error("XenAPI exception:" + ex.toString());
        if (ex instanceof Types.VmBadPowerState) {
            return new BadVMPowerStateException(ex.toString());
        }
        return new VMMException(ex.toString());
    }
}
