/**
 * JASMINe VMMapi: JASMINe Virtual Machine Management API
 * Copyright (C) 2009-2010 France Telecom R&D
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: Domain.java 7453 2011-01-19 22:52:49Z dangtran $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.vmm.agent.domain;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.management.InstanceNotFoundException;
import javax.management.MBeanRegistrationException;
import javax.management.ObjectName;

import org.apache.log4j.Logger;
import org.ow2.jasmine.vmm.agent.driver.Driver;
import org.ow2.jasmine.vmm.agent.driver.DriverFactory;
import org.ow2.jasmine.vmm.agent.jmx.MBeanObjectNamer;
import org.ow2.jasmine.vmm.agent.main.AgentCommon;
import org.ow2.jasmine.vmm.agent.main.VirtManagerAgent;
import org.ow2.jasmine.vmm.api.DomainMXBean;
import org.ow2.jasmine.vmm.api.InsufficientResourcesException;
import org.ow2.jasmine.vmm.api.InvalidVMConfigException;
import org.ow2.jasmine.vmm.api.NotificationTypes;
import org.ow2.jasmine.vmm.api.ServerPoolMXBean;
import org.ow2.jasmine.vmm.api.VMConfigSpec;
import org.ow2.jasmine.vmm.api.VMMException;
import org.ow2.jasmine.vmm.api.VirtualMachineMXBean;

/**
 * The Domain class is the implementation of the DomainMXBean interface. It
 * represents a domain managed entity acting as container of sub-domains and/or
 * server pools.
 */
public final class Domain extends ManagedResource implements DomainMXBean {
    static private Logger logger = Logger.getLogger(Domain.class);

    private String name;

    private final List<DomainMXBean> subDomains = new ArrayList<DomainMXBean>();

    private final List<ServerPoolMXBean> serverPools = new ArrayList<ServerPoolMXBean>();

    private final Map<String, String> attributes = new HashMap<String, String>();

    /**
     * Constucts a new domain with a given name and JMX object name
     * 
     * @param name user-friendly name of the domain
     * @param objectName JMX object name of the domain
     */
    public Domain(final String name, final ObjectName objectName, final Map<String, String> attributes) {
        super(objectName);
        this.attributes.putAll(attributes);
        this.name = name;
    }

    /*
     * (non-Javadoc)
     * @see org.ow2.jasmine.vmm.api.DomainMXBean#getLastSubDomainMXBean()
     */

    public DomainMXBean getLastSubDomainMXBean() {
        return this.subDomains.get(this.subDomains.size() - 1);
    }

    /*
     * (non-Javadoc)
     * @see org.ow2.jasmine.vmm.api.DomainMXBean#getName()
     */
    public String getName() {
        return this.name;
    }

    /*
     * (non-Javadoc)
     * @see org.ow2.jasmine.vmm.api.DomainMXBean#getSubDomains()
     */
    public List<DomainMXBean> getSubDomains() {
        return this.subDomains;
    }

    public void addSubDomain(final Domain d) {
        this.subDomains.add(d);
    }

    /*
     * (non-Javadoc)
     * @see
     * org.ow2.jasmine.vmm.api.DomainMXBean#addSubDomain(java.lang.String,javax
     * .management.ObjectName, java.util.Map)
     */
    public DomainMXBean addSubDomain(final String name, final Map<String, String> attributes) throws VMMException {
        if (this.lookUpSubDomainByName(name) != null) {
            throw new VMMException("Domain " + name + " already exists");
        }
        ObjectName domainObjectName = MBeanObjectNamer.makeDomainName(this.getPath() + "/" + name);
        Domain d = new Domain(name, domainObjectName, attributes);
        // this.addSubDomain(d);
        this.subDomains.add(d);
        try {
            AgentCommon.getMBeanServer().registerMBean(d, domainObjectName);
        } catch (Exception ex) {
            Domain.logger.error("Skipping domain " + name, ex);
            return null;
        }
        Domain.logger.info("Added Domain " + domainObjectName);
        this.emitNotification(NotificationTypes.DOMAIN_INVENTORY_ADD, "Add Domain", d.getObjectName());
        VirtManagerAgent.getInstance().saveConfiguration();
        return d;
    }

    /*
     * (non-Javadoc)
     * @see
     * org.ow2.jasmine.vmm.api.DomainMXBean#deleteDomain(org.ow2.jasmine.vmm
     * .api.DomainMXBean)
     */
    public void deleteSubDomain(final DomainMXBean domain) throws VMMException {
        int indexToRemove = -1;
        for (DomainMXBean domain_existant : this.subDomains) {
            if (domain_existant.getName().equals(domain.getName())) {
                indexToRemove = this.subDomains.indexOf(domain_existant);
                break;
            }
        }
        if (indexToRemove == -1) {
            throw new VMMException("Cannot find subdomain to delete");
        }
        DomainMXBean domainToRemove = this.subDomains.get(indexToRemove);
        while (domainToRemove.getSubDomains().size() > 0) {
            try {
                domainToRemove.deleteSubDomain(domainToRemove.getSubDomains().get(0));
            } catch (VMMException e) {
                Domain.logger.error(e);
            }
        }
        while (domainToRemove.getServerPools().size() > 0) {
            try {
                domainToRemove.deleteServerPool(domainToRemove.getServerPools().get(0));
            } catch (VMMException e) {
                Domain.logger.error(e);
            }
        }

        this.subDomains.remove(indexToRemove);
        ObjectName domainObjectName = domainToRemove.getObjectName();
        try {
            AgentCommon.getMBeanServer().unregisterMBean(domainObjectName);
            this.emitNotification(NotificationTypes.DOMAIN_INVENTORY_DEL, "suppresion domain", domainObjectName);
            VirtManagerAgent.getInstance().saveConfiguration();
        } catch (MBeanRegistrationException e) {
            Domain.logger.error(e);
        } catch (InstanceNotFoundException e) {
            Domain.logger.error(e);
        }
    }

    /*
     * (non-Javadoc)
     * @see
     * org.ow2.jasmine.vmm.api.DomainMXBean#deleteServer(org.ow2.jasmine.vmm
     * .api.ServerPoolMXBean)
     */
    public void deleteServerPool(final ServerPoolMXBean server) throws VMMException {
        ObjectName serverObjectName = server.getObjectName();
        int indexToRemove = -1;
        for (ServerPoolMXBean server_existant : this.serverPools) {
            try {
                if (server_existant.getName().equals(server.getName())) {
                    indexToRemove = this.serverPools.indexOf(server_existant);
                }
            } catch (Throwable e) {
                Domain.logger.error(e);
            }
        }
        ServerPool serverPoolToRemove = (ServerPool) this.serverPools.get(indexToRemove);
        while (serverPoolToRemove.getManagedHosts().size() > 0) {
            serverPoolToRemove.deleteHost(serverPoolToRemove.getManagedHosts().get(0));
        }

        serverPoolToRemove.deleteImageStore();

        this.serverPools.remove(indexToRemove);
        try {
            AgentCommon.getMBeanServer().unregisterMBean(server.getObjectName());
            VirtManagerAgent.getInstance().saveConfiguration();
        } catch (MBeanRegistrationException e) {
            Domain.logger.error(e);
        } catch (InstanceNotFoundException e) {
            Domain.logger.error(e);
        }
        this.emitNotification(NotificationTypes.SERVER_POOL_INVENTORY_DEL, "delete serverpool", serverObjectName);
    }

    /*
     * (non-Javadoc)
     * @see org.ow2.jasmine.vmm.api.DomainMXBean#getAttribute(java.lang.String)
     */
    public String getAttribute(final String key) {
        return this.attributes.get(key);
    }

    /**
     * Add a server pool to the domain
     * 
     * @param pool server pool to add
     */
    public void addServerPool(final ServerPool pool) {
        this.serverPools.add(pool);
    }

    /*
     * (non-Javadoc)
     * @see
     * org.ow2.jasmine.vmm.api.DomainMXBean#addServerPool(java.lang.String,javax
     * .management.ObjectName,java.util.Map,ava.lang.String)
     */
    public ServerPoolMXBean addServerPool(final String name, final Map<String, String> attributes, final String driverName)
        throws VMMException {
        if (this.lookUpServerPoolByName(name) != null) {
            throw new VMMException("ServerPool " + name + " already exists");
        }
        final ObjectName serverObjectName = MBeanObjectNamer.makeServerPoolName(this.getPath() + "/" + name);
        Driver driver = DriverFactory.getInstance().newDriver(driverName);
        ServerPool pool;
        if (driver != null) {
            try {
                pool = driver.newServerPool(name, serverObjectName, attributes);
            } catch (IllegalArgumentException ex) {
                Domain.logger.error("Skipping ServerPool " + name + ":" + ex.getMessage());
                this.emitNotification(NotificationTypes.SERVER_POOL_ERROR, "Skipping ServerPool", serverObjectName);
                throw new VMMException(ex.getMessage());
            }
            try {
                AgentCommon.getMBeanServer().registerMBean(pool, serverObjectName);
            } catch (Exception ex) {
                Domain.logger.error("Skipping ServerPool " + name, ex.getCause());
                this.emitNotification(NotificationTypes.SERVER_POOL_ERROR, "ServerPool already exists", serverObjectName);
                throw new VMMException(ex);
            }
            Domain.logger.info("Added ServerPool " + serverObjectName);
            this.addServerPool(pool);
            this.emitNotification(NotificationTypes.SERVER_POOL_INVENTORY_ADD, "Add ServerPool", serverObjectName);
            VirtManagerAgent.getInstance().saveConfiguration();
            return pool;
        } else {
            Domain.logger.error("Driver " + driverName + " not found");
            this.emitNotification(NotificationTypes.SERVER_POOL_ERROR, "Skipping ServerPool", serverObjectName);
            throw new VMMException("Driver " + driverName + " not found");
        }
    }

    /*
     * (non-Javadoc)
     * @see org.ow2.jasmine.vmm.api.DomainMXBean#getServerPools()
     */
    public List<ServerPoolMXBean> getServerPools() {
        return this.serverPools;
    }

    /*
     * (non-Javadoc)
     * @see org.ow2.jasmine.vmm.api.DomainMXBean#getLatestServerPool()
     */
    public ServerPoolMXBean getLatestServerPool() {
        return this.serverPools.get(this.serverPools.size() - 1);
    }

    private boolean areConstraintsSatisfied(final Map<String, String> constraints) {
        if (constraints == null) {
            return true;
        }
        String locationConstraint = constraints.get("location");
        if (locationConstraint == null || locationConstraint.equals("")) {
            return true;
        }
        String domainLocation = this.attributes.get("location");
        if (domainLocation == null) {
            return true;
        }
        if (domainLocation.equals(locationConstraint)) {
            return true;
        }
        return false;
    }

    /*
     * (non-Javadoc)
     * @see
     * org.ow2.jasmine.vmm.api.DomainMXBean#provisionVM(org.ow2.jasmine.vmm.
     * api.VMConfigSpec, java.util.Map, boolean)
     */
    public VirtualMachineMXBean provisionVM(final VMConfigSpec vmSpec, final Map<String, String> constraints, final boolean sync)
        throws InsufficientResourcesException, InvalidVMConfigException, VMMException {
        VirtualMachineMXBean vm = null;
        Domain.logger.info("Domain " + this.name + ": provisioning VM(name=" + vmSpec.getName() + ",imageID="
            + vmSpec.getVmImageUUID() + ")...");
        List<VMMException> raisedExceptions = new ArrayList<VMMException>();
        // first we try to provision the VM using one
        // of the server pools of the domain if any

        if (this.areConstraintsSatisfied(constraints)) {
            for (ServerPoolMXBean serverPool : this.serverPools) {
                try {
                    vm = serverPool.provisionVM(vmSpec, constraints, sync);
                    return vm;
                } catch (VMMException ex) {
                    raisedExceptions.add(ex);
                }
            }
        } else {
            throw new InsufficientResourcesException("Cannot meet constraints");
        }

        // next we recursively provision the VM on sub-domains
        for (DomainMXBean subDomain : this.subDomains) {
            try {
                vm = subDomain.provisionVM(vmSpec, constraints, sync);
                return vm;
            } catch (VMMException ex) {
                raisedExceptions.add(ex);
            }
        }
        for (VMMException ex : raisedExceptions) {
            if (ex instanceof InsufficientResourcesException) {
                throw (InsufficientResourcesException) ex;
            }
        }
        for (VMMException ex : raisedExceptions) {
            if (ex instanceof InvalidVMConfigException) {
                throw (InvalidVMConfigException) ex;
            }
        }
        for (VMMException ex : raisedExceptions) {
            throw ex;
        }
        throw new InsufficientResourcesException();
    }

    private DomainMXBean lookUpSubDomainByName(final String name) {
        for (DomainMXBean domain : this.subDomains) {
            if (domain.getName().equals(name)) {
                return domain;
            }
        }
        return null;
    }

    private ServerPoolMXBean lookUpServerPoolByName(final String name) {
        for (ServerPoolMXBean pool : this.serverPools) {
            if (pool.getName().equals(name)) {
                return pool;
            }
        }
        return null;
    }

}
