/**
 * JASMINe VMMapi: JASMINe Virtual Machine Management API
 * Copyright (C) 2009-2010 France Telecom R&D
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: ServerPool.java 7453 2011-01-19 22:52:49Z dangtran $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.vmm.agent.domain;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.management.MBeanNotificationInfo;
import javax.management.Notification;
import javax.management.ObjectName;

import org.apache.log4j.Logger;
import org.ow2.jasmine.vmm.agent.domain.policy.PolicyRegistry;
import org.ow2.jasmine.vmm.agent.domain.policy.VMPlacementPolicy;
import org.ow2.jasmine.vmm.agent.main.AgentCommon;
import org.ow2.jasmine.vmm.agent.main.VirtManagerAgent;
import org.ow2.jasmine.vmm.api.HostMXBean;
import org.ow2.jasmine.vmm.api.InsufficientResourcesException;
import org.ow2.jasmine.vmm.api.InvalidVMConfigException;
import org.ow2.jasmine.vmm.api.NotificationTypes;
import org.ow2.jasmine.vmm.api.ServerPoolMXBean;
import org.ow2.jasmine.vmm.api.VMConfigSpec;
import org.ow2.jasmine.vmm.api.VMMException;
import org.ow2.jasmine.vmm.api.VirtualMachineImageStoreMXBean;
import org.ow2.jasmine.vmm.api.VirtualMachineMXBean;

/**
 * Base class partially implementing the ServerPoolMXBean interface and expected
 * to be sub-classed on a per driver basis.
 * <p>
 * A ServerPool is associated with a VM placement policy which determines where
 * a provisioned VM is to be placed on one host among the members of the server
 * pool
 */
public abstract class ServerPool extends ManagedResource implements ServerPoolMXBean {
    static protected Logger logger = Logger.getLogger(ServerPool.class);

    protected String servername;

    protected Map<String, String> attributes = new HashMap<String, String>();

    protected final List<HostMXBean> hosts = new ArrayList<HostMXBean>();

    protected VMPlacementPolicy placementPolicy = PolicyRegistry.getInstance().newDefaultPolicy();;

    /**
     * Constructs a new server pool
     * 
     * @param name symbolic name of the server pool
     * @param objectName JMX object name of the server pool
     */
    public ServerPool(final String name, final ObjectName objectName, final Map<String, String> attributes) {
        super(objectName);
        this.servername = name;
        this.attributes = attributes;
    }

    /*
     * @see org.ow2.jasmine.vmm.api.ServerPoolMXBean#getName()
     */
    public String getName() {
        return this.servername;
    }

    abstract public HostMXBean newHost(String hostName, Map<String, String> props) throws VMMException;

    public Map<String, String> getAttributes() {
        return this.attributes;
    }

    /**
     * Adds a new host to the server pool
     * 
     * @param host host to add
     */
    protected void addHost(final HostMXBean host) {
        this.hosts.add(host);
        try {
            ServerPool.logger.info("Added host " + host.getHostName());
        } catch (VMMException ex) {
            ServerPool.logger.error(ex);
        }
    }

    /*
     * @see org.ow2.jasmine.vmm.api.ServerPoolMXBean#addHost(java.util.Map)
     */
    public HostMXBean addHost(final Map<String, String> attributes) throws VMMException {
        String hostName = attributes.get("name");
        if (hostName == null) {
            throw new VMMException("Missing host name attribute");
        }
        for (HostMXBean host : this.hosts) {
            if (host.getHostName().equals(hostName)) {
                throw new VMMException("Host " + hostName + " already exists");
            }
        }
        HostMXBean host = null;
        try {
            host = this.newHost(hostName, attributes);
        } catch (IllegalArgumentException ex) {
            DomainBuilder.logger.error("Cannot add host " + hostName + ":" + ex.getMessage());
            throw new VMMException(ex.getMessage());
        }
        if (host != null) {
            this.emitNotification(NotificationTypes.HOST_INVENTORY_ADD, "Ajout host", host.getObjectName());
            VirtManagerAgent.getInstance().saveConfiguration();
        }
        return host;
    }

    /*
     * @see org.ow2.jasmine.vmm.api.ServerPoolMXBean#deleteHost(org.ow2.jasmine.
     * vmm.api.HostMXBean)
     */
    public void deleteHost(final HostMXBean host) throws VMMException {
        ObjectName hostObjectName = null;

        hostObjectName = host.getObjectName();

        int indexToRemove = -1;
        for (HostMXBean h : this.hosts) {
            if (h.getObjectName().equals(hostObjectName)) {
                indexToRemove = this.hosts.indexOf(h);
            }
        }
        if (indexToRemove == -1) {
            throw new VMMException("Cannot find host " + host.getHostName());
        }
        HostMXBean hostToRemove = this.hosts.get(indexToRemove);
        if (hostToRemove instanceof AbstractHost) {
            ((AbstractHost) hostToRemove).removeFromInventory();
            this.hosts.remove(indexToRemove);
            try {
                AgentCommon.getMBeanServer().unregisterMBean(host.getObjectName());
            } catch (Exception ex) {
                ServerPool.logger.error(ex);
            }
            this.emitNotification(NotificationTypes.HOST_INVENTORY_DEL, "host delete", hostObjectName);
            VirtManagerAgent.getInstance().saveConfiguration();
        } else {
            this.emitNotification(NotificationTypes.HOST_ERROR, "host delete error", hostObjectName);
        }
    }

    /*
     * @see
     * org.ow2.jasmine.vmm.api.ServerPoolMXBean#moveHost(org.ow2.jasmine.vmm
     * .api.HostMXBean, org.ow2.jasmine.vmm.api.ServerPoolMXBean)
     */
    public void moveHost(final HostMXBean host, final ServerPoolMXBean server) throws VMMException {
        Map<String, String> attributes = host.getAttributes();
        this.deleteHost(host);
        server.addHost(attributes);
    }

    public void setProvisioningPolicy(final String policyId) {
        VMPlacementPolicy newPlacementPolicy = PolicyRegistry.getInstance().newPolicy(policyId);
        if (newPlacementPolicy != null) {
            this.placementPolicy = newPlacementPolicy;
            ServerPool.logger.info("Domain " + this.servername + ": policy set to " + policyId);
        }
    }

    /*
     * @see
     * org.ow2.jasmine.vmm.api.ServerPoolMXBean#provisionVM(org.ow2.jasmine.
     * vmm.api.VMConfigSpec, java.util.Map, boolean)
     */
    public VirtualMachineMXBean provisionVM(final VMConfigSpec vmSpec, final Map<String, String> constraints, final boolean sync)
        throws InsufficientResourcesException, InvalidVMConfigException, VMMException {
        if (this.getVMImageStore().lookUpByUUID(vmSpec.getVmImageUUID()) == null) {
            throw new InvalidVMConfigException("Illegal VMImage UUID " + vmSpec.getVmImageUUID());
        }
        HostMXBean candidateHost = this.placementPolicy.placeVM(this, vmSpec);
        if (candidateHost == null) {
            throw new InsufficientResourcesException();
        }
        return candidateHost.createVM(vmSpec, sync);
    }

    /*
     * @see org.ow2.jasmine.vmm.api.ServerPoolMXBean#getVMImageStore()
     */
    abstract public VirtualMachineImageStoreMXBean getVMImageStore();

    abstract public void deleteImageStore();

    /*
     * @see org.ow2.jasmine.vmm.api.ServerPoolMXBean#getManagedHosts()
     */
    public List<HostMXBean> getManagedHosts() {
        return this.hosts;
    }

    /*
     * @see org.ow2.jasmine.vmm.api.ServerPoolMXBean#getLatestHost()
     */
    public HostMXBean getLatestHost() {
        return this.hosts.get(this.hosts.size() - 1);
    }

    @Override
    public MBeanNotificationInfo[] getNotificationInfo() {
        return new MBeanNotificationInfo[] {new MBeanNotificationInfo(new String[] {NotificationTypes.HOST_INVENTORY_ADD,
            NotificationTypes.HOST_INVENTORY_DEL, NotificationTypes.SERVER_POOL_INVENTORY_ADD,
            NotificationTypes.SERVER_POOL_INVENTORY_DEL, NotificationTypes.LOG, NotificationTypes.ERROR},
            Notification.class.getName(), "Server Pool event")};
    }
}
