/**
 * JASMINe VMMapi: JASMINe Virtual Machine Management API
 * Copyright (C) 2009 France Telecom R&D
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: PolicyRegistry.java 6497 2010-05-24 14:47:27Z dangtran $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.vmm.agent.domain.policy;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Map;
import java.util.Properties;

import org.apache.log4j.Logger;
import org.ow2.jasmine.vmm.agent.main.AgentControl;

/**
 * The PolicyRegistry is a singleton class which acts as a factory of VM
 * placement policies. A VM placement policy can be specified through an
 * user-friendly policy Id in a domain XML configuration file. The mapping
 * between theses aliases and Java classes is resolved by a a property file
 * (etc/vm_placement_map.properties).
 */
public final class PolicyRegistry {
    private static Logger logger = Logger.getLogger(PolicyRegistry.class);

    private static final String VM_PLACEMENT_MAP_FILENAME = "vm_placement_map.properties";

    private static final String DEFAULT_POLICY_ID = "room-for-growth";

    private static PolicyRegistry instance;

    private final Map<String, String> map = new HashMap<String, String>();

    private PolicyRegistry() {
        Properties properties = new Properties();
        String agentPropFileName = "etc" + File.separator + PolicyRegistry.VM_PLACEMENT_MAP_FILENAME;
        try {
            properties.load(new FileInputStream(new File(AgentControl.vmmHome, agentPropFileName)));
        } catch (IOException ex) {
            PolicyRegistry.logger.warn("Cannot find " + PolicyRegistry.VM_PLACEMENT_MAP_FILENAME + " file");
        }
        for (Enumeration<Object> it = properties.keys(); it.hasMoreElements();) {
            String policyAlias = (String) it.nextElement();
            String policyClass = properties.getProperty(policyAlias);
            this.map.put(policyAlias, policyClass);
            PolicyRegistry.logger.info("Added alias " + policyAlias + " for " + policyClass);

        }
    }

    /**
     * Returns the single PolicyRegistry instance
     * 
     * @return
     */
    public static PolicyRegistry getInstance() {
        if (PolicyRegistry.instance == null) {
            PolicyRegistry.instance = new PolicyRegistry();
        }
        return PolicyRegistry.instance;
    }

    /**
     * Returns the VM placement policy corresponding to the provided policy Id
     * 
     * @param policyId the policy Id of the VM placement policy to create
     * @return
     */
    public VMPlacementPolicy newPolicy(final String policyId) {
        String policyClassName = this.map.get(policyId);
        VMPlacementPolicy policy = null;
        if (policyClassName != null) {
            try {
                policy = ((VMPlacementPolicy) Class.forName(policyClassName).newInstance());
            } catch (Exception ex) {
                PolicyRegistry.logger.error("Failed to instance policy " + policyId, ex);
            }
        } else {
            PolicyRegistry.logger.error("Unknown driver " + policyId);
        }
        return policy;
    }

    /**
     * Returns the default VM placement policy
     * 
     * @return
     */
    public VMPlacementPolicy newDefaultPolicy() {
        return this.newPolicy(PolicyRegistry.DEFAULT_POLICY_ID);
    }
}
