/**
 * JASMINe VMMapi: JASMINe Virtual Machine Management API
 * Copyright (C) 2009 France Telecom R&D
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: DriverFactory.java 6497 2010-05-24 14:47:27Z dangtran $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.vmm.agent.driver;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Map;
import java.util.Properties;

import org.apache.log4j.Logger;
import org.ow2.jasmine.vmm.agent.main.AgentControl;

/**
 * The DriverFactory class is a singleton which acts a factory of Driver
 * objects. Driver are identified with symbolic aliases in domain XML files. The
 * mapping between driver aliases and Java Driver classes is specified in the
 * etc/drivermap.properties file.
 */
public class DriverFactory {
    static protected Logger logger = Logger.getLogger(DriverFactory.class);

    private static final String DRIVERMAP_FILENAME = "drivermap.properties";

    static private DriverFactory instance;

    private Map<String, String> map = new HashMap<String, String>();

    /**
     * Returns the single DriverFactory instance
     * 
     * @return
     */
    static public DriverFactory getInstance() {
        if (DriverFactory.instance == null) {
            DriverFactory.instance = new DriverFactory();
        }
        return DriverFactory.instance;
    }

    private DriverFactory() {
        Properties properties = new Properties();
        String agentPropFileName = "etc" + File.separator + DriverFactory.DRIVERMAP_FILENAME;
        try {
            properties.load(new FileInputStream(new File(AgentControl.vmmHome, agentPropFileName)));
        } catch (IOException ex) {
            DriverFactory.logger.warn("Cannot find " + DriverFactory.DRIVERMAP_FILENAME + " file");
        }
        for (Enumeration<Object> it = properties.keys(); it.hasMoreElements();) {
            String driverAlias = (String) it.nextElement();
            String driverClass = properties.getProperty(driverAlias);
            this.map.put(driverAlias, driverClass);
            DriverFactory.logger.info("Added alias " + driverAlias + " for " + driverClass);

        }
        // built-in drivers
        if (!this.map.containsKey("xen")) {
            this.map.put("xen", "org.ow2.jasmine.vmmapi.driver.xen.XenDriver");
        }
        if (!this.map.containsKey("vmware-vi")) {
            this.map.put("vmware-vi", "org.ow2.jasmine.vmmapi.driver.vmware.VMwareDriver");
        }
        if (!this.map.containsKey("dummy")) {
            this.map.put("dummy", "org.ow2.jasmine.vmmapi.driver.dummy.DummyDriver");
        }

    }

    /**
     * Builds a new Driver object
     * 
     * @param driverName alias of the driver to create
     * @return the Driver object
     */
    public Driver newDriver(final String driverName) {
        String driverClassName = this.map.get(driverName);
        Driver driver = null;
        if (driverClassName != null) {
            try {
                driver = ((Driver) Class.forName(driverClassName).newInstance());
            } catch (Exception ex) {
                DriverFactory.logger.error("Failed to instance driver " + driverName, ex);
            }
        } else {
            DriverFactory.logger.error("Unknown driver " + driverName);
        }
        return driver;
    }

}
