/**
 * JASMINe VMMapi: JASMINe Virtual Machine Management API
 * Copyright (C) 2009-2010 France Telecom R&D
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: DummyHost.java 7454 2011-01-19 22:56:21Z dangtran $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.vmm.agent.driver.dummy;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Random;
import java.util.Set;
import java.util.Timer;
import java.util.TimerTask;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

import javax.management.MBeanNotificationInfo;
import javax.management.Notification;
import javax.management.ObjectName;
import javax.management.openmbean.CompositeData;

import org.apache.log4j.Logger;
import org.ow2.jasmine.vmm.agent.domain.AbstractHost;
import org.ow2.jasmine.vmm.agent.driver.util.ResourceUsageHelper;
import org.ow2.jasmine.vmm.agent.jmx.MBeanObjectNamer;
import org.ow2.jasmine.vmm.agent.main.AgentCommon;
import org.ow2.jasmine.vmm.api.InsufficientResourcesException;
import org.ow2.jasmine.vmm.api.InvalidVMConfigException;
import org.ow2.jasmine.vmm.api.NotificationTypes;
import org.ow2.jasmine.vmm.api.ResourceUsage;
import org.ow2.jasmine.vmm.api.ServerPoolMXBean;
import org.ow2.jasmine.vmm.api.VMConfigSpec;
import org.ow2.jasmine.vmm.api.VMCustomizationSpec;
import org.ow2.jasmine.vmm.api.VMMException;
import org.ow2.jasmine.vmm.api.VirtualMachineImageStoreMXBean;
import org.ow2.jasmine.vmm.api.VirtualMachineMXBean;

/**
 * Dummy driver Host MXBean implementation
 */
class DummyHost extends AbstractHost {
    static Logger logger = Logger.getLogger(DummyHost.class);

    private List<DummyVirtualMachine> vms = new CopyOnWriteArrayList<DummyVirtualMachine>();

    private static ExecutorService executorService = Executors.newFixedThreadPool(3);

    private Timer perfTimer;

    private String hostName;

    private final ObjectName objectName;

    private HostPowerState hostPowerState = HostPowerState.UNKNOWN;

    private int numCPU = 4;

    private int totalMemoryMB, freeMemoryMB;

    private int cpuCapacityMHz;

    private DummyServerPool pool;

    private static Random rand = new Random(System.currentTimeMillis());

    public DummyHost(final DummyServerPool pool, final String hostName, final ObjectName objectName, final Boolean active) {
        super(objectName);
        this.pool = pool;
        this.hostName = hostName;
        this.objectName = objectName;
        this.hostPowerState = HostPowerState.RUNNING;

        if (DummyHost.rand.nextInt() % 2 == 0) {
            this.totalMemoryMB = this.freeMemoryMB = 4096;
            this.cpuCapacityMHz = 1000;
        } else {
            this.totalMemoryMB = this.freeMemoryMB = 8192;
            this.cpuCapacityMHz = 2300;
        }
        this.startPerfMonitor();
    }

    public boolean isConnectionLost() {
        return false;
    }

    public ServerPoolMXBean getServerPool() {
        return this.pool;
    }

    private void addVM(final DummyVirtualMachine vm) {
        this.vms.add(vm);
    }

    private void removeVM(final DummyVirtualMachine vm) {
        this.vms.remove(vm);
    }

    void postMigrateVM(final DummyVirtualMachine vm, final DummyHost newHost) {
        this.removeVM(vm);
        newHost.addVM(vm);
        this.freeMemoryMB += vm.getMemorySizeMB();
        newHost.freeMemoryMB -= vm.getMemorySizeMB();
    }

    public void postDestroyVM(final DummyVirtualMachine vm) {
        this.removeVM(vm);
        this.freeMemoryMB += vm.getMemorySizeMB();
    }

    @Override
    public ObjectName getObjectName() {
        return this.objectName;
    }

    public synchronized void onDestroyVM(final DummyVirtualMachine vm) {
        try {
            AgentCommon.getMBeanServer().unregisterMBean(vm.getObjectName());
        } catch (Exception ex) {
            DummyHost.logger.error("Cannot unregister VM MBean", ex);
        }
        this.removeVM(vm);
        this.freeMemoryMB += vm.getMemorySizeMB();
        ((DummyServerPool.DummyVirtualMachineImageStore) this.pool.getVMImageStore()).updateFreeSpace(vm.getDiskSizeMB());
        this.emitNotification(NotificationTypes.VM_DEL, "Destroyed", vm.getObjectName());
        DummyHost.logger.info("destroyed VM " + vm.getObjectName());
    }

    public synchronized void removeVMFromInventory(final DummyVirtualMachine vm) {
        try {
            AgentCommon.getMBeanServer().unregisterMBean(vm.getObjectName());
        } catch (Exception ex) {
            DummyHost.logger.error("Cannot unregister VM MBean", ex);
        }
        this.removeVM(vm);
        this.emitNotification(NotificationTypes.VM_INVENTORY_DEL, "Deleted", vm.getObjectName());
        DummyHost.logger.info("deleted VM " + vm.getObjectName());
    }

    public VirtualMachineMXBean createVM(final VMConfigSpec vmSpec, final boolean sync) throws InsufficientResourcesException,
        InvalidVMConfigException, VMMException {
        final VMConfigSpec vmSpec2 = new VMConfigSpec(vmSpec);
        // check name uniqueness
        for (DummyVirtualMachine dvm : this.vms) {
            if (dvm.getNameLabel().equals(vmSpec2.getName())) {
                throw new InvalidVMConfigException("VM name already exists");
            }
        }
        if (vmSpec2.getMemorySizeMB() > this.getFreeMemoryMB()) {
            throw new InsufficientResourcesException();
        }

        if (!sync) {
            DummyHost.executorService.execute(new Runnable() {
                public void run() {
                    try {
                        DummyHost.this.createVM2(vmSpec2);
                    } catch (Exception ex) {
                    }
                };
            });
            return null;
        } else {
            return this.createVM2(vmSpec2);
        }

    }

    private VirtualMachineMXBean createVM2(final VMConfigSpec vmSpec) throws InsufficientResourcesException,
        InvalidVMConfigException, VMMException {
        DummyVirtualMachine vm = null;

        try {
            Thread.sleep(5000);
            Random random = new Random(System.currentTimeMillis());
            String uuid = Integer.toString(random.nextInt(10000));
            ObjectName vmName = MBeanObjectNamer.makeVirtualMachineName(
                this.getServerPool().getPath() + "/" + vmSpec.getName(), uuid);
            vm = new DummyVirtualMachine(vmSpec.getName(), uuid, vmName, this);
            this.addVM(vm);
            vm.setDiskSizeMB(vmSpec.getDiskSizeMB());
            vm.setNumVCPUs(vmSpec.getNumVCPU());
            vm.setMemorySizeMB((int) vmSpec.getMemorySizeMB());
            vm.addUserData("imageID", vmSpec.getVmImageUUID());
            AgentCommon.getMBeanServer().registerMBean(vm, vm.getObjectName());
            DummyHost.logger.info("VM Creation: registered VirtualMachineMBean " + vm.getObjectName());
        } catch (Exception ex) {
            DummyHost.logger.error("Failed to create VM", ex);
            return null;
        }
        this.emitNotification(NotificationTypes.VM_ADD, "Created", vm.getObjectName());

        this.freeMemoryMB -= vmSpec.getMemorySizeMB();
        ((DummyServerPool.DummyVirtualMachineImageStore) this.pool.getVMImageStore()).updateFreeSpace(-vmSpec.getDiskSizeMB());
        return vm;
    }

    public VirtualMachineMXBean cloneVM(final DummyVirtualMachine sourceVM, final String clonedVMName,
        final VMCustomizationSpec custSpec, final boolean sync) throws InsufficientResourcesException,
        InvalidVMConfigException, VMMException {
        // check name uniqueness
        for (DummyVirtualMachine dvm : this.vms) {
            if (dvm.getNameLabel().equals(clonedVMName)) {
                throw new InvalidVMConfigException("VM name already exists");
            }
        }

        if (!sync) {
            DummyHost.executorService.execute(new Runnable() {
                public void run() {
                    try {
                        DummyHost.this.cloneVM2(sourceVM, clonedVMName, custSpec);
                    } catch (Exception ex) {
                    }
                };
            });
            return null;
        } else {
            return this.cloneVM2(sourceVM, clonedVMName, custSpec);
        }
    }

    private VirtualMachineMXBean cloneVM2(final DummyVirtualMachine sourceVM, final String clonedVMName,
        final VMCustomizationSpec custSpec) throws InsufficientResourcesException, InvalidVMConfigException, VMMException {
        DummyVirtualMachine vm = null;

        try {
            Thread.sleep(5000);
            Random random = new Random(System.currentTimeMillis());
            String uuid = Integer.toString(random.nextInt(10000));
            ObjectName vmName = MBeanObjectNamer.makeVirtualMachineName(this.getServerPool().getPath() + "/" + clonedVMName,
                uuid);
            vm = new DummyVirtualMachine(clonedVMName, uuid, vmName, this);
            this.addVM(vm);
            vm.setDiskSizeMB(sourceVM.getDiskSizeMB());
            vm.setNumVCPUs(sourceVM.getNumVCPUs());
            vm.setMemorySizeMB((int) sourceVM.getMemorySizeMB());
            AgentCommon.getMBeanServer().registerMBean(vm, vm.getObjectName());
            DummyHost.logger.info("VM Cloning: registered VirtualMachineMBean " + vm.getObjectName());
        } catch (Exception ex) {
            DummyHost.logger.error("Failed to clone VM", ex);
            return null;
        }
        this.emitNotification(NotificationTypes.VM_ADD, "Created", vm.getObjectName());

        this.freeMemoryMB -= sourceVM.getMemorySizeMB();
        ((DummyServerPool.DummyVirtualMachineImageStore) this.pool.getVMImageStore())
            .updateFreeSpace(-sourceVM.getDiskSizeMB());
        return vm;
    }

    HashMap<String, String> hypervisorInfo;

    public Map<String, String> getHypervisorInfo() {
        if (this.hypervisorInfo == null) {
            this.hypervisorInfo = new HashMap<String, String>();
            this.hypervisorInfo.put("name", "DummyHypervisor");
            this.hypervisorInfo.put("vendor", "");
            this.hypervisorInfo.put("version", "1.0");
        }
        return this.hypervisorInfo;
    }

    HashMap<String, String> cpuInfo;

    public Map<String, String> getCPUInfo() {
        if (this.cpuInfo == null) {
            this.cpuInfo = new HashMap<String, String>();
            this.cpuInfo.put("model", "Core 2 Duo");
            this.cpuInfo.put("speedMHz", Integer.toString(this.cpuCapacityMHz));
            this.cpuInfo.put("vendor", "Intel");
        }
        return this.cpuInfo;
    }

    public int getNumCPU() {
        return this.numCPU;
    }

    public float[] getLoadPerCPU() {
        return new float[] {0, 0};
    }

    public float getCPULoad() {
        float load = 0;
        for (DummyVirtualMachine vm : DummyHost.this.vms) {
            load += vm.getCPULoad();
        }
        return Math.min(1, load);
    }

    public Map<String, Float> getVMCPULoads() {
        HashMap<String, Float> cpuLoads = new HashMap<String, Float>();
        for (DummyVirtualMachine vm : DummyHost.this.vms) {
            cpuLoads.put(vm.getNameLabel(), vm.getResourceUsage().getCpuLoad());
        }
        return cpuLoads;
    }

    public long getFreeMemoryMB() {
        return this.freeMemoryMB;
    }

    public String getHostName() {
        return this.hostName;
    }

    public List<VirtualMachineMXBean> getResidentVMs() {
        List<VirtualMachineMXBean> vmList = new ArrayList<VirtualMachineMXBean>();
        for (VirtualMachineMXBean vm : this.vms) {
            vmList.add(vm);
        }
        return vmList;
    }

    public long getTotalMemoryMB() {
        return this.totalMemoryMB;
    }

    public VirtualMachineImageStoreMXBean getVMImageStore() {
        return this.pool.getVMImageStore();
    }

    public String getUUID() {
        return null;
    }

    public Map<String, ResourceUsage> getVMResourceUsage() {
        Map<String, ResourceUsage> result = new HashMap<String, ResourceUsage>();
        for (DummyVirtualMachine vm : DummyHost.this.vms) {
            result.put(vm.getNameLabel(), vm.getResourceUsage());
        }
        return result;
    }

    @Override
    public MBeanNotificationInfo[] getNotificationInfo() {
        return new MBeanNotificationInfo[] {new MBeanNotificationInfo(new String[] {NotificationTypes.VM_STATE_CHANGE,
            NotificationTypes.VM_ADD, NotificationTypes.VM_DEL, NotificationTypes.VM_INVENTORY_DEL, NotificationTypes.LOG,
            NotificationTypes.ERROR, NotificationTypes.PERF_REPORT}, Notification.class.getName(), "Host event")};
    }

    private Set<PerfMetric> currentMonitoredMetrics;

    private long currentMonitoringPeriod = 1000;

    public void configurePerfMonitor(final Set<PerfMetric> metricsOfInterest, long periodMillis) {
        this.stopPerfMonitor();
        if (periodMillis < 1000) {
            periodMillis = 1000;
        }
        if (metricsOfInterest.contains(PerfMetric.VM_CPU_LOAD)) {
            this.currentMonitoringPeriod = periodMillis;
            this.currentMonitoredMetrics = metricsOfInterest;
            this.startPerfMonitor();
        }
    }

    private void startPerfMonitor() {
        if (this.perfTimer == null) {
            DummyHost.logger.info("Host " + this.hostName + ": starting periodic perf monitor task");
            this.perfTimer = new Timer();
            this.perfTimer.schedule(new PerfTimerTask(), 0, this.currentMonitoringPeriod);
        }
    }

    private void stopPerfMonitor() {
        if (this.perfTimer != null) {
            DummyHost.logger.info("Host " + this.hostName + ": stopping periodic perf monitor task");
            this.perfTimer.cancel();
        }
        this.perfTimer = null;
    }

    private class PerfTimerTask extends TimerTask {
        @Override
        public void run() {
            Map<String, Object> result = new HashMap<String, Object>();
            try {
                float totalCpuLoad = 0;
                for (DummyVirtualMachine vm : DummyHost.this.vms) {
                    vm.currentCpuLoad += DummyHost.rand.nextFloat() * 0.2 - 0.1;
                    if (vm.currentCpuLoad <= 0.0) {
                        vm.currentCpuLoad = 0;
                    } else if (vm.currentCpuLoad > 1 - totalCpuLoad) {
                        vm.currentCpuLoad = 1 - totalCpuLoad;
                    }
                    totalCpuLoad += vm.currentCpuLoad;

                    CompositeData cd = (CompositeData) ResourceUsageHelper.serialize(vm.getResourceUsage());
                    result.put(vm.getNameLabel(), cd);
                }
            } catch (Exception ex) {
                DummyHost.logger.error(ex);
                return;
            }
            DummyHost.this.emitNotification(NotificationTypes.PERF_REPORT, "Resource Usage", result);
        }
    }

    /*
     * (non-Javadoc)
     * @see org.ow2.jasmine.vmm.api.HostMXBean#getAttribut()
     */
    public Map<String, String> getAttributes() {
        Map<String, String> attributes = new HashMap<String, String>();
        attributes.put("name", this.hostName);
        attributes.put("driver", "dummy");
        return attributes;
    }

    public String getDriver() {
        return "dummy";
    }

    /*
     * (non-Javadoc)
     * @see org.ow2.jasmine.vmm.api.HostMXBean#stopSynchronizer()
     */
    @Override
    public void removeFromInventory() {
        while (this.getResidentVMs().size() > 0) {
            this.removeVMFromInventory((DummyVirtualMachine) this.getResidentVMs().get(0));
        }
    }

    /*
     * (non-Javadoc)
     * @see org.ow2.jasmine.vmm.api.HostMXBean#start()
     */
    public void start() {
        this.startPerfMonitor();
        this.hostPowerState = HostPowerState.RUNNING;
    }

    /*
     * (non-Javadoc)
     * @see org.ow2.jasmine.vmm.api.HostMXBean#stop()
     */
    public void stop() {
        while (this.getResidentVMs().size() > 0) {
            this.removeVMFromInventory((DummyVirtualMachine) this.getResidentVMs().get(0));
        }
        this.stopPerfMonitor();
        this.hostPowerState = HostPowerState.HALTED;
    }

    /*
     * (non-Javadoc)
     * @see org.ow2.jasmine.vmm.api.HostMXBean#getHostState()
     */
    public HostPowerState getHostState() throws VMMException {
        return this.hostPowerState;
    }
}
