/**
 * JASMINe VMMapi: JASMINe Virtual Machine Management API
 * Copyright (C) 2009-2010 France Telecom R&D
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: DummyServerPool.java 7809 2011-03-09 14:28:26Z dangtran $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.vmm.agent.driver.dummy;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CopyOnWriteArrayList;

import javax.management.ObjectName;

import org.apache.log4j.Logger;
import org.ow2.jasmine.vmm.agent.domain.ManagedResource;
import org.ow2.jasmine.vmm.agent.domain.ServerPool;
import org.ow2.jasmine.vmm.agent.jmx.MBeanObjectNamer;
import org.ow2.jasmine.vmm.agent.main.AgentCommon;
import org.ow2.jasmine.vmm.api.HostMXBean;
import org.ow2.jasmine.vmm.api.VMMException;
import org.ow2.jasmine.vmm.api.VirtualMachineImageMXBean;
import org.ow2.jasmine.vmm.api.VirtualMachineImageStoreMXBean;

/**
 * Dummy driver ServerPool MXBean implementation
 */
class DummyServerPool extends ServerPool {
    static protected Logger logger = Logger.getLogger(DummyServerPool.class);

    private VirtualMachineImageStoreMXBean imageStore;

    private static int counter = 0;

    public DummyServerPool(final String name, final ObjectName objectName, final Map<String, String> attributs) {
        super(name, objectName, attributs);
        ObjectName storeObjectName = MBeanObjectNamer.makeVMImageStoreName(name + "-store" + DummyServerPool.counter++);
        try {
            this.imageStore = new DummyVirtualMachineImageStore(storeObjectName);
            AgentCommon.getMBeanServer().registerMBean(this.imageStore, storeObjectName);
        } catch (Exception ex) {
            DummyServerPool.logger.error("Cannot register Dummy VMImageStore", ex);
        }
    }

    @Override
    public String getHypervisor() {
        return "DummyHypervisor";
    }

    @Override
    public HostMXBean newHost(final String hostName, final Map<String, String> props) throws VMMException {
        ObjectName objectName = MBeanObjectNamer.makeHostName(this.getPath() + "/" + hostName, hostName);
        DummyHost host = new DummyHost(this, hostName, objectName, true);
        try {
            AgentCommon.getMBeanServer().registerMBean(host, objectName);
        } catch (Exception ex) {
            DummyServerPool.logger.error("Unable to register MBean " + objectName, ex);
            throw new VMMException(ex);
        }
        this.addHost(host);
        return host;
    }

    @Override
    public VirtualMachineImageStoreMXBean getVMImageStore() {
        return this.imageStore;
    }

    static private class DummyVirtualMachineImage extends ManagedResource implements VirtualMachineImageMXBean {
        private String uuid;

        private String name;

        private String description;

        private Map<String, String> metaData;

        public DummyVirtualMachineImage(final ObjectName objectName, final String uuid, final String name,
            final String description, final Map<String, String> metaData) {
            super(objectName);
            this.name = name;
            this.uuid = uuid;
            this.description = description;
            this.metaData = metaData;
        }

        public String getUUID() {
            return this.uuid;
        }

        public String getName() {
            return this.name;
        }

        public Map<String, String> getMetaData() {
            return this.metaData;
        }

        public String getDescription() {
            return this.description;
        }

    }

    static class DummyVirtualMachineImageStore extends ManagedResource implements VirtualMachineImageStoreMXBean {
        private static long INIT_STORAGE_CAPACITY_MB = 1024 * 128;

        private CopyOnWriteArrayList<DummyVirtualMachineImage> images;

        private long freeSpaceMB;

        private static int counter = 100;

        public DummyVirtualMachineImageStore(final ObjectName objectName) throws Exception {
            super(objectName);
            this.images = new CopyOnWriteArrayList<DummyVirtualMachineImage>();
            this.addImage("Used by unit tests", "A dummy Image", null);
            this.addImage("Debian Lenny", "A dummy Image", null);
            this.addImage("Fedora 10", "A dummy Image", null);
            this.freeSpaceMB = DummyVirtualMachineImageStore.INIT_STORAGE_CAPACITY_MB;
        }

        public DummyVirtualMachineImage addImage(final String name, final String description, final Map<String, String> metaData)
            throws Exception {
            String uuid = "vmi-" + (DummyVirtualMachineImageStore.counter++);
            ObjectName on = MBeanObjectNamer.makeVMImageName(uuid);
            DummyVirtualMachineImage image = new DummyVirtualMachineImage(on, uuid, name, description, metaData);
            AgentCommon.getMBeanServer().registerMBean(image, on);
            this.images.add(image);
            return image;
        }

        public long getCapacityMB() {
            return DummyVirtualMachineImageStore.INIT_STORAGE_CAPACITY_MB;
        }

        public long getFreeSpaceMB() {
            return this.freeSpaceMB;
        }

        void updateFreeSpace(final long deltaMB) {
            this.freeSpaceMB += deltaMB;
        }

        public String getName() {
            return "ImageStore(default)";
        }

        public VirtualMachineImageMXBean lookUpByUUID(final String uuid) {
            for (DummyVirtualMachineImage image : this.images) {
                if (image.getUUID().equals(uuid)) {
                    return image;
                }
            }
            return null;
        }

        public List<VirtualMachineImageMXBean> listVMImageTemplates() {
            ArrayList<VirtualMachineImageMXBean> result = new ArrayList<VirtualMachineImageMXBean>(this.images);
            return result;
        }

        public void removeVMImageTemplate(final VirtualMachineImageMXBean image) throws VMMException {
            DummyServerPool.logger.info("Removing VM image " + image.getName());
            for (VirtualMachineImageMXBean im : this.images) {
                if (im.getUUID().equals(image.getUUID())) {
                    this.images.remove(im);
                    break;
                }
            }
        }

    }

    @Override
    public void deleteImageStore() {
        try {
            int i = this.imageStore.listVMImageTemplates().size();
            while (i > 0) {
                AgentCommon.getMBeanServer().unregisterMBean(this.imageStore.listVMImageTemplates().get(i - 1).getObjectName());
                i--;
            }
        } catch (Throwable e) {
            DummyServerPool.logger.error(e);
        }
        DummyServerPool.logger.info("Removing Image Store");
        try {
            AgentCommon.getMBeanServer().unregisterMBean(this.imageStore.getObjectName());
        } catch (Exception ex) {
            DummyServerPool.logger.error(ex);
        }
        this.imageStore = null;
    }
}
