/**
 * JASMINe VMMapi: JASMINe Virtual Machine Management API
 * Copyright (C) 2010 France Telecom R&D
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: HyperVVirtualMachine.java 7456 2011-01-19 23:05:51Z dangtran $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.vmm.agent.driver.hyperv;

import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.Map;

import javax.management.MBeanNotificationInfo;
import javax.management.Notification;
import javax.management.ObjectName;

import org.apache.log4j.Logger;
import org.ow2.jasmine.vmm.agent.domain.ManagedResource;
import org.ow2.jasmine.vmm.agent.driver.hyperv.HyperVConnection.VM;
import org.ow2.jasmine.vmm.api.BadVMPowerStateException;
import org.ow2.jasmine.vmm.api.HostMXBean;
import org.ow2.jasmine.vmm.api.IllegalOperationException;
import org.ow2.jasmine.vmm.api.InsufficientResourcesException;
import org.ow2.jasmine.vmm.api.NotificationTypes;
import org.ow2.jasmine.vmm.api.ResourceUsage;
import org.ow2.jasmine.vmm.api.VMCustomizationSpec;
import org.ow2.jasmine.vmm.api.VMMException;
import org.ow2.jasmine.vmm.api.VirtualMachineImageMXBean;
import org.ow2.jasmine.vmm.api.VirtualMachineMXBean;

public class HyperVVirtualMachine extends ManagedResource implements VirtualMachineMXBean {
    static Logger logger = Logger.getLogger(HyperVVirtualMachine.class);

    private HyperVHost host;

    private HyperVConnection connection;

    private VM vm;

    private String uuid;

    private String name;

    private String ipAddress;

    private Date startTime;

    public HyperVVirtualMachine(final ObjectName objectName, final HyperVHost host, final HyperVConnection connection,
        final VM vm, final Map<String, String> userData) {
        super(objectName);
        try {
            this.host = host;
            this.connection = connection;
            this.vm = vm;
            this.uuid = this.vm.getUuid();
            this.name = this.vm.getNameLabel();

        } catch (Exception ex) {
            ex.printStackTrace();
            HyperVVirtualMachine.logger.error("Failed to init VM", ex);
        }
    }

    @Override
    public VirtualMachineMXBean cloneVM(final String name, final VMCustomizationSpec custSpec, final boolean sync)
        throws InsufficientResourcesException {
        throw new UnsupportedOperationException("not implemented");
    }

    // TODO
    public boolean canLiveMigrateToHost(final HostMXBean targetHost) {
        throw new UnsupportedOperationException("not implemented");
    }

    public String getUuid() {
        return this.uuid;
    }

    public HostMXBean getHostMBean() {
        return this.host;
    }

    public Date getStartTime() {
        if (this.startTime == null) {

            try {
                String stringStartTime = this.connection.getStartTime(this.host, this.getNameLabel());
                HyperVVMImageStore.logger.debug("GetStartTime of " + this.getNameLabel());

                SimpleDateFormat sdf = new SimpleDateFormat("yyyyMMddhhmmss.'000000-000'");
                this.startTime = sdf.parse(stringStartTime);

            } catch (Exception ex) {
                HyperVVirtualMachine.logger.error("Failed to getStartTime of VM " + this.name, ex);
            }

        }
        HyperVVirtualMachine.logger.debug("VM " + this.name + " Start Time =" + this.startTime);
        return this.startTime;
    }

    public float getCPULoad() {
        float cpuLoad = 0;

        try {
            cpuLoad = this.connection.getCPULoad(this.getNameLabel());
            HyperVVirtualMachine.logger.debug("VM " + this.name + " CPULoad = " + cpuLoad);
        } catch (Exception ex) {
            HyperVVirtualMachine.logger.error("Failed to get CPULoad of VM " + this.name, ex);
        }

        return cpuLoad;
    }

    // TODO
    public float[] getLoadPerVCPU() {
        throw new UnsupportedOperationException("not implemented");
    }

    public long getMemorySizeMB() {
        long memorySize = 0;

        try {
            memorySize = this.connection.getMemorySize(this.getNameLabel());
            HyperVVirtualMachine.logger.debug("VM " + this.name + " MemorySizeMB = " + memorySize);
        } catch (Exception ex) {
            HyperVVirtualMachine.logger.error("Failed to get MemorySizeMB of VM " + this.name, ex);
        }

        return memorySize;
    }

    @Override
    public ResourceUsage getResourceUsage() {
        ResourceUsage ru = new ResourceUsage();
        // XX approximate timestamp
        ru.setSamplingTime(new Date(System.currentTimeMillis()));
        ru.setCpuLoad(this.getCPULoad());
        ru.setMemoryUsedKBytes(this.getMemoryUsedMB() * 1024);
        // TODO net and memory perf
        return ru;
    }

    public void setMemorySizeMB(final long size) {

        try {
            boolean success = this.connection.setMemorySize(this.getNameLabel(), size);
            if (!success) {
                HyperVVirtualMachine.logger.error("Command setMemorySizeMB failed");
                return;
            }
            HyperVVirtualMachine.logger.debug("Command setMemorySizeMB succeed");

        } catch (Exception ex) {
            HyperVVirtualMachine.logger.error("setMemorySizeMB failure", ex);
        }

    }

    public long getMemoryUsedMB() {
        long memoryUsed = 0;

        try {
            memoryUsed = this.connection.getMemoryUsage(this.getNameLabel());
            HyperVVirtualMachine.logger.debug("VM " + this.name + " MemoryUsedMB = " + memoryUsed);
        } catch (Exception ex) {
            HyperVVirtualMachine.logger.error("Failed to get MemoryUsedMB of VM " + this.name, ex);
        }

        return memoryUsed;
    }

    public String getNameLabel() {
        return this.name;
    }

    /*
     * (non-Javadoc)
     * @see
     * org.ow2.jasmine.vmm.api.VirtualMachineMXBean#getUserData(java.lang.String
     * )
     */
    // TODO
    public String getUserData(final String key) {

        throw new UnsupportedOperationException("not implemented");
    }

    public String getConsole() {
        throw new UnsupportedOperationException("not implemented");
    }

    /*
     * (non-Javadoc)
     * @see org.ow2.jasmine.vmm.api.VirtualMachineMXBean#getNumVCPUs()
     */
    public int getNumVCPUs() {
        int numVCpus = 0;

        try {
            numVCpus = this.connection.getNumberOfProcessors(this.getNameLabel());
            HyperVVirtualMachine.logger.debug("VM " + this.name + " Num VCPUs = " + numVCpus);
        } catch (Exception ex) {
            HyperVVirtualMachine.logger.error("Failed to get NumVCPUs of VM " + this.name, ex);
        }

        return numVCpus;
    }

    @Override
    public boolean[][] getCPUAffinity() {
        // TODO
        return null;
    }

    @Override
    public void setCPUAffinity(final boolean[][] affinity) {
        // TODO
    }

    /*
     * (non-Javadoc)
     * @see org.ow2.jasmine.vmm.api.VirtualMachineMXBean#getSchedulingCap()
     */
    public int getSchedulingCap() {
        int schedulingCap = 0;

        try {
            schedulingCap = this.connection.getSchedulingCap(this.getNameLabel());
            HyperVVirtualMachine.logger.debug("VM " + this.name + " SchedulingCap = " + schedulingCap);
        } catch (Exception ex) {
            HyperVVirtualMachine.logger.error("Failed to get SchedulingCap of VM " + this.name, ex);
        }

        return schedulingCap;
    }

    /*
     * (non-Javadoc)
     * @see org.ow2.jasmine.vmm.api.VirtualMachineMXBean#getSchedulingWeight()
     */
    public int getSchedulingWeight() {
        int schedulingWeight = 0;

        try {
            schedulingWeight = this.connection.getSchedulingWeight(this.getNameLabel());
            HyperVVirtualMachine.logger.debug("VM " + this.name + " SchedulingWeight = " + schedulingWeight);
        } catch (Exception ex) {
            HyperVVirtualMachine.logger.error("Failed to get SchedulingWeightp of VM " + this.name, ex);
        }

        return schedulingWeight;
    }

    /*
     * (non-Javadoc)
     * @see org.ow2.jasmine.vmm.api.VirtualMachineMXBean#getState()
     */
    public PowerState getState() {
        return this.getState(null);
    }

    public PowerState getState(HyperVConnection connection) {
        if (connection == null) {
            connection = this.connection;
        }
        try {
            int state = connection.getEnabledState(this.host, this.getNameLabel());
            if (state == HyperVConnection.VmState.Disabled) {
                return PowerState.HALTED;
            } else if (state == HyperVConnection.VmState.Enabled) {
                return PowerState.RUNNING;
            } else if (state == HyperVConnection.VmState.Paused) {
                return PowerState.PAUSED;
            } else if (state == HyperVConnection.VmState.Suspended) {
                return PowerState.SUSPENDED;
            }
        } catch (Exception ex) {
            HyperVVirtualMachine.logger.debug("Failed to get power state", ex);
        }

        return PowerState.UNKNOWN;
    }

    /*
     * (non-Javadoc)
     * @see org.ow2.jasmine.vmm.api.VirtualMachineMXBean#getGuestIpAddress()
     */
    // TODO
    public String getGuestIpAddress() {
        throw new UnsupportedOperationException("not implemented");
    }

    /*
     * (non-Javadoc)
     * @see org.ow2.jasmine.vmm.api.VirtualMachineMXBean#getMacAddress()
     */
    public String getMacAddress() {
        String macAddress = null;

        try {
            macAddress = this.connection.getMacAddress(this.getNameLabel());
            macAddress = macAddress.substring(0, 2) + ':' + macAddress.substring(2, 4) + ':' + macAddress.substring(4, 6) + ':'
                + macAddress.substring(6, 8) + ':' + macAddress.substring(8, 10) + ':' + macAddress.substring(10);
            HyperVVirtualMachine.logger.debug("VM " + this.name + " Mac address = " + macAddress);
        } catch (Exception ex) {
            HyperVVirtualMachine.logger.error("Failed to get MAC address of VM " + this.name, ex);
        }

        return macAddress;
    }

    public void setNumVCPUs(final int numVCPUs) {

        try {
            boolean success = this.connection.setCPUCount(this.getNameLabel(), numVCPUs);
            if (!success) {
                HyperVVirtualMachine.logger.error("Command setNumVCPUs failed");
                return;
            }
            HyperVVirtualMachine.logger.debug("Command setNumVCPUs succeed");

        } catch (Exception ex) {
            HyperVVirtualMachine.logger.error("Failed to setNumVCPUs of VM" + this.name, ex);
        }

    }

    /*
     * (non-Javadoc)
     * @see org.ow2.jasmine.vmm.api.VirtualMachineMXBean#setSchedulingCap(int)
     */
    public void setSchedulingCap(final int schedulingCap) {

        try {
            boolean success = this.connection.setSchedulingCap(this.getNameLabel(), schedulingCap);
            if (!success) {
                HyperVVirtualMachine.logger.error("Command setSchedulingCap failed");
                return;
            }
            HyperVVirtualMachine.logger.debug("Command setSchedulingCap succeed");
        } catch (Exception ex) {
            HyperVVirtualMachine.logger.error("Failed to set sched cap of VM " + this.name, ex);
        }

    }

    /*
     * (non-Javadoc)
     * @see
     * org.ow2.jasmine.vmm.api.VirtualMachineMXBean#setSchedulingWeight(int)
     */
    public void setSchedulingWeight(final int schedulingWeight) {

        try {
            boolean success = this.connection.setSchedulingWeight(this.getNameLabel(), schedulingWeight);
            if (!success) {
                HyperVVirtualMachine.logger.error("Command setSchedulingWeight failed");
                return;
            }
            HyperVVirtualMachine.logger.debug("Command setSchedulingWeight succeed");
        } catch (Exception ex) {
            HyperVVirtualMachine.logger.error("Failed to set sched weight of VM " + this.name, ex);
        }

    }

    /*
     * (non-Javadoc)
     * @see org.ow2.jasmine.vmm.api.VirtualMachineMXBean#suspend()
     */
    public void suspend() throws BadVMPowerStateException {

        try {
            boolean success = this.connection.suspendVM(this.host, this.getNameLabel());
            if (!success) {
                HyperVVirtualMachine.logger.error("Command suspendVM failed");
                return;
            }
            HyperVVirtualMachine.logger.debug("Command suspendVM succeed");

        } catch (BadVMPowerStateException e) {
            throw new BadVMPowerStateException(e.getMessage());
        } catch (Exception ex) {
            HyperVVirtualMachine.logger.error("suspendVM failure", ex);
        }

    }

    /*
     * (non-Javadoc)
     * @see
     * org.ow2.jasmine.vmm.api.VirtualMachineMXBean#migrate(org.ow2.jasmine.
     * vmm.api.HostMXBean, boolean)
     */
    // TODO
    public void migrate(final HostMXBean targetHost, final boolean live) throws IllegalOperationException, VMMException {

        throw new UnsupportedOperationException("not implemented");
    }

    /*
     * (non-Javadoc)
     * @see org.ow2.jasmine.vmm.api.VirtualMachineMXBean#shutdown()
     */
    public void shutdown() throws BadVMPowerStateException {

        try {
            boolean success = this.connection.shutdownVM(this.host, this.getNameLabel());
            if (!success) {
                HyperVVirtualMachine.logger.error("Command shutdownVM failed");
                return;
            }
            HyperVVirtualMachine.logger.debug("Command shutdownVM succeed");

        } catch (BadVMPowerStateException e) {
            throw new BadVMPowerStateException(e.getMessage());
        } catch (Exception ex) {
            HyperVVirtualMachine.logger.error("shutdownVM failure", ex);
        }

    }

    /*
     * (non-Javadoc)
     * @see org.ow2.jasmine.vmm.api.VirtualMachineMXBean#start()
     */
    public void start() throws BadVMPowerStateException {

        try {
            boolean success = this.connection.startVM(this.host, this.getNameLabel());
            if (!success) {
                HyperVVirtualMachine.logger.error("Command startVM failed");
                return;
            }
            HyperVVirtualMachine.logger.debug("Command startVM succeed");

        } catch (BadVMPowerStateException e) {
            throw new BadVMPowerStateException(e.getMessage());
        } catch (Exception ex) {
            HyperVVirtualMachine.logger.error("Start failure", ex);
        }

    }

    private PowerState lastState = PowerState.UNKNOWN;

    synchronized void onVMStateChanged(final VirtualMachineMXBean.PowerState state) {
        if (state != this.lastState) {
            this.emitNotification(NotificationTypes.VM_STATE_CHANGE, state.toString(), null);
            HyperVVirtualMachine.logger.debug("VM " + this.getNameLabel() + " new power state: " + state.toString());
            this.lastState = state;
        }

    }

    /*
     * (non-Javadoc)
     * @see org.ow2.jasmine.vmm.api.VirtualMachineMXBean#reboot()
     */
    public void reboot() throws BadVMPowerStateException {

        try {
            boolean success = this.connection.rebootVM(this.host, this.getNameLabel());
            if (!success) {
                HyperVVirtualMachine.logger.error("Command rebootVM failed");
                return;
            }
            HyperVVirtualMachine.logger.debug("VM " + this.name + " rebooted");

        } catch (BadVMPowerStateException e) {
            throw new BadVMPowerStateException(e.getMessage());
        } catch (Exception ex) {
            HyperVVirtualMachine.logger.error("Failed to reboot VM", ex);
        }
    }

    /*
     * (non-Javadoc)
     * @see org.ow2.jasmine.vmm.api.VirtualMachineMXBean#resume()
     */
    public void resume() throws BadVMPowerStateException {
        this.start();
    }

    @Override
    public void pause() throws BadVMPowerStateException {
        throw new UnsupportedOperationException();
    }

    @Override
    public void unpause() throws BadVMPowerStateException {
        throw new UnsupportedOperationException();
    }

    /*
     * (non-Javadoc)
     * @see org.ow2.jasmine.vmm.api.VirtualMachineMXBean#destroy()
     */
    public void destroy() {

        try {
            try {
                this.shutdown();
            } catch (BadVMPowerStateException ex) {
            }
            boolean success = this.connection.destroyVM(this.host, this.getNameLabel());

            if (!success) {
                HyperVVirtualMachine.logger.error("DestroyVM failed");
            }
            HyperVVirtualMachine.logger.info("VM " + this.name + " destroyed");

        } catch (Exception ex) {
            HyperVVirtualMachine.logger.error("Command destroyVM failed", ex);
        }

        try {
            boolean success = this.connection.hostDeleteFile(this.host.getVMFolderpath(), this.getNameLabel().concat(".vhd"));

            if (!success) {
                HyperVVirtualMachine.logger.error("Delete vm file failed");
            }
            HyperVVirtualMachine.logger.info("VM file " + this.name + " deleted");

        } catch (Exception ex) {
            HyperVVirtualMachine.logger.error("Command deleteFile failed", ex);
        }

        try {
            this.host.onVMDestroy(this);
        } catch (Exception ex) {
            HyperVVirtualMachine.logger.error("Failed to unregister VM MBean", ex);
        }
    }

    @Override
    public MBeanNotificationInfo[] getNotificationInfo() {
        return new MBeanNotificationInfo[] {new MBeanNotificationInfo(new String[] {NotificationTypes.VM_STATE_CHANGE,
            NotificationTypes.VM_MIGRATION_ABORT, NotificationTypes.VM_MIGRATION_START, NotificationTypes.VM_MIGRATION,
            NotificationTypes.LOG, NotificationTypes.ERROR}, Notification.class.getName(), "VM event")};

    }

    @Override
    public VirtualMachineImageMXBean makeTemplate(final String name, final String description,
        final Map<String, String> metadata) throws InsufficientResourcesException, IllegalOperationException,
        BadVMPowerStateException, VMMException {
        // shutdown VM to make sure disk image is a proper state
        PowerState state = this.getState();
        this.shutdown();

        String vmImageID = name + "-" + System.currentTimeMillis();
        boolean success = false;
        try {
            success = this.connection.hostCopyFile(this.host.getVMFolderpath().concat(this.getNameLabel().concat(".vhd")),
                this.host.getVMTemplateFolderpath().concat(vmImageID.concat(".vhd")));

        } catch (Exception ex) {
            HyperVHost.logger.error("CopyFile failed ", ex);
            if (state.equals(PowerState.RUNNING)) {
                this.start();
            } else if (state.equals(PowerState.SUSPENDED)) {
                this.suspend();
            }
            throw new VMMException("Template creation failed: WMI connection failure", ex);
        }
        if (!success) {
            HyperVHost.logger.error("Command copyFile failed");
            if (state.equals(PowerState.RUNNING)) {
                this.start();
            } else if (state.equals(PowerState.SUSPENDED)) {
                this.suspend();
            }
            throw new VMMException("Template creation failed");
        }
        ((HyperVVMImageStore) this.host.getVMImageStore()).sync();

        if (state.equals(PowerState.RUNNING)) {
            this.start();
        } else if (state.equals(PowerState.SUSPENDED)) {
            this.suspend();
        }
        return ((HyperVVMImageStore) this.host.getVMImageStore()).lookUpByUUID(vmImageID);
    }
}
