/**
 * JASMINe VMMapi: JASMINe Virtual Machine Management API
 * Copyright (C) 2009-2010 France Telecom R&D
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: LibvirtHost.java 7883 2011-03-30 16:54:29Z dangtran $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.vmm.agent.driver.libvirt;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

import javax.management.MBeanNotificationInfo;
import javax.management.Notification;
import javax.management.NotificationEmitter;
import javax.management.ObjectName;

import org.apache.log4j.Logger;
import org.libvirt.Connect;
import org.libvirt.Domain;
import org.libvirt.LibvirtException;
import org.libvirt.NodeInfo;
import org.ow2.jasmine.vmm.agent.domain.AbstractHost;
import org.ow2.jasmine.vmm.agent.driver.libvirt.ImageCatalog.VirtualMachineImage;
import org.ow2.jasmine.vmm.agent.driver.util.RemoteExec;
import org.ow2.jasmine.vmm.agent.jmx.MBeanObjectNamer;
import org.ow2.jasmine.vmm.agent.main.AgentCommon;
import org.ow2.jasmine.vmm.api.BadVMPowerStateException;
import org.ow2.jasmine.vmm.api.InsufficientResourcesException;
import org.ow2.jasmine.vmm.api.InvalidVMConfigException;
import org.ow2.jasmine.vmm.api.NotificationTypes;
import org.ow2.jasmine.vmm.api.ResourceUsage;
import org.ow2.jasmine.vmm.api.ServerPoolMXBean;
import org.ow2.jasmine.vmm.api.VMConfigSpec;
import org.ow2.jasmine.vmm.api.VMCustomizationSpec;
import org.ow2.jasmine.vmm.api.VMMException;
import org.ow2.jasmine.vmm.api.VirtualMachineImageStoreMXBean;
import org.ow2.jasmine.vmm.api.VirtualMachineMXBean;
import org.ow2.jasmine.vmm.api.VirtualMachineMXBean.PowerState;
import org.ow2.jasmine.vmm.api.VnicIPSettings;
import org.ow2.jasmine.vmm.api.VnicIPSettings.IpAssignmentMode;

public class LibvirtHost extends AbstractHost implements NotificationEmitter {
    static Logger logger = Logger.getLogger(LibvirtHost.class);

    // the thread pool size determines the maximum number of
    // simultaneous VM creation on the same host
    private static final int MAX_PARALEL_VM_CREATION_PER_HOST = 3;

    private static final int THREADPOOL_SIZE = LibvirtHost.MAX_PARALEL_VM_CREATION_PER_HOST;

    private static ExecutorService executorService = Executors.newFixedThreadPool(LibvirtHost.THREADPOOL_SIZE);

    private String uri;

    private List<LibvirtVirtualMachine> vmList = new CopyOnWriteArrayList<LibvirtVirtualMachine>();

    LibvirtServerPool serverPool;

    private LibvirtPerfCollector perfCollectionTask;

    private String hostName;

    private HostPowerState hostPowerState = HostPowerState.HALTED;

    private Connect connection;

    private int numCPU = -1;

    private long cpuFrequencyMhz = -1;

    private Map<String, String> hostAttributes;

    private RemoteExec.SshAuthInfo authInfo;

    private HostSynchronizer hostSynchronizer;

    public static LibvirtHost newHost(final LibvirtServerPool serverPool, final ObjectName objectName, final String hostName,
        final String uri, final RemoteExec.SshAuthInfo authInfo, final Map<String, String> hostAttributes,
        final Boolean isNotStop) {

        try {
            return new LibvirtHost(serverPool, uri, objectName, hostName, authInfo, hostAttributes, isNotStop);
        } catch (LibvirtException ex) {
            LibvirtHost.logger.error("Failed to establish libvirt connection with uri " + uri, ex);
            return null;
        }
    }

    private LibvirtHost(final LibvirtServerPool serverPool, final String uri, final ObjectName objectName,
        final String hostName, final RemoteExec.SshAuthInfo authInfo, final Map<String, String> hostAttributes,
        final Boolean isNotStop) throws LibvirtException {
        super(objectName);
        this.hostPowerState = HostPowerState.HALTED;
        this.uri = uri;
        this.serverPool = serverPool;
        LibvirtHost.logger.debug("Connecting to URI " + this.uri);
        if (isNotStop) {
            try {
                this.connection = this.getLibvirtAPIConnection();
                this.hostPowerState = HostPowerState.RUNNING;
            } catch (LibvirtException ex) {
                LibvirtHost.logger.warn("Host " + hostName + " not responding");
                this.hostPowerState = HostPowerState.NOT_RESPONDING;
            }
        }
        this.hostName = hostName;
        this.authInfo = authInfo;
        this.hostAttributes = new HashMap<String, String>(hostAttributes);
        this.hostSynchronizer = new HostSynchronizer(serverPool.getSyncPeriodMillis());
        if (this.hostPowerState == HostPowerState.RUNNING || this.hostPowerState == HostPowerState.NOT_RESPONDING) {
            this.hostSynchronizer.start();
        }
        if (this.hostPowerState == HostPowerState.RUNNING) {
            this.startPerfMonitor();
        }
    }

    public boolean isConnectionLost() {
        return this.hostPowerState == HostPowerState.NOT_RESPONDING;
    }

    public RemoteExec.SshAuthInfo getSshAuthInfo() {
        return this.authInfo;
    }

    Connect getLibvirtAPIConnection() throws LibvirtException {
        if (this.connection == null) {
            this.connection = new Connect(this.uri);
        }
        return this.connection;
    }

    List<LibvirtVirtualMachine> getVMs() {
        return this.vmList;
    }

    LibvirtVirtualMachine lookUpVMByName(final String name) {
        for (LibvirtVirtualMachine vm : this.vmList) {
            if (vm.getNameLabel().equals(name)) {
                return vm;
            }
        }
        return null;
    }

    LibvirtVirtualMachine lookUpVMByUUID(final String uuid) {
        for (LibvirtVirtualMachine vm : this.vmList) {
            if (vm.getUuid().equals(uuid)) {
                return vm;
            }
        }
        return null;
    }

    private void addVM(final LibvirtVirtualMachine vm) {
        this.vmList.add(vm);
    }

    private boolean removeVM(final LibvirtVirtualMachine vm) {
        return this.vmList.remove(vm);
    }

    public ServerPoolMXBean getServerPool() {
        return this.serverPool;
    }

    public void postMigrateVM(final LibvirtVirtualMachine vm, final LibvirtHost newHost) {
        this.removeVM(vm);
        newHost.addVM(vm);
    }

    public void removeVMFromInventory(final LibvirtVirtualMachine vm) {
        try {
            AgentCommon.getMBeanServer().unregisterMBean(vm.getObjectName());
        } catch (Exception ex) {
            LibvirtHost.logger.error("Cannot unregister VM MBean", ex);
        }
        if (this.removeVM(vm)) {
            this.emitNotification(NotificationTypes.VM_INVENTORY_DEL, "Deleted", vm.getObjectName());
            LibvirtHost.logger.info("deleted VM " + vm.getObjectName());
        }
    }

    private Set<PerfMetric> currentMonitoredMetrics;

    private long currentMonitoringPeriod;

    public void configurePerfMonitor(final Set<PerfMetric> metricsOfInterest, long periodMillis) {
        this.stopPerfMonitor();
        if (periodMillis < 1000) {
            periodMillis = 1000;
        }
        if (metricsOfInterest.contains(PerfMetric.VM_CPU_LOAD)) {
            this.currentMonitoringPeriod = periodMillis;
            this.currentMonitoredMetrics = metricsOfInterest;
            try {
                this.startPerfMonitor();
            } catch (LibvirtException e) {
                LibvirtHost.logger.error("Failed to start perfmonitor", e);
            }
        }
    }

    String getIPAddress(String macAddress) throws Exception {
        macAddress = macAddress.toUpperCase();

        LibvirtHost.logger.debug("Determining IP address from MAC address " + macAddress + " ...");

        RemoteExec.Result result = null;
        String command = "getIPfromMAC.sh " + macAddress;

        LibvirtHost.logger.debug("Launching command: " + command);

        try {
            result = RemoteExec.commandAsRoot(this.hostName, this.authInfo, command);
        } catch (RemoteExec.SshException ex) {
            LibvirtHost.logger.debug("SSH failure", ex);
            throw new Exception(ex);
        }

        if (result.exitCode != 0) {
            LibvirtHost.logger.error("Cannot get IP from MAC " + result.output);
            throw new Exception(result.error);
        }

        String ipAddress = result.output;
        if (ipAddress != null && ipAddress.length() > 0 && ipAddress.endsWith("\n")) {
            ipAddress = ipAddress.substring(0, ipAddress.length() - 1);
        }
        LibvirtHost.logger.debug("Mac-to-IP " + macAddress + " -> " + ipAddress);
        return ipAddress;
    }

    private synchronized void startPerfMonitor() throws LibvirtException {
        if (this.perfCollectionTask == null) {
            this.perfCollectionTask = new LibvirtPerfCollector(this);
            this.perfCollectionTask.start();
        }
    }

    private synchronized void stopPerfMonitor() {
        if (this.perfCollectionTask != null) {
            this.perfCollectionTask.stop();
            this.perfCollectionTask = null;
        }
    }

    @Override
    public ObjectName getObjectName() {
        return this.objectName;
    }

    private HashMap<String, String> hypervisorInfo;

    public Map<String, String> getHypervisorInfo() {
        if (this.hypervisorInfo == null) {
            this.hypervisorInfo = new HashMap<String, String>();
            try {
                this.hypervisorInfo.put("name", this.connection.getType());
                long v = this.connection.getVersion();
                long major = v / 1000000;
                long minor = (v - major * 1000000) / 1000;
                this.hypervisorInfo.put("version", "" + major + "." + minor);
            } catch (LibvirtException ex) {
                LibvirtHost.logger.error("Failed to get hypervisor info", ex);
            }
        }
        return this.hypervisorInfo;
    }

    private HashMap<String, String> cpuInfo;

    public Map<String, String> getCPUInfo() {
        if (this.cpuInfo == null) {
            this.cpuInfo = new HashMap<String, String>();
            NodeInfo nodeInfo;
            try {
                nodeInfo = this.connection.nodeInfo();
            } catch (LibvirtException e) {
                e.printStackTrace();
                return this.cpuInfo;
            }
            this.cpuInfo.put("model", new String(nodeInfo.model));
            this.cpuInfo.put("speedMHz", Integer.toString(nodeInfo.mhz));
            this.numCPU = nodeInfo.cpus;
        }
        return this.cpuInfo;
    }

    public int getNumCPU() {
        if (this.numCPU == -1) {
            this.getCPUInfo();
        }
        return this.numCPU;
    }

    public String getHostName() {
        return this.hostName;
    }

    public VirtualMachineImageStoreMXBean getVMImageStore() {
        return this.serverPool.getVMImageStore();
    }

    public long getFreeMemoryMB() {
        try {
            if (this.serverPool.getHypervisor().equals("xen")) {
                return this.getLibvirtFreeMemoryMB();
            } else if (this.serverPool.getHypervisor().equals("kvm")) {
                return this.getLinuxHostFreeMemoryMB();
            }
        } catch (Exception ex) {
            LibvirtHost.logger.error("getFreeMemoryMB", ex);
        }
        return -1;
    }

    private long getLibvirtFreeMemoryMB() throws Exception {
        long freeMemKB = -1;
        freeMemKB = this.connection.getFreeMemory();
        return freeMemKB / (1024 * 1024);
    }

    private long getLinuxHostFreeMemoryMB() throws Exception {
        RemoteExec.Result result;
        String command = "free -m | sed '3!d' | awk '{print $4}'";
        result = RemoteExec.commandAsRoot(this.hostName, this.authInfo, command);
        if (result.output.endsWith("\n")) {
            result.output = result.output.substring(0, result.output.length() - 1);
        }
        long osFreeMemMB = Long.parseLong(result.output);
        long vmTotalMemoryMB = 0;
        for (VirtualMachineMXBean vm : this.vmList) {
            vmTotalMemoryMB += vm.getMemorySizeMB();
        }
        long vmFreeMemMB = this.getTotalMemoryMB() - vmTotalMemoryMB;
        return Math.min(osFreeMemMB, vmFreeMemMB);
    }

    public void sync() {
        this.hostSynchronizer.sync();
    }

    private synchronized boolean doSync(final Connect connection) {
        try {
            String vmNames[] = connection.listDefinedDomains();
            for (String vmName : vmNames) {
                Domain domain;
                try {
                    domain = connection.domainLookupByName(vmName);
                } catch (LibvirtException ex) {
                    continue;
                }
                String uuid = domain.getUUIDString();
                ObjectName name = MBeanObjectNamer.makeVirtualMachineName(this.serverPool.getPath() + "/" + vmName, uuid);
                LibvirtVirtualMachine vm = this.lookUpVMByName(vmName);
                if (AgentCommon.getMBeanServer().isRegistered(name)) {
                    if (vm != null) {
                        vm.setSynchronized(true);
                    }
                    continue;
                }
                if (vm == null) {
                    vm = new LibvirtVirtualMachine(name, this, domain, null);
                    this.addVM(vm);
                    AgentCommon.getMBeanServer().registerMBean(vm, name);
                    this.emitNotification(NotificationTypes.VM_ADD, "Created", name);
                }
            }

            int domIDs[] = connection.listDomains();
            for (int domID : domIDs) {
                Domain domain;
                try {
                    domain = connection.domainLookupByID(domID);
                } catch (LibvirtException ex) {
                    continue;
                }
                String uuid = domain.getUUIDString();
                String vmName = domain.getName();
                if (vmName.equals("Domain-0")) {
                    // uuid = this.hostName; // XXX
                    continue; // Exclude Domain-0 from the vm list
                }
                ObjectName name = MBeanObjectNamer.makeVirtualMachineName(this.serverPool.getPath() + "/" + vmName, uuid);
                LibvirtVirtualMachine vm = this.lookUpVMByName(vmName);
                if (AgentCommon.getMBeanServer().isRegistered(name)) {
                    if (vm != null) {
                        vm.setSynchronized(true);
                    }
                    continue;
                }
                if (vm == null) {
                    vm = new LibvirtVirtualMachine(name, this, domain, null);
                    this.addVM(vm);
                    AgentCommon.getMBeanServer().registerMBean(vm, name);
                    this.emitNotification(NotificationTypes.VM_ADD, "Created", name);
                }
            }

            for (LibvirtVirtualMachine vm : this.vmList) {
                try {
                    if (vm.isMigrating()) {
                        continue;
                    }
                    if (!vm.isSynchronized()) {
                        if (this.removeVM(vm)) {
                            this.emitNotification(NotificationTypes.VM_DEL, "Destroyed", vm.getObjectName());
                            AgentCommon.getMBeanServer().unregisterMBean(vm.getObjectName());
                        }
                    } else {
                        try {
                            vm.getState();
                        } catch (VMMException ignore) {
                        }
                    }
                } finally {
                    vm.setSynchronized(false);
                }
            }

        } catch (LibvirtException ex) {
            return false;
        } catch (Exception ex) {
            LibvirtHost.logger.error("error while syncing", ex);
        }
        return true;
    }

    public List<VirtualMachineMXBean> getResidentVMs() {
        List<VirtualMachineMXBean> result = new ArrayList<VirtualMachineMXBean>(this.vmList);
        return result;
    }

    public VirtualMachineMXBean createVM(final VMConfigSpec vmSpecIn, final boolean sync) throws InvalidVMConfigException,
        VMMException {
        final VMConfigSpec vmSpec = new VMConfigSpec(vmSpecIn);
        LibvirtHost.logger.info("Creating new VM [name=" + vmSpec.getName() + ",memorySizeMB=" + vmSpec.getMemorySizeMB()
            + ",diskSize=" + vmSpec.getDiskSizeMB() + ",numVCPU=" + vmSpec.getNumVCPU() + "]");

        final VirtualMachineImage template = (VirtualMachineImage) this.serverPool.getVMImageStore().lookUpByUUID(
            vmSpec.getVmImageUUID());
        if (template == null) {
            throw new InvalidVMConfigException("Invalid VM Image UUID");
        }

        final String vmName = vmSpec.getName();

        if (!sync) {
            LibvirtHost.executorService.execute(new Runnable() {
                public void run() {
                    try {
                        LibvirtHost.this.createVM2(vmSpec, vmName, template);
                    } catch (Exception ex) {
                    }
                }
            });
            return null;
        } else {
            return this.createVM2(vmSpec, vmName, template);
        }
    }

    private VirtualMachineMXBean createVM2(final VMConfigSpec vmSpec, final String vmName, final VirtualMachineImage template)
        throws InvalidVMConfigException, VMMException {

        template.newVM(vmSpec, vmName, this);

        synchronized (this) {
            String vmUUID = null;
            ObjectName name = null;
            LibvirtVirtualMachine vm = null;
            Domain domain = null;

            try {
                domain = this.connection.domainLookupByName(vmName);
                vmUUID = domain.getUUIDString();
                vm = this.lookUpVMByUUID(vmUUID);
                if (vm != null) {
                    return vm;
                }
                name = MBeanObjectNamer.makeVirtualMachineName(this.serverPool.getPath() + "/" + vmName, vmUUID);
                HashMap<String, String> map = new HashMap<String, String>();
                map.put("imageID", vmSpec.getVmImageUUID());
                domain = this.connection.domainLookupByName(vmName);
                vm = new LibvirtVirtualMachine(name, this, domain, map);
                this.addVM(vm);

                AgentCommon.getMBeanServer().registerMBean(vm, name);
            } catch (Exception ex) {
                LibvirtHost.logger.debug("Host " + this.hostName, ex);
                throw new VMMException(ex);
            }

            this.emitNotification(NotificationTypes.VM_ADD, "Created", name);

            return vm;
        }
    }

    public VirtualMachineMXBean cloneVM(final String targetVmName, final String clonedVmName,
        final VMCustomizationSpec custSpec, final boolean sync) throws InsufficientResourcesException, VMMException {
        String command = "cloneVM --src " + clonedVmName + " --name " + targetVmName + " --force ";
        if (custSpec != null) {
            if (custSpec.getGuestOsHostName() != null && !custSpec.getGuestOsHostName().equals("")) {
                command += " --hostname " + custSpec.getGuestOsHostName();
            }
            // XXX configure only first NIC
            if (custSpec.getVnicIpSettingsList() != null && custSpec.getVnicIpSettingsList().size() > 0) {
                VnicIPSettings nic0Settings = custSpec.getVnicIpSettingsList().get(0);
                if (nic0Settings.getIpAssignmentMode() == IpAssignmentMode.DHCP) {
                    command += " --net eth0/dhcp";
                } else if (nic0Settings.getIpAssignmentMode() == IpAssignmentMode.FIXED) {
                    command += " --net eth0/static/" + nic0Settings.getIpAddress() + "/" + nic0Settings.getSubnetMask() + "/"
                        + nic0Settings.getGateway();
                }
            }
        }
        RemoteExec.Result result;
        try {
            result = RemoteExec.commandAsRoot(this.getHostName(), this.getSshAuthInfo(), command);
        } catch (RemoteExec.SshException ex) {
            throw new VMMException("VM cloning failed: SSH connection failure", ex);
        }
        if (result.exitCode != 0) {
            throw new VMMException("VM cloning failed: " + result.error);
        }

        Domain newVM = null;
        String vmUUID = "";

        try {
            newVM = this.connection.domainLookupByName(clonedVmName);
            vmUUID = newVM.getUUIDString();
        } catch (LibvirtException ex) {
            LibvirtHost.logger.error("Failed to look up VM " + clonedVmName);
            throw new VMMException("VM cloning failure");
        }

        synchronized (this) {
            ObjectName name = null;
            LibvirtVirtualMachine vm = null;
            vm = this.lookUpVMByUUID(vmUUID);
            if (vm != null) {
                return vm;
            }
            try {
                name = MBeanObjectNamer.makeVirtualMachineName(this.serverPool.getPath() + "/" + clonedVmName, vmUUID);
                HashMap<String, String> map = new HashMap<String, String>();
                vm = new LibvirtVirtualMachine(name, this, newVM, map);
                this.addVM(vm);

                AgentCommon.getMBeanServer().registerMBean(vm, name);
            } catch (Exception ex) {
                LibvirtHost.logger.debug("Host " + this.hostName, ex);
                throw new VMMException(ex);
            }

            this.emitNotification(NotificationTypes.VM_ADD, "Created", name);

            return vm;
        }
    }

    public long getTotalMemoryMB() {
        NodeInfo nodeInfo;
        try {
            nodeInfo = this.connection.nodeInfo();
        } catch (LibvirtException e) {
            e.printStackTrace();
            return -1;
        }
        return nodeInfo.memory / 1024;
    }

    // XX unsupported
    public float[] getLoadPerCPU() {
        float[] result = new float[0];
        return result;
    }

    public float getCPULoad() {
        float aggregatedCPULoad = 0;
        for (LibvirtVirtualMachine vm : this.vmList) {
            aggregatedCPULoad += vm.getResourceUsage().getCpuLoad();
        }
        return aggregatedCPULoad;
    }

    @Override
    public MBeanNotificationInfo[] getNotificationInfo() {
        return new MBeanNotificationInfo[] {new MBeanNotificationInfo(new String[] {NotificationTypes.VM_STATE_CHANGE,
            NotificationTypes.VM_ADD, NotificationTypes.VM_INVENTORY_DEL, NotificationTypes.VM_DEL, NotificationTypes.ERROR,
            NotificationTypes.LOG}, Notification.class.getName(), "Host event")};
    }

    public Map<String, ResourceUsage> getVMResourceUsage() {
        HashMap<String, ResourceUsage> result = new HashMap<String, ResourceUsage>();
        for (LibvirtVirtualMachine vm : this.vmList) {
            result.put(vm.getNameLabel(), vm.getResourceUsage());
        }
        return result;
    }

    public Map<String, Float> getVMCPULoads() {
        HashMap<String, Float> cpuLoads = new HashMap<String, Float>();
        for (LibvirtVirtualMachine vm : this.vmList) {
            cpuLoads.put(vm.getNameLabel(), vm.getResourceUsage().getCpuLoad());
        }
        return cpuLoads;
    }

    private class HostSynchronizer implements Runnable {
        private long periodMillis;

        private volatile boolean stopRequested;

        private Thread thread;

        private Connect connection;

        HostSynchronizer(final long periodMillis) {
            this.periodMillis = periodMillis;
        }

        public void requestStop() {
            this.stopRequested = true;
        }

        void start() {
            this.thread = new Thread(this);
            this.stopRequested = false;
            this.thread.start();
        }

        synchronized void sync() {
            this.notify();
        }

        @Override
        public void run() {
            while (!this.stopRequested) {
                try {
                    this.connection = LibvirtHost.this.getLibvirtAPIConnection();
                    if (LibvirtHost.this.hostPowerState == HostPowerState.NOT_RESPONDING) {
                        LibvirtHost.this.setHostStatusFromNonRespondingToRunning();
                    } else {
                        if (LibvirtHost.this.doSync(this.connection) == false) {
                            LibvirtHost.this.setHostStatusFromRunningToNonResponding();
                            this.connection = null;
                        }
                    }
                } catch (LibvirtException ex) {
                    // LibvirtHost.logger.error("Failed to create Libvirt connection with "
                    // + LibvirtHost.this.uri, ex);
                    if (LibvirtHost.this.hostPowerState == HostPowerState.RUNNING) {
                        LibvirtHost.this.setHostStatusFromRunningToNonResponding();
                        this.connection = null;
                    }
                }

                synchronized (this) {
                    try {
                        this.wait(this.periodMillis);
                    } catch (InterruptedException ex) {
                    }
                }
            }
        }
    }

    @Override
    public Map<String, String> getAttributes() {
        return this.hostAttributes;
    }

    @Override
    public void removeFromInventory() {
        this.stopPerfMonitor();
        this.hostSynchronizer.requestStop();
        while (this.getResidentVMs().size() > 0) {
            this.removeVMFromInventory((LibvirtVirtualMachine) this.getResidentVMs().get(0));
        }
    }

    private void setHostStatusFromRunningToNonResponding() {
        this.stopPerfMonitor();
        while (this.getResidentVMs().size() > 0) {
            this.removeVMFromInventory((LibvirtVirtualMachine) this.getResidentVMs().get(0));
        }
        this.hostPowerState = HostPowerState.NOT_RESPONDING;
        this.connection = null;
        LibvirtHost.logger.warn("Host " + LibvirtHost.this.hostName + " not responding");
        this.emitNotification(NotificationTypes.HOST_STATE_CHANGE, this.hostPowerState.toString(), null);
    }

    private void setHostStatusFromNonRespondingToRunning() {
        try {
            this.startPerfMonitor();
        } catch (LibvirtException ex) {
            LibvirtHost.logger.error(ex);
        }
        this.hostPowerState = HostPowerState.RUNNING;
        this.emitNotification(NotificationTypes.HOST_STATE_CHANGE, this.hostPowerState.toString(), null);
    }

    @Override
    public void start() {
        try {
            this.connection = this.getLibvirtAPIConnection();
            this.hostSynchronizer.start();
            this.startPerfMonitor();
        } catch (LibvirtException e) {
            // TODO Auto-generated catch block
            e.printStackTrace();
        }
        this.hostPowerState = HostPowerState.RUNNING;
        this.emitNotification(NotificationTypes.HOST_STATE_CHANGE, this.hostPowerState.toString(), null);
    }

    @Override
    public void stop() {

        this.stopPerfMonitor();
        this.hostSynchronizer.requestStop();
        // this.connection.close();
        // this.connection = null;

        while (this.getResidentVMs().size() > 0) {
            try {
                while (this.getResidentVMs().get(0).getState() == PowerState.RUNNING) {
                    this.getResidentVMs().get(0).shutdown();
                }
                this.removeVMFromInventory((LibvirtVirtualMachine) this.getResidentVMs().get(0));
            } catch (BadVMPowerStateException e) {
                // TODO Auto-generated catch block
                e.printStackTrace();
            } catch (VMMException e) {
                // TODO Auto-generated catch block
                e.printStackTrace();

            }
        }

        this.hostPowerState = HostPowerState.HALTED;
        this.emitNotification(NotificationTypes.HOST_STATE_CHANGE, this.hostPowerState.toString(), null);
    }

    /*
     * (non-Javadoc)
     * @see org.ow2.jasmine.vmm.api.HostMXBean#getHostState()
     */
    public HostPowerState getHostState() throws VMMException {
        return this.hostPowerState;
    }

}
