/**
 * JASMINe VMMapi: JASMINe Virtual Machine Management API
 * Copyright (C) 2009-2010 France Telecom R&D
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: LibvirtServerPool.java 7811 2011-03-09 14:46:02Z dangtran $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.vmm.agent.driver.libvirt;

import java.util.Map;

import javax.management.ObjectName;

import org.apache.log4j.Logger;
import org.ow2.jasmine.vmm.agent.domain.ServerPool;
import org.ow2.jasmine.vmm.agent.driver.util.RemoteExec;
import org.ow2.jasmine.vmm.agent.jmx.MBeanObjectNamer;
import org.ow2.jasmine.vmm.agent.main.AgentCommon;
import org.ow2.jasmine.vmm.api.HostMXBean;
import org.ow2.jasmine.vmm.api.HostMXBean.HostPowerState;
import org.ow2.jasmine.vmm.api.VMMException;
import org.ow2.jasmine.vmm.api.VirtualMachineImageStoreMXBean;

public class LibvirtServerPool extends ServerPool {
    static Logger logger = Logger.getLogger(LibvirtServerPool.class);

    static long DEFAULT_HOST_SYNC_PERIOD_MILLIS = 10000;

    private RemoteExec.SshAuthInfo defaultAuthInfo;

    private String vmImageStoreDir, diskStoreDir;

    private String vmImageStoreHostname;

    private String hypervisor;

    private VirtualMachineImageStoreMXBean imageStore;

    private long syncPeriodMillis = LibvirtServerPool.DEFAULT_HOST_SYNC_PERIOD_MILLIS;

    public LibvirtServerPool(final String name, final ObjectName objectName, final Map<String, String> attributes)
        throws IllegalArgumentException {
        super(name, objectName, attributes);
        String sshPassword = attributes.get("sshRootPassword");
        String sshPrivateKeyFile = attributes.get("sshPrivateKeyFile");
        if (sshPassword != null || sshPrivateKeyFile != null) {
            this.defaultAuthInfo = new RemoteExec.SshAuthInfo(sshPassword, sshPrivateKeyFile);
        }
        this.vmImageStoreDir = attributes.get("sharedImageStore");
        if (this.vmImageStoreDir == null) {
            throw new IllegalArgumentException("Missing parameter: sharedImageStore");
        }
        if (this.vmImageStoreDir != null) {
            this.vmImageStoreHostname = attributes.get("imageStoreHost");
        }
        this.diskStoreDir = attributes.get("sharedDiskStore");
        if (this.diskStoreDir == null) {
            throw new IllegalArgumentException("Missing parameter: sharedDiskStore");
        }
        this.hypervisor = attributes.get("hypervisor");
        if (this.hypervisor == null) {
            throw new IllegalArgumentException("Missing parameter: hypervisor");
        }
        String period = attributes.get("syncPeriodMillis");
        if (period != null) {
            try {
                this.syncPeriodMillis = Integer.parseInt(period);
            } catch (NumberFormatException ex) {
                throw new IllegalArgumentException("Wrong value for parameter syncPeriodMillis");
            }
        }
    }

    @Override
    public String getHypervisor() {
        return this.hypervisor;
    }

    @Override
    public HostMXBean newHost(final String hostName, final Map<String, String> props) throws VMMException {
        try {
            for (HostMXBean host : this.hosts) {
                if (host.getHostName().equals(hostName)) {
                    return host;
                }
            }
            ObjectName mbeanObjectName = MBeanObjectNamer.makeHostName(this.getPath() + "/" + hostName, hostName);
            RemoteExec.SshAuthInfo hostAuthInfo;
            if (props.get("sshRootPassword") != null) {
                hostAuthInfo = new RemoteExec.SshAuthInfo(props.get("sshRootPassword"), null);
            } else if (props.get("sshPrivateKeyFile") != null) {
                hostAuthInfo = new RemoteExec.SshAuthInfo(null, props.get("sshPrivateKeyFile"));
            } else {
                hostAuthInfo = this.defaultAuthInfo;
            }
            if (hostAuthInfo == null) {
                throw new IllegalArgumentException("Missing SSH authentification info for host " + hostName);
            }

            String url = props.get("url");
            if (url == null) {
                throw new IllegalArgumentException("Missing parameter: url");
            }

            LibvirtHost host = LibvirtHost.newHost(this, mbeanObjectName, hostName, url, hostAuthInfo, props, true);
            if (host != null) {
                AgentCommon.getMBeanServer().registerMBean(host, mbeanObjectName);
                this.hosts.add(host);
                LibvirtServerPool.logger.info("In server pool " + this.servername + ": added Host MBean " + mbeanObjectName);
                if (this.imageStore == null && host.getHostState() == HostPowerState.RUNNING) {
                    ObjectName defaultDataStoreName = MBeanObjectNamer.makeVMImageStoreName(hostName + ".default");
                    this.imageStore = new ImageCatalog(defaultDataStoreName, this.vmImageStoreDir, hostName,
                        this.defaultAuthInfo);
                    AgentCommon.getMBeanServer().registerMBean(this.imageStore, defaultDataStoreName);
                }
            } else {
                host = LibvirtHost.newHost(this, mbeanObjectName, hostName, url, hostAuthInfo, props, false);
                if (host != null) {
                    AgentCommon.getMBeanServer().registerMBean(host, mbeanObjectName);
                    this.hosts.add(host);
                    LibvirtServerPool.logger.info("In server pool " + this.servername + ": added sleeping Host MBean "
                        + mbeanObjectName);
                }
            }

            return host;
        } catch (Exception ex) {
            LibvirtServerPool.logger.error("Failed to add LibvirtHost", ex);
            throw new VMMException(ex);
        }
    }

    @Override
    public VirtualMachineImageStoreMXBean getVMImageStore() {
        return this.imageStore;
    }

    public String getSharedDiskStoreDir() {
        return this.diskStoreDir;
    }

    public String getVmImageStoreHostname() {
        return this.vmImageStoreHostname;
    }

    public synchronized long getSyncPeriodMillis() {
        return this.syncPeriodMillis;
    }

    @Override
    public void deleteImageStore() {
        try {
            int i = this.imageStore.listVMImageTemplates().size();
            while (i > 0) {
                AgentCommon.getMBeanServer().unregisterMBean(this.imageStore.listVMImageTemplates().get(i - 1).getObjectName());
                i--;
            }
        } catch (Throwable e) {
            LibvirtServerPool.logger.error(e);
        }
        try {
            AgentCommon.getMBeanServer().unregisterMBean(this.imageStore.getObjectName());
        } catch (Exception ex) {
            LibvirtServerPool.logger.error(ex);
        }
        this.imageStore = null;
    }

}
