/**
 * JASMINe VMMapi: JASMINe Virtual Machine Management API
 * Copyright (C) 2009 France Telecom R&D
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: VMwarePerfMonitor.java 7456 2011-01-19 23:05:51Z dangtran $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.vmm.agent.driver.vmware;

import java.math.BigInteger;
import java.util.HashMap;
import java.util.TimerTask;

import org.apache.log4j.Logger;

import com.vmware.vim.ArrayOfPerfCounterInfo;
import com.vmware.vim.ManagedObjectReference;
import com.vmware.vim.PerfCompositeMetric;
import com.vmware.vim.PerfCounterInfo;
import com.vmware.vim.PerfEntityMetric;
import com.vmware.vim.PerfEntityMetricBase;
import com.vmware.vim.PerfInterval;
import com.vmware.vim.PerfMetricId;
import com.vmware.vim.PerfMetricIntSeries;
import com.vmware.vim.PerfQuerySpec;
import com.vmware.vim.PerfSampleInfo;

/**
 * VMwarePerfMonitor is a periodic task which collect performance metrics of all
 * VMs on a given host and of the host itself
 */
class VMwarePerfMonitor extends TimerTask {
    static private Logger logger = Logger.getLogger(VMwarePerfMonitor.class);

    private VMwareServerPool.ConnectionPool connectionPool;

    private PerfQuerySpec qSpec = null;

    private PerfMetricId[] midList = null;

    private ManagedObjectReference perfManager = null;

    private HashMap<String, String> counterMap = null;

    private VMwareHost host = null;

    public VMwarePerfMonitor(final VMwareServerPool.ConnectionPool connectionPool, final VMwareHost host) {
        this.host = host;
        this.connectionPool = connectionPool;
    }

    public void configure(final String[][] counterNames) {
        final VMwareServiceConnection connection = this.connectionPool.getConnection();
        try {
            this.counterMap = new HashMap<String, String>();
            int nbCounters = counterNames.length;

            // getMetaData and Refs...
            if (this.perfManager == null) {
                this.perfManager = connection.getServiceContent().getPerfManager();
            }
            ArrayOfPerfCounterInfo cArray = (ArrayOfPerfCounterInfo) connection.getObjectProperty(this.perfManager,
                "perfCounter");
            PerfCounterInfo[] counterInfoList = cArray.getPerfCounterInfo();
            this.midList = new PerfMetricId[nbCounters];
            for (int i = 0; i < this.midList.length; i++) {
                PerfCounterInfo counterInfo = this.getCounterInfo(counterInfoList, counterNames[i][0], counterNames[i][1],
                    counterNames[i][2]);
                this.midList[i] = new PerfMetricId(null, null, counterInfo.getKey(), "");
            }
        } catch (Exception ex) {
            VMwarePerfMonitor.logger.error("Failed to configure VMwarePerfMonitor", ex);
        } finally {
            connection.release();
        }
        PerfInterval interval = new PerfInterval();
        interval.setSamplingPeriod(5);
        interval.setLength(3600);
        // com.vmware.vim.UpdatePerfInterval(perfManager, interval); ??

        this.qSpec = new PerfQuerySpec();
        this.qSpec.setMetricId(this.midList);
        this.qSpec.setIntervalId(new Integer(20)); // sampling period cannot be
        // lower
        // than 20 seconds
        this.qSpec.setMaxSample(1);
        this.qSpec.setEntity(this.host.getMOR());
    }

    @Override
    public void run() {
        HashMap<String, BigInteger> hostMetricValues = new HashMap<String, BigInteger>();
        HashMap<String, HashMap<String, BigInteger>> vmMetricValues = new HashMap<String, HashMap<String, BigInteger>>();
        long now = System.currentTimeMillis();
        final VMwareServiceConnection connection = this.connectionPool.getConnection();
        try {
            // VMwarePerfMonitor.logger.debug("Sending periodic perfQuery");
            PerfCompositeMetric cMetric = null;
            cMetric = connection.getService().queryPerfComposite(this.perfManager, this.qSpec);
            if (cMetric == null) {
                return;
            }
            PerfEntityMetric eMetric = (PerfEntityMetric) cMetric.getEntity();
            PerfMetricIntSeries iSeries = (PerfMetricIntSeries) eMetric.getValue(0);
            PerfEntityMetricBase[] childEntityMetric = cMetric.getChildEntity();
            PerfSampleInfo[] rootInfo = eMetric.getSampleInfo();
            int numSamples = eMetric.getSampleInfo().length;
            // Main result - host level
            for (int i = 0; i < eMetric.getValue().length; i++) {
                PerfMetricIntSeries rawSeries = (PerfMetricIntSeries) eMetric.getValue(i);
                String tName = this.counterMap.get("" + rawSeries.getId().getCounterId());
                long val = rawSeries.getValue(0);
                hostMetricValues.put(tName, BigInteger.valueOf(val));
            }
            // results for child (aka running VMs)
            if (childEntityMetric != null) {
                for (PerfEntityMetricBase element : childEntityMetric) {
                    PerfEntityMetric cStats = (PerfEntityMetric) element;
                    String eName = (String) connection.getObjectProperty(cStats.getEntity(), "name");
                    int numChildSamples = cStats.getSampleInfo().length;
                    HashMap<String, BigInteger> vmResult = new HashMap<String, BigInteger>();
                    vmMetricValues.put(eName, vmResult);
                    for (int k = 0; k < cStats.getValue().length; k++) {
                        PerfMetricIntSeries cSeries = (PerfMetricIntSeries) cStats.getValue(k);
                        String tName = this.counterMap.get("" + cSeries.getId().getCounterId());
                        long val = cSeries.getValue(cSeries.getValue().length - 1);
                        vmResult.put(tName, BigInteger.valueOf(val));
                    }
                }
            }
            this.host.updateMetrics(hostMetricValues, vmMetricValues);
        } catch (Exception ex) {
            VMwarePerfMonitor.logger.error("Failed to query perf metrics", ex);
        } finally {
            connection.release();
        }
    }

    private PerfCounterInfo getCounterInfo(final PerfCounterInfo[] cInfo, final String groupName, final String counterName,
        final String rollup) {
        String fullName = new String(groupName + "." + counterName);
        for (PerfCounterInfo current : cInfo) {
            if (current.getGroupInfo().getKey().equals(groupName) && current.getNameInfo().getKey().equals(counterName)
                && current.getRollupType().getValue().equals(rollup)) {
                // logger.debug("Using counter # " + current.getKey() + " [for "
                // + fullName + "]");
                this.counterMap.put("" + current.getKey(), fullName);
                return current;
            }
        }
        VMwarePerfMonitor.logger.debug("Warning - unable to find correct PerfCounterInfo for " + groupName + "." + counterName
            + " - " + rollup);
        return null;
    }

}
