/**
 * JASMINe VMMapi: JASMINe Virtual Machine Management API
 * Copyright (C) 2009-2010 France Telecom R&D
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: VMwareServerPool.java 7579 2011-01-31 13:53:59Z dangtran $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.vmm.agent.driver.vmware;

import java.util.ArrayList;
import java.util.Calendar;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

import javax.management.ObjectName;

import org.apache.log4j.Logger;
import org.ow2.jasmine.vmm.agent.domain.ManagedResource;
import org.ow2.jasmine.vmm.agent.domain.ServerPool;
import org.ow2.jasmine.vmm.agent.driver.util.RemoteExec;
import org.ow2.jasmine.vmm.agent.jmx.MBeanObjectNamer;
import org.ow2.jasmine.vmm.agent.main.AgentCommon;
import org.ow2.jasmine.vmm.api.HostMXBean;
import org.ow2.jasmine.vmm.api.VMMException;
import org.ow2.jasmine.vmm.api.VirtualMachineImageMXBean;
import org.ow2.jasmine.vmm.api.VirtualMachineImageStoreMXBean;

import com.vmware.vim.ArrayOfManagedObjectReference;
import com.vmware.vim.DynamicProperty;
import com.vmware.vim.ManagedObjectReference;
import com.vmware.vim.ObjectContent;

/**
 * VMware driver ServerPool MXBean implementation
 */
class VMwareServerPool extends ServerPool {
    static protected Logger logger = Logger.getLogger(VMwareServerPool.class);

    static private int KEEP_ALIVE_PERIOD_MS = 2 * 60 * 1000;

    private static final int THREADPOOL_SIZE = 20;

    private static final int CONNECTION_DEFAULT_POOL_SIZE = 4;

    private static ExecutorService executorService = Executors.newFixedThreadPool(VMwareServerPool.THREADPOOL_SIZE);

    private ConnectionPool connectionPool;

    private HashMap<String, VMwareHost> hostRefs = new HashMap<String, VMwareHost>();

    private String datacenter, vmFolderName, datastoreName, networkName;

    private ManagedObjectReference datacenterRef, vmFolderRef, datastoreRef, vmTemplateFolderRef, hostFolderRef;

    private VMwareVMTemplateDataStore imageStore;

    private RemoteExec.SshAuthInfo defaultAuthInfo;

    public VMwareServerPool(final String name, final ObjectName objectName, final Map<String, String> attributes)
        throws IllegalArgumentException, VMMException {
        super(name, objectName, attributes);
        String hostName = attributes.get("vCenterHostName");
        if (hostName == null) {
            throw new IllegalArgumentException("Missing parameter: vCenterHostName");
        }
        this.datacenter = attributes.get("datacenter");
        if (this.datacenter == null) {
            throw new IllegalArgumentException("Missing parameter: datacenter");
        }
        this.vmFolderName = attributes.get("vmFolderPath");
        if (this.vmFolderName == null) {
            throw new IllegalArgumentException("Missing parameter: vmFolderName");
        }
        String vmTemplateFolderName = attributes.get("vmTemplateFolderPath");
        if (vmTemplateFolderName == null) {
            throw new IllegalArgumentException("Missing parameter: vmTemplateFolderName");
        }
        String userName = attributes.get("user");
        if (userName == null) {
            throw new IllegalArgumentException("Missing parameter: userName");
        }
        String password = attributes.get("password");
        if (password == null) {
            throw new IllegalArgumentException("Missing parameter: password");
        }
        this.datastoreName = attributes.get("datastore");
        if (this.datastoreName == null) {
            throw new IllegalArgumentException("Missing parameter: datastoreName");
        }
        this.networkName = attributes.get("network");

        String sshPassword = attributes.get("sshDefaultRootPassword");
        String sshPrivateKeyFile = attributes.get("sshDefaultPrivateKeyFile");
        try {
            this.defaultAuthInfo = new RemoteExec.SshAuthInfo(sshPassword, sshPrivateKeyFile);
        } catch (IllegalArgumentException ex) {
            // XXX
        }

        VMwareServiceConnection connection = null;
        try {
            this.connectionPool = new ConnectionPool(VMwareServerPool.CONNECTION_DEFAULT_POOL_SIZE, hostName, userName,
                password);
            connection = this.connectionPool.getConnection();

            VMwareServerPool.logger.debug("Finding datacenter " + this.datacenter);
            this.datacenterRef = connection.getService().findByInventoryPath(connection.getServiceContent().getSearchIndex(),
                this.datacenter);
            if (this.datacenterRef == null) {
                String msg = "VMware Driver initialization error: datacenter " + this.datacenter + " not found ";
                VMwareServerPool.logger.error(msg);
                throw new VMMException(msg);
            }

            // Find the virtual machine folder for this folder.
            if (this.vmFolderName != null) {
                this.vmFolderRef = connection.getService().findByInventoryPath(connection.getServiceContent().getSearchIndex(),
                    this.vmFolderName);
            } else {
                this.vmFolderRef = (ManagedObjectReference) this.getDynamicProarray(connection, this.datacenterRef, "vmFolder")[0]
                    .getVal();
            }
            if (this.vmFolderRef == null) {
                String msg = "VMware Driver initialization error: virtual machine folder " + this.vmFolderName + " not found";
                VMwareServerPool.logger.error(msg);
                throw new VMMException(msg);
            }

            // Find the virtual machine folder for this folder.
            // if (this.vmFolderName != null) {
            // ArrayOfManagedObjectReference subVmFolderRefs =
            // (ArrayOfManagedObjectReference) this.getDynamicProarray(
            // this.vmFolderRef, "childEntity")[0].getVal();
            // if (subVmFolderRefs == null) {
            // String msg =
            // "VMware Driver initialization error: virtual machine sub folders is not found";
            // VMwareServerPool.logger.error(msg);
            // throw new VMMException(msg);
            // }
            // for (ManagedObjectReference mor :
            // subVmFolderRefs.getManagedObjectReference()) {
            // String folderName = (String) this.getDynamicProarray(mor,
            // "name")[0].getVal();
            // if (folderName.equals(this.vmFolderName)) {
            // this.vmFolderRef = mor;
            // break;
            // }
            // }
            // }

            // Find the virtual machine template folder
            if (vmTemplateFolderName != null) {
                this.vmTemplateFolderRef = connection.getService().findByInventoryPath(
                    connection.getServiceContent().getSearchIndex(), vmTemplateFolderName);
            } else {
                this.vmTemplateFolderRef = this.vmFolderRef;
            }
            if (this.vmTemplateFolderRef == null) {
                String msg = "VMware Driver initialization error: virtual machine template folder " + vmTemplateFolderName
                    + " not found";
                VMwareServerPool.logger.error(msg);
                throw new VMMException(msg);
            }

            // this.vmTemplateFolderRef = this.vmFolderRef;
            // if (vmTemplateFolderName != null) {
            // ArrayOfManagedObjectReference subVmFolderRefs =
            // (ArrayOfManagedObjectReference) this.getDynamicProarray(
            // this.vmFolderRef, "childEntity")[0].getVal();
            // if (subVmFolderRefs == null) {
            // String msg =
            // "VMware Driver initialization error: virtual machine sub folders is not found";
            // VMwareServerPool.logger.error(msg);
            // throw new VMMException(msg);
            // }
            // for (ManagedObjectReference mor :
            // subVmFolderRefs.getManagedObjectReference()) {
            // String folderName = (String) this.getDynamicProarray(mor,
            // "name")[0].getVal();
            // if (folderName.equals(vmTemplateFolderName)) {
            // this.vmTemplateFolderRef = mor;
            // break;
            // }
            // }
            // }

            // Find the host folder for this folder.
            this.hostFolderRef = (ManagedObjectReference) this.getDynamicProarray(connection, this.datacenterRef, "hostFolder")[0]
                .getVal();
            if (this.hostFolderRef == null) {
                String msg = "VMware Driver initialization error: host folder not found for datacenter " + this.datacenter;
                VMwareServerPool.logger.error(msg);
                throw new VMMException(msg);
            }

            // find the datastore
            ArrayOfManagedObjectReference dataStoreRefs = (ArrayOfManagedObjectReference) this.getDynamicProarray(connection,
                this.datacenterRef, "datastore")[0].getVal();
            if (dataStoreRefs != null) {
                for (int i = 0; i < dataStoreRefs.getManagedObjectReference().length; i++) {
                    String dsName = (String) this.getDynamicProarray(connection, dataStoreRefs.getManagedObjectReference(i),
                        "info.name")[0].getVal();
                    if (dsName.equals(this.datastoreName)) {
                        this.datastoreRef = dataStoreRefs.getManagedObjectReference(i);
                        break;
                    }
                }
            }
            if (this.datastoreRef == null) {
                String msg = "VMware Driver initialization error: Datastore " + this.datastoreName + " cannot be found";
                VMwareServerPool.logger.error(msg);
                throw new VMMException(msg);
            }

            String vmImageStoreName = "ImageStore(" + vmTemplateFolderName + ")";
            this.imageStore = new VMwareVMTemplateDataStore(MBeanObjectNamer.makeVMImageStoreName(vmImageStoreName),
                vmImageStoreName, this.datastoreRef, attributes);
            AgentCommon.getMBeanServer().registerMBean(this.imageStore, this.imageStore.getObjectName());

        } catch (Exception ex) {
            String msg = "VMware Driver initialization error: " + ex.getMessage();
            VMwareServerPool.logger.error(msg, ex);
            throw new VMMException(msg);
        } finally {
            if (connection != null) {
                connection.release();
            }
        }

        new VMwareEventCollector(this, new VMwareServiceConnection(hostName, userName, password), this.datacenter).start();

    }

    @Override
    public String getHypervisor() {
        return "ESX";
    }

    public ExecutorService getExecutorService() {
        return VMwareServerPool.executorService;
    }

    public ConnectionPool getConnectionPool() {
        return this.connectionPool;
    }

    @Override
    public VirtualMachineImageStoreMXBean getVMImageStore() {
        return this.imageStore;
    }

    public String getMountPoint() {
        return this.datacenter;
    }

    VMwareHost getHostByName(final String name) {
        return this.hostRefs.get(name);
    }

    @Override
    public HostMXBean newHost(final String hostName, final Map<String, String> props) throws VMMException {
        ManagedObjectReference mor = null;
        VMwareServiceConnection connection = this.connectionPool.getConnection();
        try {
            mor = connection.getDecendentMoRef(this.getHostFolderRef(), "HostSystem", hostName);
        } catch (Exception ex) {
            VMwareServerPool.logger.error("Unable to find HostSystem managed object " + hostName + " in VMware VC inventory",
                ex);
            return null;
        } finally {
            connection.release();
        }

        if (mor == null) {
            VMwareServerPool.logger.error("Host " + hostName + " not found");
            return null;
        }

        RemoteExec.SshAuthInfo hostAuthInfo;
        if (props.get("sshRootPassword") != null) {
            hostAuthInfo = new RemoteExec.SshAuthInfo(props.get("sshRootPassword"), null);
        } else if (props.get("sshPrivateKeyFile") != null) {
            hostAuthInfo = new RemoteExec.SshAuthInfo(null, props.get("sshPrivateKeyFile"));
        } else {
            hostAuthInfo = this.defaultAuthInfo;
        }

        ObjectName mbeanObjectName = MBeanObjectNamer.makeHostName(this.getPath() + "/" + hostName, hostName);
        VMwareHost host = new VMwareHost(this, mor, mbeanObjectName, hostName, hostAuthInfo, props);
        host.getResidentVMs();

        try {
            AgentCommon.getMBeanServer().registerMBean(host, mbeanObjectName);
        } catch (Exception ex) {
            VMwareServerPool.logger.error("Unable to register MBean " + mbeanObjectName, ex);
            return null;
        }
        this.hostRefs.put(hostName, host);
        // logger.info("Added host "+hostName+" "+host.getObjectName());
        this.addHost(host);
        try {
            this.connectionPool.ensureCapacity(2 + this.hostRefs.size() * 3); // XXX
        } catch (Exception ex) {
            VMwareServerPool.logger.error("Cannot increase connection pool", ex);
        }
        return host;
    }

    public ManagedObjectReference getDatacenterRef() {
        return this.datacenterRef;
    }

    public ManagedObjectReference getVmFolderRef() {
        return this.vmFolderRef;
    }

    public ManagedObjectReference getHostFolderRef() {
        return this.hostFolderRef;
    }

    public ManagedObjectReference getVmTemplateFolderRef() {
        return this.vmTemplateFolderRef;
    }

    public ManagedObjectReference getDatastoreRef() {
        return this.datastoreRef;
    }

    public String getDatastoreName() {
        return this.datastoreName;
    }

    public String getNetworkName() {
        return this.networkName;
    }

    private DynamicProperty[] getDynamicProarray(final VMwareServiceConnection connection, final ManagedObjectReference MOR,
        final String pName) throws Exception {
        ObjectContent[] objContent;
        objContent = connection.getObjectProperties(null, MOR, new String[] {pName});
        ObjectContent contentObj = objContent[0];
        DynamicProperty[] objArr = contentObj.getPropSet();
        return objArr;
    }

    class ConnectionPool {
        private List<VMwareServiceConnection> connections = new ArrayList<VMwareServiceConnection>();

        private String hostName, userName, password;

        ConnectionPool(final int initialCapacity, final String hostName, final String userName, final String password)
            throws Exception {
            this.userName = userName;
            this.hostName = hostName;
            this.password = password;
            this.ensureCapacity(initialCapacity);
            VMwareServerPool.executorService.execute(new Runnable() {
                public void run() {
                    while (true) {
                        try {
                            Thread.sleep(VMwareServerPool.KEEP_ALIVE_PERIOD_MS);
                            synchronized (ConnectionPool.this) {
                                for (VMwareServiceConnection connection : ConnectionPool.this.connections) {
                                    if (connection.lease()) {
                                        try {
                                            Calendar time = connection.getService().currentTime(
                                                connection.getServiceInstanceRef());
                                        } finally {
                                            connection.release();
                                        }
                                    }
                                }
                            }
                        } catch (Exception ex) {
                            VMwareServerPool.logger.error("error", ex);
                        }
                    }
                }
            });
        }

        synchronized void ensureCapacity(final int minCapacity) throws Exception {
            if (this.connections.size() < minCapacity) {
                for (int i = minCapacity - this.connections.size(); i > 0; i--) {
                    VMwareServiceConnection connection = new VMwareServiceConnection(this.hostName, this.userName,
                        this.password);
                    VMwareServerPool.logger.debug("Connecting to VMware VirtualCenter " + this.hostName + " ...");
                    connection.connect();
                    VMwareServerPool.logger.debug("Done");
                    this.connections.add(connection);
                }
            }
        }

        private synchronized VMwareServiceConnection getFirstFreeConnection() {
            for (VMwareServiceConnection connection : this.connections) {
                if (connection.lease()) {
                    return connection;
                }
            }
            return null;
        }

        synchronized VMwareServiceConnection getConnection() {
            VMwareServiceConnection connection = this.getFirstFreeConnection();
            if (connection == null) {
                try {
                    this.ensureCapacity(this.connections.size() + 5);
                } catch (Exception ex) {
                    // ignore
                }
                connection = this.getFirstFreeConnection();
            }
            return connection;
        }
    }

    class VMwareVMTemplateDataStore extends ManagedResource implements VirtualMachineImageStoreMXBean {
        private String name;

        private ArrayList<VirtualMachineImageMXBean> imageList = new ArrayList<VirtualMachineImageMXBean>();

        private ManagedObjectReference datastoreRef;

        private boolean needSync = true;

        public VMwareVMTemplateDataStore(final ObjectName objectName, final String name,
            final ManagedObjectReference datastoreRef, final Map<String, String> attributes) {
            super(objectName);
            this.name = name;
            this.datastoreRef = datastoreRef;
            this.listVMImageTemplates();
            Set<String> templateNames = new HashSet<String>();
            for (String key : attributes.keySet()) {
                if (key.startsWith("template")) {
                    templateNames.add(key.substring(0, key.indexOf('.')));
                }
            }
            for (String template : templateNames) {
                String isoFileName = attributes.get(template + ".isoFileName");
                String description = attributes.get(template + ".description");
                String uuid = attributes.get(template + ".uuid");
                String makeFloppyShellScript = attributes.get(template + ".makeFloppyShellScript");
                String guestId = attributes.get(template + ".guestId");

                ObjectName on = MBeanObjectNamer.makeVMImageName(uuid);
                VMwareVirtualMachineISOTemplate image = new VMwareVirtualMachineISOTemplate(on, uuid, uuid);
                image.setMakeFloppyShellScript(makeFloppyShellScript);
                image.setGuestId(guestId);
                image.setIsoFileName(isoFileName);
                VMwareServerPool.logger.debug("New VMware template: " + image);
                try {
                    AgentCommon.getMBeanServer().registerMBean(image, on);
                } catch (Exception ex) {
                    VMwareServerPool.logger.error("Failed to register image", ex);
                    continue;
                }
                this.imageList.add(image);

            }
        }

        public long getCapacityMB() {
            VMwareServiceConnection connection = VMwareServerPool.this.connectionPool.getConnection();
            try {
                long capacityInBytes = (Long) VMwareServerPool.this.getDynamicProarray(connection, this.datastoreRef,
                    "summary.capacity")[0].getVal();
                return capacityInBytes / (1024 * 1024);
            } catch (Exception ex) {
                VMwareServerPool.logger.error("Failed to get datastore capacity", ex);
                return -1;
            } finally {
                connection.release();
            }
        }

        public long getFreeSpaceMB() {
            VMwareServiceConnection connection = VMwareServerPool.this.connectionPool.getConnection();
            try {
                long freeSpaceInBytes = (Long) VMwareServerPool.this.getDynamicProarray(connection, this.datastoreRef,
                    "summary.freeSpace")[0].getVal();
                return freeSpaceInBytes / (1024 * 1024);
            } catch (Exception ex) {
                VMwareServerPool.logger.error("Failed to get datastore free space", ex);
                return -1;
            } finally {
                connection.release();
            }
        }

        public String getName() {
            return this.name;
        }

        public synchronized void sync() {
            this.needSync = true;
        }

        public synchronized List<VirtualMachineImageMXBean> listVMImageTemplates() {
            VMwareServerPool.logger.debug("listVMImages");
            if (this.needSync) {
                VMwareServiceConnection connection = VMwareServerPool.this.connectionPool.getConnection();
                try {
                    ArrayList<ManagedObjectReference> vms = connection.getDecendentMoRefs(
                        VMwareServerPool.this.getVmTemplateFolderRef(), "VirtualMachine");

                    for (ManagedObjectReference vmMor : vms) {
                        boolean isTemplate = (Boolean) VMwareServerPool.this.getDynamicProarray(connection, vmMor,
                            "summary.config.template")[0].getVal();
                        if (isTemplate) {
                            DynamicProperty[] properties = VMwareServerPool.this.getDynamicProarray(connection, vmMor,
                                "config.name");
                            if (properties == null || properties.length < 1) {
                                continue;
                            }
                            String vmName = (String) properties[0].getVal();
                            String uuid = (String) VMwareServerPool.this.getDynamicProarray(connection, vmMor, "config.uuid")[0]
                                .getVal();

                            boolean found = false;
                            for (VirtualMachineImageMXBean image : this.imageList) {
                                if (image.getUUID().equals(vmName)) {
                                    found = true;
                                }
                            }
                            if (!found) {
                                ObjectName on = MBeanObjectNamer.makeVMImageName(uuid);
                                VMwareVirtualMachineTemplate image = new VMwareVirtualMachineTemplate(on, vmName, uuid);
                                VMwareServerPool.logger.debug("New VMware template: " + image);
                                AgentCommon.getMBeanServer().registerMBean(image, on);
                                this.imageList.add(image);
                            }
                        }

                    }

                } catch (Exception ex) {
                    VMwareServerPool.logger.error("Failed to list VM templates", ex);
                } finally {
                    connection.release();
                }
                this.needSync = false;
            }
            return this.imageList;
        }

        public synchronized VirtualMachineImageMXBean lookUpByUUID(final String uuid) {
            VMwareServerPool.logger.debug("lookUpByUUID " + uuid);
            for (VirtualMachineImageMXBean image : this.listVMImageTemplates()) {
                if (image.getUUID().equals(uuid)) {
                    return image;
                }
            }
            return null;
        }

        public synchronized void removeVMImageTemplate(final VirtualMachineImageMXBean vmImage) throws VMMException {
            String vmImageID = vmImage.getUUID();
            VMwareServiceConnection connection = VMwareServerPool.this.connectionPool.getConnection();
            try {
                ManagedObjectReference vmMOR = connection.getDecendentMoRef(null, "VirtualMachine", vmImageID);
                // asynchronous op, forget about result ?
                connection.getService().destroy_Task(vmMOR);
            } catch (Exception ex) {
                VMwareServerPool.logger.error("Failed to retrieve VM reference " + vmImageID, ex);
            } finally {
                connection.release();
            }
        }

    }

    static class VMwareVirtualMachineTemplate extends ManagedResource implements VirtualMachineImageMXBean {
        private String name, uuid;

        public VMwareVirtualMachineTemplate(final ObjectName objectName, final String name, final String uuid) {
            super(objectName);
            this.name = name;
            this.uuid = uuid;
        }

        public String getName() {
            return this.name;
        }

        // TODO we ignore the UUID and use the name as a unique
        public String getUUID() {
            return this.name;
        }

        public Map<String, String> getMetaData() {
            return null;
        }

        public String getDescription() {
            return "";
        }

        @Override
        public String toString() {
            return "VMwareTemplate(name=" + this.getName() + ",uuid=" + this.getUUID() + ")";
        }
    }

    static class VMwareVirtualMachineISOTemplate extends VMwareVirtualMachineTemplate implements VirtualMachineImageMXBean {
        private String isoFileName, makeFloppyShellScript, guestId;

        public VMwareVirtualMachineISOTemplate(final ObjectName objectName, final String name, final String uuid) {
            super(objectName, name, uuid);
        }

        public String getIsoFileName() {
            return this.isoFileName;
        }

        public void setIsoFileName(final String isoFileName) {
            this.isoFileName = isoFileName;
        }

        public String getMakeFloppyShellScript() {
            return this.makeFloppyShellScript;
        }

        public void setMakeFloppyShellScript(final String makeFloppyShellScript) {
            this.makeFloppyShellScript = makeFloppyShellScript;
        }

        public String getGuestId() {
            return this.guestId;
        }

        public void setGuestId(final String guestId) {
            this.guestId = guestId;
        }

        @Override
        public String toString() {
            return "VMwareISOTemplate(uuid=" + this.getUUID() + ",iso=" + this.getIsoFileName() + ")";
        }

    }

    @Override
    public void deleteImageStore() {
        try {
            int i = this.imageStore.listVMImageTemplates().size();
            while (i > 0) {
                AgentCommon.getMBeanServer().unregisterMBean(this.imageStore.listVMImageTemplates().get(i - 1).getObjectName());
                i--;
            }
        } catch (Throwable e) {
            ServerPool.logger.error(e);
        }
        try {
            AgentCommon.getMBeanServer().unregisterMBean(this.imageStore.getObjectName());
        } catch (Exception e) {
            ServerPool.logger.error(e);
        }
        this.imageStore = null;
    }
}
