/**
 * JASMINe VMMapi: JASMINe Virtual Machine Management API
 * Copyright (C) 2009-2010 France Telecom R&D
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: XenHost.java 7456 2011-01-19 23:05:51Z dangtran $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.vmm.agent.driver.xenapi;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

import javax.management.MBeanNotificationInfo;
import javax.management.Notification;
import javax.management.NotificationEmitter;
import javax.management.ObjectName;

import org.apache.log4j.Logger;
import org.ow2.jasmine.vmm.agent.domain.AbstractHost;
import org.ow2.jasmine.vmm.agent.driver.xenapi.XenVMImageStore.XenVirtualMachineImage;
import org.ow2.jasmine.vmm.agent.jmx.MBeanObjectNamer;
import org.ow2.jasmine.vmm.agent.main.AgentCommon;
import org.ow2.jasmine.vmm.api.InsufficientResourcesException;
import org.ow2.jasmine.vmm.api.InvalidVMConfigException;
import org.ow2.jasmine.vmm.api.NotificationTypes;
import org.ow2.jasmine.vmm.api.ResourceUsage;
import org.ow2.jasmine.vmm.api.ServerPoolMXBean;
import org.ow2.jasmine.vmm.api.VMConfigSpec;
import org.ow2.jasmine.vmm.api.VMCustomizationSpec;
import org.ow2.jasmine.vmm.api.VMMException;
import org.ow2.jasmine.vmm.api.VirtualMachineImageStoreMXBean;
import org.ow2.jasmine.vmm.api.VirtualMachineMXBean;

import com.xensource.xenapi.Connection;
import com.xensource.xenapi.Host;
import com.xensource.xenapi.HostCpu;
import com.xensource.xenapi.Network;
import com.xensource.xenapi.SR;
import com.xensource.xenapi.Types.VmPowerState;
import com.xensource.xenapi.VIF;
import com.xensource.xenapi.VM;

/**
 * XenAPI driver Host MXBean implementation
 */
public class XenHost extends AbstractHost implements NotificationEmitter {
    static Logger logger = Logger.getLogger(XenHost.class);

    // the thread pool size determines the maximum number of
    // simultaneous VM creation on the same host
    private static final int MAX_PARALELL_VM_CREATION_PER_HOST = 3;

    // one extra thread for the event reader
    private static final int THREADPOOL_SIZE = 1 + XenHost.MAX_PARALELL_VM_CREATION_PER_HOST;

    private static ExecutorService executorService = Executors.newFixedThreadPool(XenHost.THREADPOOL_SIZE);

    private List<XenVirtualMachine> vmList = new CopyOnWriteArrayList<XenVirtualMachine>();

    private XenServerPool serverPool;

    private Connection connection;

    com.xensource.xenapi.Host host;

    private SR storageRepository;

    private String hostName;

    private HostPowerState hostPowerState = HostPowerState.UNKNOWN;

    private int numCPU = -1;

    private long cpuFrequencyMhz = -1;

    private String uuid;

    private Map<String, String> hostAttributes;

    public static XenHost newHost(final XenServerPool resourcePool, final ObjectName objectName, final String hostName,
        final String poolUser, final String poolPassword, final int port, final Connection masterCon,
        final Map<String, String> attributes) {
        String user = attributes.get("user");
        if (user == null) {
            user = poolUser;
            attributes.put("user", poolUser);
        }
        if (user == null) {
            throw new IllegalArgumentException("Host: " + hostName + " Missing parameter: user");
        }
        String password = attributes.get("password");
        if (password == null) {
            password = poolPassword;
            attributes.put("password", poolPassword);
        }
        if (password == null) {
            throw new IllegalArgumentException("Host: " + hostName + " Missing parameter: password");
        }

        try {
            Set<Host> hosts = Host.getAll(masterCon);
            XenHost xenHost = null;
            for (Host h : hosts) {
                if (h.getNameLabel(masterCon).equals(hostName) || h.getAddress(masterCon).equals(hostName)) {
                    xenHost = new XenHost(resourcePool, masterCon, objectName, hostName, h, attributes);
                }
            }
            return xenHost;
        } catch (Exception ex) {
            XenHost.logger.error("Failed to establish Xen-API connection with host " + hostName + " with login " + user, ex);
            return null;
        }

    }

    private XenHost(final XenServerPool serverPool, final Connection connection, final ObjectName objectName,
        final String hostName, final Host host, final Map<String, String> hostAttributes) {
        super(objectName);
        this.serverPool = serverPool;
        this.connection = connection;
        this.hostName = hostName;
        this.host = host;
        this.hostAttributes = new HashMap<String, String>(hostAttributes);
        try {
            this.uuid = host.getUuid(this.connection);
            this.populate();
        } catch (Exception ex) {
            XenHost.logger.debug("Host " + hostName, ex);
        }
        this.storageRepository = this.getStorageRepository();
        this.hostPowerState = HostPowerState.RUNNING;
    }

    public void removeVMFromInventory(final XenVirtualMachine vm) {
        try {
            AgentCommon.getMBeanServer().unregisterMBean(vm.getObjectName());
        } catch (Exception ex) {
            XenHost.logger.error("Cannot unregister VM MBean", ex);
        }
        this.vmList.remove(vm.getNameLabel());
        this.emitNotification(NotificationTypes.VM_INVENTORY_DEL, "Deleted", vm.getObjectName());
        XenHost.logger.info("deleted VM " + vm.getObjectName());
    }

    public boolean isConnectionLost() {
        synchronized (this.connection) {
            try {
                this.host.getAPIVersionMajor(this.connection);
            } catch (Exception ex) {
                return true;
            }
            return false;
        }
    }

    Connection getXenAPIConnection() {
        return this.connection;
    }

    private SR getStorageRepository() {
        if (this.storageRepository == null) {
            String sharedStorageRepository = this.serverPool.getSharedStorageRepository();
            try {
                Set<SR> srSet = SR.getByNameLabel(this.connection, sharedStorageRepository);
                if (srSet.isEmpty()) {
                    return null;
                }
                this.storageRepository = srSet.iterator().next();
            } catch (Exception ex) {
                XenHost.logger.error("Failed to get SR " + sharedStorageRepository, ex);
            }
        }
        return this.storageRepository;
    }

    List<XenVirtualMachine> getVMs() {
        return this.vmList;
    }

    XenVirtualMachine lookUpByUuid(final String uuid) {
        for (XenVirtualMachine vm : this.vmList) {
            if (vm.getUuid().equals(uuid)) {
                return vm;
            }
        }
        return null;
    }

    protected synchronized void addVM(final XenVirtualMachine vm) {
        this.vmList.add(vm);
    }

    protected synchronized void removeVM(final XenVirtualMachine vm) {
        this.vmList.remove(vm);
    }

    public ServerPoolMXBean getServerPool() {
        return this.serverPool;
    }

    public void postMigrateVM(final XenVirtualMachine vm, final XenHost newHost) {
        this.removeVM(vm);
        newHost.addVM(vm);
    }

    public void configurePerfMonitor(final Set<PerfMetric> metricsOfInterest, final long periodMillis) {
    }

    @Override
    public ObjectName getObjectName() {
        return this.objectName;
    }

    HashMap<String, String> hypervisorInfo;

    // TODO
    public Map<String, String> getHypervisorInfo() throws VMMException {
        if (this.hypervisorInfo == null) {
            this.hypervisorInfo = new HashMap<String, String>();
            this.hypervisorInfo.put("vendor", "");
            this.hypervisorInfo.put("name", "XenServer");
        }
        return this.hypervisorInfo;
    }

    HashMap<String, String> cpuInfo;

    public Map<String, String> getCPUInfo() throws VMMException {
        if (this.cpuInfo == null) {
            this.cpuInfo = new HashMap<String, String>();
            synchronized (this.connection) {
                try {
                    HostCpu[] cpus = this.host.getHostCPUs(this.connection).toArray(new HostCpu[0]);
                    this.numCPU = cpus.length;
                    HostCpu.Record rec = cpus[0].getRecord(this.connection);
                    this.cpuInfo.put("model", rec.modelname);
                    this.cpuFrequencyMhz = rec.speed;
                    this.cpuInfo.put("speedMHz", Long.toString(rec.speed));
                    this.cpuInfo.put("vendor", rec.vendor);
                    this.cpuInfo.put("features", rec.features);
                    this.cpuInfo.put("flags", rec.flags);
                } catch (Exception ex) {
                    throw XenVirtualMachine.translateXenAPIException(ex);
                }
            }
        }
        return this.cpuInfo;
    }

    public long getCPUFrequencyMhz() throws VMMException {
        if (this.cpuFrequencyMhz == -1) {
            this.getCPUInfo();
        }
        return this.cpuFrequencyMhz;

    }

    public int getNumCPU() throws VMMException {
        if (this.numCPU == -1) {
            this.getCPUInfo();
        }
        return this.numCPU;
    }

    public String getHostName() {
        return this.hostName;
    }

    public String getUuid() {
        return this.uuid;
    }

    public VirtualMachineImageStoreMXBean getVMImageStore() {
        return this.serverPool.getVMImageStore();
    }

    public long getFreeMemoryMB() throws VMMException {
        synchronized (this.connection) {
            try {
                return this.host.getMetrics(this.connection).getMemoryFree(this.connection) / (1024 * 1024);
            } catch (Exception ex) {
                throw XenVirtualMachine.translateXenAPIException(ex);
            }
        }
    }

    private void populate() {
        try {
            synchronized (this.connection) {
                Set<VM> vms = VM.getAll(this.connection);
                for (VM vm : vms) {
                    if (vm.getIsATemplate(this.connection)) {
                        continue;
                    }
                    ObjectName name;
                    if (vm.getDomid(this.connection) == 0) {
                        // Do not add Domain-0 in the VM list
                        continue;
                    } else {
                        if (vm.getPowerState(this.connection) == VmPowerState.HALTED) {
                            if (!this.host.getUuid(this.connection).equals(
                                vm.getAffinity(this.connection).getUuid(this.connection))) {
                                continue;
                            }
                        } else {
                            String vmUuid = vm.getUuid(this.connection);
                            Set<VM> hostVMs = this.host.getResidentVMs(this.connection);
                            boolean isHostVM = false;
                            for (VM hostVM : hostVMs) {
                                if (vmUuid.equals(hostVM.getUuid(this.connection))) {
                                    isHostVM = true;
                                    break;
                                }
                            }
                            if (!isHostVM) {
                                continue;
                            }
                        }
                        name = MBeanObjectNamer.makeVirtualMachineName(
                            this.serverPool.getPath() + "/" + vm.getNameLabel(this.connection), vm.getUuid(this.connection));
                    }

                    XenVirtualMachine xenVM = new XenVirtualMachine(name, this, this.connection, vm, null);
                    this.addVM(xenVM);
                    AgentCommon.getMBeanServer().registerMBean(xenVM, name);
                    XenHost.logger.info("Added XenVirtualMachineMBean " + name);
                }
            }
        } catch (Exception ex) {
            XenHost.logger.debug("Host " + this.hostName, ex);
        }
    }

    public List<VirtualMachineMXBean> getResidentVMs() {
        List<VirtualMachineMXBean> result = new ArrayList<VirtualMachineMXBean>(this.vmList);
        return result;
    }

    public VirtualMachineMXBean createVM(final VMConfigSpec vmSpecIn, final boolean sync) throws InvalidVMConfigException,
        VMMException {
        final VMConfigSpec vmSpec = new VMConfigSpec(vmSpecIn);
        XenHost.logger.info("Creating new VM [name=" + vmSpec.getName() + ",memorySizeMB=" + vmSpec.getMemorySizeMB()
            + ",diskSize=" + vmSpec.getDiskSizeMB() + ",numVCPU=" + vmSpec.getNumVCPU() + "]");

        final XenVirtualMachineImage template = (XenVirtualMachineImage) this.serverPool.getVMImageStore().lookUpByUUID(
            vmSpec.getVmImageUUID());
        if (template == null) {
            throw new InvalidVMConfigException("Invalid VM Image UUID");
        }
        final VM templateVM;
        try {
            templateVM = VM.getByUuid(this.connection, template.getUUID());
        } catch (Exception ex) {
            throw XenVirtualMachine.translateXenAPIException(ex);
        }
        if (templateVM == null) {
            throw new InvalidVMConfigException("Invalid VM Image UUID");
        }

        final String vmName = vmSpec.getName();

        if (!sync) {
            XenHost.executorService.execute(new Runnable() {
                public void run() {
                    try {
                        XenHost.this.cloneVM(vmSpec, vmName, templateVM);
                    } catch (Exception ex) {
                    }
                }
            });
            return null;
        } else {
            return this.cloneVM(vmSpec, vmName, templateVM);
        }
    }

    public VirtualMachineMXBean cloneVM(final VM vm, final String clonedVmName, final VMCustomizationSpec custSpec,
        final boolean sync) throws InsufficientResourcesException, VMMException {
        if (!sync) {
            XenHost.executorService.execute(new Runnable() {
                public void run() {
                    try {
                        XenHost.this.cloneVM(null, clonedVmName, vm);
                    } catch (Exception ex) {
                    }
                }
            });
            return null;
        } else {
            return this.cloneVM(null, clonedVmName, vm);
        }
    }

    private VirtualMachineMXBean cloneVM(final VMConfigSpec vmSpec, final String vmName, final VM templateVM)
        throws InvalidVMConfigException, VMMException {
        try {
            VM newVM;
            String uuid;
            // synchronized (this.connection) {
            newVM = templateVM.createClone(this.connection, vmName);

            if (vmSpec != null) {
                // network configuration
                Set<Network> networks = Network.getAll(this.connection);
                Network network = null;
                // XXX hardcoded bridge name xenbr0
                for (Network net : networks) {
                    String bridge = net.getBridge(this.connection);
                    XenHost.logger.debug("Network bridge=" + bridge);
                    if (bridge.equals("xenbr0")) {
                        network = net;
                        break;
                    }
                }
                if (network == null) {
                    throw new VMMException("Failed to find bridge xenbr0");
                }
                VIF.Record newvifrecord = new VIF.Record();
                newvifrecord.VM = newVM;
                newvifrecord.network = network;
                newvifrecord.device = "0";
                newvifrecord.MTU = 1500L;
                VIF.create(this.connection, newvifrecord);

                // set VM storage repository

                Map<String, String> otherConfig = newVM.getOtherConfig(this.connection);
                String disks = otherConfig.get("disks");
                disks = disks.replace("sr=\"\"", "sr=\"" + this.storageRepository.getUuid(this.connection) + "\"");
                otherConfig.put("disks", disks);
                newVM.setOtherConfig(this.connection, otherConfig);

                // to boot non-interactively
                newVM.setPVArgs(this.connection, "noninteractive");

                // provision VM
                newVM.provision(this.connection);
            }

            uuid = newVM.getUuid(this.connection);
            // }

            synchronized (this.serverPool.eventCollector) {
                ObjectName name = MBeanObjectNamer.makeVirtualMachineName(XenHost.this.serverPool.getPath() + "/" + vmName,
                    uuid);
                if (!AgentCommon.getMBeanServer().isRegistered(name)) {
                    XenVirtualMachine xenVM = new XenVirtualMachine(name, XenHost.this, XenHost.this.connection, newVM, null);
                    XenHost.this.addVM(xenVM);
                    AgentCommon.getMBeanServer().registerMBean(xenVM, name);
                    XenHost.this.emitNotification(NotificationTypes.VM_ADD, "Created", name);
                }
                return this.lookUpByUuid(uuid);
            }
        } catch (Exception ex) {
            throw XenVirtualMachine.translateXenAPIException(ex);
        }
    }

    public long getTotalMemoryMB() throws VMMException {
        synchronized (this.connection) {
            try {
                return this.host.getMetrics(this.connection).getMemoryTotal(this.connection) / (1024 * 1024);
            } catch (Exception ex) {
                throw XenVirtualMachine.translateXenAPIException(ex);
            }
        }
    }

    public float[] getLoadPerCPU() throws VMMException {
        float[] result = new float[0];
        synchronized (this.connection) {
            try {
                Set<HostCpu> cpus = this.host.getHostCPUs(this.connection);
                result = new float[cpus.size()];
                int i = 0;
                for (HostCpu cpu : cpus) {
                    result[i++] = (float) (cpu.getUtilisation(this.connection) * 100);
                }
            } catch (Exception ex) {
                throw XenVirtualMachine.translateXenAPIException(ex);
            }
        }
        return result;
    }

    public float getCPULoad() throws VMMException {
        float aggregatedCPULoad = 0;
        for (XenVirtualMachine vm : this.vmList) {
            aggregatedCPULoad += vm.getCPULoad();
        }
        return aggregatedCPULoad;
    }

    public Map<String, Float> getVMCPULoads() throws VMMException {
        HashMap<String, Float> cpuLoads = new HashMap<String, Float>();
        for (XenVirtualMachine vm : this.vmList) {
            cpuLoads.put(vm.getNameLabel(), vm.getCPULoad());
        }
        return cpuLoads;
    }

    @Override
    public MBeanNotificationInfo[] getNotificationInfo() {
        return new MBeanNotificationInfo[] {new MBeanNotificationInfo(new String[] {NotificationTypes.VM_STATE_CHANGE,
            NotificationTypes.VM_ADD, NotificationTypes.VM_DEL, NotificationTypes.VM_INVENTORY_DEL, NotificationTypes.ERROR,
            NotificationTypes.LOG}, Notification.class.getName(), "Host event")};
    }

    public Map<String, ResourceUsage> getVMResourceUsage() throws VMMException {
        HashMap<String, ResourceUsage> result = new HashMap<String, ResourceUsage>();
        for (XenVirtualMachine vm : this.vmList) {
            result.put(vm.getNameLabel(), vm.getResourceUsage());
        }
        return result;
    }

    @Override
    public Map<String, String> getAttributes() {
        return this.hostAttributes;
    }

    @Override
    public void removeFromInventory() {
        this.connection.dispose();
        while (this.getResidentVMs().size() > 0) {
            this.removeVMFromInventory((XenVirtualMachine) this.getResidentVMs().get(0));
        }
    }

    /*
     * (non-Javadoc)
     * @see org.ow2.jasmine.vmm.api.HostMXBean#getHostState()
     */
    public HostPowerState getHostState() throws VMMException {
        return this.hostPowerState;
    }

    @Override
    public void start() {
        // TODO Auto-generated method stub
        this.hostPowerState = HostPowerState.RUNNING;
    }

    @Override
    public void stop() {
        // TODO Auto-generated method stub
        this.hostPowerState = HostPowerState.HALTED;
    }
}
