/**
 * JASMINe VMMapi: JASMINe Virtual Machine Management API
 * Copyright (C) 2009-2010 France Telecom R&D
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: XenServerPool.java 7456 2011-01-19 23:05:51Z dangtran $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.vmm.agent.driver.xenapi;

import java.net.URL;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import javax.management.ObjectName;

import org.ow2.jasmine.vmm.agent.domain.ServerPool;
import org.ow2.jasmine.vmm.agent.jmx.MBeanObjectNamer;
import org.ow2.jasmine.vmm.agent.main.AgentCommon;
import org.ow2.jasmine.vmm.api.HostMXBean;
import org.ow2.jasmine.vmm.api.NotificationTypes;
import org.ow2.jasmine.vmm.api.VirtualMachineImageStoreMXBean;

import com.xensource.xenapi.APIVersion;
import com.xensource.xenapi.Connection;
import com.xensource.xenapi.Event;
import com.xensource.xenapi.Host;
import com.xensource.xenapi.Pool;
import com.xensource.xenapi.Session;
import com.xensource.xenapi.Types.HostAllowedOperations;
import com.xensource.xenapi.Types.VmOperations;
import com.xensource.xenapi.VM;

/**
 * The XenServerPool is the XenAPI-driver-specific implementation of the
 * ServerPoolMXBean interface.
 * <p>
 * The XenServerPool accepts the following properties:
 * <ul>
 * <li>user : XenAPI default login
 * <li>password : XenAPI default password
 * <li>port : XenAPI default port
 * </ul>
 * These properties will be used to connect to each host member of the
 * XenServerPool.
 */
public class XenServerPool extends ServerPool {
    private int port;

    private String user;

    private String password;

    private String master;

    private VirtualMachineImageStoreMXBean imageStore;

    private String sharedStorageRepository;

    private Pool pool;

    private Connection masterCon;

    protected EventCollector eventCollector;

    public XenServerPool(final String name, final ObjectName objectName, final Map<String, String> attributes)
        throws IllegalArgumentException {
        super(name, objectName, attributes);
        String port = attributes.get("port");
        if (port == null) {
            this.port = 9363;
        } else {
            try {
                this.port = Integer.parseInt(port);
            } catch (NumberFormatException ex) {
                throw new IllegalArgumentException("Wrong value for port");
            }
        }
        this.user = attributes.get("user");
        this.password = attributes.get("password");

        this.sharedStorageRepository = attributes.get("sharedStorageRepository");
        if (this.sharedStorageRepository == null) {
            throw new IllegalArgumentException("Missing parameter: sharedStorageRepository");
        }
        this.master = attributes.get("master");
        try {
            this.masterCon = new Connection(new URL("http://" + this.master));
            Session.loginWithPassword(this.masterCon, this.user, this.password, APIVersion.latest().toString());
            this.eventCollector = new EventCollector(this.masterCon);
            Set<Pool> pools = Pool.getAll(this.masterCon);
            for (Pool p : pools) {
                if (p.getNameLabel(this.masterCon).equals(name)) {
                    this.pool = p;
                }
            }
            if (this.pool == null) {
                throw new Exception("Pool " + name + " not found.");
            }
            this.initialize();
        } catch (Exception ex) {
            XenHost.logger.error(
                "Failed to establish Xen-API connection with host " + this.master + " with login " + this.user, ex);
        }
    }

    @Override
    public String getHypervisor() {
        return "XenServer";
    }

    public String getSharedStorageRepository() {
        return this.sharedStorageRepository;
    }

    @Override
    public HostMXBean newHost(final String hostName, final Map<String, String> attributes) throws IllegalArgumentException {
        try {
            for (HostMXBean host : this.hosts) {
                if (host.getHostName().equals(hostName)) {
                    return host;
                }
            }
            ObjectName mbeanObjectName = MBeanObjectNamer.makeHostName(this.getPath() + "/" + hostName, hostName);

            XenHost host = XenHost.newHost(this, mbeanObjectName, hostName, this.user, this.password, this.port,
                this.masterCon, attributes);
            if (host != null) {
                AgentCommon.getMBeanServer().registerMBean(host, mbeanObjectName);
                this.hosts.add(host);
                ServerPool.logger.info("In server pool " + this.servername + ": added Host MBean " + mbeanObjectName);

                if (this.imageStore == null && this.sharedStorageRepository != null) {
                    ObjectName sharedImageStoreName = MBeanObjectNamer.makeVMImageStoreName(this.sharedStorageRepository);
                    try {
                        this.imageStore = new XenVMImageStore(sharedImageStoreName, this.sharedStorageRepository, host);
                        AgentCommon.getMBeanServer().registerMBean(this.imageStore, sharedImageStoreName);
                    } catch (Exception ex) {
                        ServerPool.logger.error(ex);
                    }
                }
            }
            return host;
        } catch (Exception ex) {
            ServerPool.logger.error("Failed to add XenHost", ex);
            return null;
        }
    }

    @Override
    public VirtualMachineImageStoreMXBean getVMImageStore() {
        return this.imageStore;
    }

    public void initialize() {
        new Thread(this.eventCollector).start();
    }

    XenHost lookUpHostByVMUuid(final String uuid) {
        for (HostMXBean host : this.hosts) {
            XenHost h = (XenHost) host;
            XenVirtualMachine vm = h.lookUpByUuid(uuid);
            if (vm != null) {
                return h;
            }
        }
        return null;
    }

    XenHost lookUpHostByUuid(final String uuid) {
        for (HostMXBean host : this.hosts) {
            if (((XenHost) host).getUuid().equals(uuid)) {
                return (XenHost) host;
            }
        }
        return null;
    }

    private class EventCollector implements Runnable {
        volatile boolean stop = false;

        Connection connection;

        EventCollector(final Connection connection) {
            this.connection = connection;
        }

        @Override
        public void run() {
            HashSet<String> eventClasses = new HashSet<String>();
            eventClasses.add("vm");
            eventClasses.add("host");
            try {
                Event.register(this.connection, eventClasses);
            } catch (Exception ex) {
                ServerPool.logger.error("Cannot register XenAPI session with event system", ex);
                return;
            }
            while (!this.stop) {
                Set<Event.Record> eventSet;
                try {
                    eventSet = Event.next(this.connection);
                } catch (Exception ex) {
                    ServerPool.logger.error("Cannot retrieve next event", ex);
                    continue;
                }
                synchronized (this) {
                    for (Event.Record event : eventSet) {
                        Object snapshot = event.snapshot;
                        // ServerPool.logger.debug("Event: clazz=" + event.clazz
                        // + " op=" + event.operation);
                        // ServerPool.logger.debug(event.toString());
                        if (snapshot instanceof VM.Record) {
                            VM.Record vmRecord = (VM.Record) event.snapshot;
                            Map<String, VmOperations> operations = vmRecord.currentOperations;
                            if (operations.isEmpty()) {
                                continue;
                            }
                            VmOperations op = operations.values().iterator().next();
                            String vmName = vmRecord.isControlDomain ? "Domain-0" : vmRecord.nameLabel;
                            // ServerPool.logger.debug("Event: clazz=" +
                            // event.clazz + " op=" + event.operation +
                            // " vmName="
                            // + vmName);
                            switch (event.operation) {
                            case ADD: {
                                try {
                                    VM vm = VM.getByUuid(this.connection, vmRecord.uuid);
                                    XenHost h = XenServerPool.this.lookUpHostByUuid(vm.getAffinity(this.connection).getUuid(
                                        this.connection));
                                    ObjectName name = MBeanObjectNamer.makeVirtualMachineName(XenServerPool.this.getPath()
                                        + "/" + vmName, vmRecord.uuid);
                                    if (!AgentCommon.getMBeanServer().isRegistered(name)) {
                                        XenVirtualMachine xenVM = new XenVirtualMachine(name, h, this.connection, vm, null);
                                        h.addVM(xenVM);
                                        AgentCommon.getMBeanServer().registerMBean(xenVM, name);
                                        h.emitNotification(NotificationTypes.VM_ADD, "Created", name);
                                    }
                                } catch (Exception ex) {
                                    ServerPool.logger.error(ex);
                                }
                                break;
                            }
                            case DEL: {
                                XenHost h = XenServerPool.this.lookUpHostByVMUuid(vmRecord.uuid);
                                XenVirtualMachine vm = h.lookUpByUuid(vmRecord.uuid);
                                if (vm != null) {
                                    h.removeVM(vm);
                                    h.emitNotification(NotificationTypes.VM_DEL, "Destroyed", vm.getObjectName());
                                    try {
                                        AgentCommon.getMBeanServer().unregisterMBean(vm.getObjectName());
                                    } catch (Exception ex) {
                                        ServerPool.logger.error(ex);
                                    }
                                }
                                break;
                            }
                            case MOD: {
                                XenHost h = XenServerPool.this.lookUpHostByVMUuid(vmRecord.uuid);
                                XenVirtualMachine vm = null;
                                if (h != null) {
                                    vm = h.lookUpByUuid(vmRecord.uuid);
                                }
                                switch (op) {
                                case CLONE:

                                    break;
                                case START:
                                case CLEAN_SHUTDOWN:
                                case HARD_SHUTDOWN:
                                case PAUSE:
                                case RESUME:
                                    if (vm != null) {
                                        vm.updatePowerState(vmRecord.powerState);
                                    }
                                    break;
                                case POOL_MIGRATE:
                                    Host dest = vmRecord.residentOn;
                                    try {
                                        if (!dest.getUuid(this.connection).equals(h.getUuid())) {
                                            XenHost dstHost = XenServerPool.this
                                                .lookUpHostByUuid(dest.getUuid(this.connection));
                                            h.postMigrateVM(vm, dstHost);
                                            vm.emitNotification(NotificationTypes.VM_MIGRATION, dstHost.getHostName(),
                                                vm.getUuid());
                                        }
                                    } catch (Exception ex) {
                                        ServerPool.logger.error(ex);
                                    }
                                    break;
                                }

                                break;
                            }
                            }

                        } else if (snapshot instanceof Host.Record) {
                            Host.Record hostRecord = (Host.Record) event.snapshot;
                            Map<String, HostAllowedOperations> operations = hostRecord.currentOperations;
                            if (operations.isEmpty()) {
                                continue;
                            }
                            HostAllowedOperations op = operations.values().iterator().next();
                        }
                    }
                }

            }

        }
    }

    @Override
    public void deleteImageStore() {
        try {
            int i = this.imageStore.listVMImageTemplates().size();
            while (i > 0) {
                AgentCommon.getMBeanServer().unregisterMBean(this.imageStore.listVMImageTemplates().get(i - 1).getObjectName());
                i--;
            }
        } catch (Throwable e) {
            ServerPool.logger.error(e);
        }
        try {
            AgentCommon.getMBeanServer().unregisterMBean(this.imageStore.getObjectName());
        } catch (Exception e) {
            ServerPool.logger.error(e);
        }
        this.imageStore = null;
    }
}
