/**
 * JASMINe VMMapi: JASMINe Virtual Machine Management API
 * Copyright (C) 2009 France Telecom R&D
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: XenVMImageStore.java 6098 2010-02-23 13:50:28Z dangtran $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.vmm.agent.driver.xenapi;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.management.ObjectName;

import org.apache.log4j.Logger;
import org.ow2.jasmine.vmm.agent.domain.ManagedResource;
import org.ow2.jasmine.vmm.agent.jmx.MBeanObjectNamer;
import org.ow2.jasmine.vmm.agent.main.AgentCommon;
import org.ow2.jasmine.vmm.api.VMMException;
import org.ow2.jasmine.vmm.api.VirtualMachineImageMXBean;
import org.ow2.jasmine.vmm.api.VirtualMachineImageStoreMXBean;

import com.xensource.xenapi.Connection;
import com.xensource.xenapi.SR;
import com.xensource.xenapi.VM;

/**
 * Xen driver VirtualMachineImageStore MXBean implementation This store is
 * implemented as a directory containing for each VM template:
 * <ul>
 * <li>a XML metadata file
 * <li>the image file (raw filesystem image)
 * </ul>
 */
public class XenVMImageStore extends ManagedResource implements VirtualMachineImageStoreMXBean {
    static Logger logger = Logger.getLogger(XenVMImageStore.class);

    private HashMap<String, XenVirtualMachineImage> vmMap = new HashMap<String, XenVirtualMachineImage>();

    private String sharedStorageRepository;

    private Connection connection;

    private SR storageRepository;

    public XenVMImageStore(final ObjectName objectName, final String sharedStorageRepository, final XenHost host)
        throws VMMException {
        super(objectName);
        this.sharedStorageRepository = sharedStorageRepository;
        this.connection = host.getXenAPIConnection();
        try {
            synchronized (this.connection) {
                Set<SR> srSet = SR.getByNameLabel(this.connection, sharedStorageRepository);
                if (srSet.isEmpty()) {
                    throw new VMMException("Cannot find storage repository " + sharedStorageRepository);
                }
                this.storageRepository = srSet.iterator().next();
            }
        } catch (Exception ex) {
            XenVMImageStore.logger.error("Failed to get shared SR " + sharedStorageRepository, ex);
            throw new VMMException(ex.toString());
        }
        this.sync();
    }

    synchronized void sync() throws VMMException {
        synchronized (this.connection) {
            try {
                Set<VM> vms = VM.getAll(this.connection);
                for (VM vm : vms) {
                    VM.Record vmRec = vm.getRecord(this.connection);
                    if (vmRec.isATemplate) {
                        if (this.vmMap.get(vmRec.uuid) == null) {
                            ObjectName on = MBeanObjectNamer.makeVMImageName(vmRec.uuid);
                            XenVirtualMachineImage image = new XenVirtualMachineImage(on, vmRec);
                            AgentCommon.getMBeanServer().registerMBean(image, image.getObjectName());
                            XenVMImageStore.logger.debug("Adding Template: " + vmRec.nameLabel);
                            this.vmMap.put(vmRec.uuid, image);
                        }
                    }
                }
            } catch (Exception ex) {
                XenVMImageStore.logger.error("In ImageStore.sync:", ex);
            }
        }
    }

    public void removeVMImageTemplate(final VirtualMachineImageMXBean vmImage) throws VMMException {
        if (vmImage instanceof XenVirtualMachineImage) {
            XenVirtualMachineImage img = (XenVirtualMachineImage) vmImage;
            if (img.getImageStore() == this) {
                // XXX destroy template
                this.vmMap.remove(img.getUUID());
            }
        }
    }

    public VirtualMachineImageMXBean lookUpByName(final String name) {
        for (XenVirtualMachineImage image : this.vmMap.values()) {
            if (image.getName().equals(name)) {
                return image;
            }
        }
        return null;
    }

    public VirtualMachineImageMXBean lookUpByUUID(final String uuid) {
        return this.vmMap.get(uuid);
    }

    public List<VirtualMachineImageMXBean> listVMImageTemplates() {
        ArrayList<VirtualMachineImageMXBean> result = new ArrayList<VirtualMachineImageMXBean>(this.vmMap.values());
        return result;
    }

    public long getCapacityMB() throws VMMException {
        try {
            synchronized (this.connection) {
                return this.storageRepository.getPhysicalSize(this.connection) / (1024 * 1024);
            }
        } catch (Exception ex) {
            throw XenVirtualMachine.translateXenAPIException(ex);
        }
    }

    public long getFreeSpaceMB() throws VMMException {
        try {
            synchronized (this.connection) {
                return this.getCapacityMB() - this.storageRepository.getPhysicalUtilisation(this.connection) / (1024 * 1024);
            }
        } catch (Exception ex) {
            throw XenVirtualMachine.translateXenAPIException(ex);
        }
    }

    public String getName() {
        return "ImageStore(" + this.sharedStorageRepository + ")";
    }

    class XenVirtualMachineImage extends ManagedResource implements VirtualMachineImageMXBean {
        private String name, uuid, description;

        private Map<String, String> metadata;

        private XenVirtualMachineImage(final ObjectName objectName, final VM.Record vm) throws Exception {
            super(objectName);
            this.name = vm.nameLabel;
            this.description = vm.nameDescription;
            this.uuid = vm.uuid;
            this.metadata = vm.otherConfig;
        }

        XenVMImageStore getImageStore() {
            return XenVMImageStore.this;
        }

        public Map<String, String> getMetaData() {
            return this.metadata;
        }

        public String getName() {
            return this.name;
        }

        public String getDescription() {
            return this.description;
        }

        public String getUUID() {
            return this.uuid;
        }

    }

}
