/**
 * JASMINe VMMapi: JASMINe Virtual Machine Management API
 * Copyright (C) 2009-2010 France Telecom R&D
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: AgentControl.java 6901 2010-09-15 12:17:55Z mleger $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.vmm.agent.main;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.util.Properties;

import javax.management.JMX;
import javax.management.MBeanServerConnection;
import javax.management.ObjectName;
import javax.management.remote.JMXConnector;
import javax.management.remote.JMXConnectorFactory;
import javax.management.remote.JMXServiceURL;

import org.apache.log4j.Logger;
import org.ow2.jasmine.vmm.api.VMMAgentMXBean;

/**
 * AgentControl provides a basic CLI program to control a local or remote VMM
 * agent
 */
public class AgentControl {
    private static Logger logger = Logger.getLogger(AgentControl.class);

    public final static String vmmHome = System.getProperty("vmm.home");

    private static enum Mode {
        START, STOP;
    }

    private static final int DEFAULT_PORT = 9999;

    private static final String DEFAULT_HOST = "localhost";

    public static void main(final String[] args) {
        if (AgentControl.vmmHome == null) {
            System.out.println("vmm.home system property must be set");
            System.exit(1);
        }

        Properties properties = new Properties();
        String agentPropFileName = "etc" + File.separator + "agent.properties";
        try {
            properties.load(new FileInputStream(new File(AgentControl.vmmHome, agentPropFileName)));
        } catch (IOException ex) {
            AgentControl.logger.fatal("Cannot load VMM agent property file " + agentPropFileName, ex);
            System.exit(1);
        }

        int port = AgentControl.DEFAULT_PORT;
        try {
            port = Integer.parseInt(properties.getProperty("vmm.port"));
        } catch (NumberFormatException ex) {
            AgentControl.logger.fatal("Cannot parse vmm.port property", ex);
            System.exit(1);
        }

        String managedResourcesFileName = properties.getProperty("vmm.resourceFile", "managed-resources.xml");

        String host = AgentControl.DEFAULT_HOST;
        Mode mode = null;

        int i;
        for (i = 0; i < args.length; i++) {
            String s = args[i];
            if (!s.startsWith("-")) {
                break;
            }
            if (s.equals("-start")) {
                mode = Mode.START;
            } else if (s.equals("-stop")) {
                mode = Mode.STOP;
            } else if (s.equals("-port")) {
                if (i < args.length - 1) {
                    port = Integer.parseInt(args[++i]);
                } else {
                    AgentControl.printUsageAndExit();
                }
            } else if (s.equals("-host")) {
                if (i < args.length - 1) {
                    host = args[++i];
                } else {
                    AgentControl.printUsageAndExit();
                }
            } else {
                AgentControl.printUsageAndExit();
            }
        }
        if (mode == null) {
            AgentControl.printUsageAndExit();
        }
        if (mode == Mode.START) {
            new VirtManagerAgent(port, managedResourcesFileName);
        } else if (mode == Mode.STOP) {

            try {
                JMXServiceURL url = new JMXServiceURL("service:jmx:rmi:///jndi/rmi://" + host + ":" + port + "/server");
                JMXConnector jmxc = JMXConnectorFactory.connect(url, null);
                MBeanServerConnection mbsc = jmxc.getMBeanServerConnection();
                ObjectName agentObjectName = new ObjectName("org.ow2.jasmine.vmm.api:type=Agent");
                VMMAgentMXBean agentMBean = JMX.newMXBeanProxy(mbsc, agentObjectName, VMMAgentMXBean.class);

                agentMBean.shutdown();

                jmxc.close();
            } catch (Exception ex) {
                ex.printStackTrace();
            }
        }
    }

    private static void printUsageAndExit() {
        System.out.println("Usage: java org.ow2.jasmine.vmm.agent.main.AgentControl -start|stop [-port <port>] [-host <host>]");
        System.exit(1);
    }

    public static String getManagedResourcesFileName() {
        Properties properties = new Properties();
        String agentPropFileName = "etc" + File.separator + "agent.properties";
        try {
            properties.load(new FileInputStream(new File(AgentControl.vmmHome, agentPropFileName)));
        } catch (IOException ex) {
            AgentControl.logger.fatal("Cannot load VMM agent property file " + agentPropFileName, ex);
            System.exit(1);
        }

        String managedResourcesFileName = properties.getProperty("vmm.resourceFile", "managed-resources.xml");
        return managedResourcesFileName;
    }
}
