/**
 * JASMINe VMMapi: JASMINe Virtual Machine Management API
 * Copyright (C) 2009 France Telecom R&D
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: VirtManagerAgent.java 7452 2011-01-19 22:47:52Z dangtran $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.vmm.agent.main;

import java.io.File;
import java.io.IOException;
import java.rmi.RemoteException;
import java.rmi.registry.LocateRegistry;

import javax.management.ObjectName;
import javax.management.remote.JMXConnectorServer;
import javax.management.remote.JMXConnectorServerFactory;
import javax.management.remote.JMXServiceURL;

import org.apache.log4j.Logger;
import org.ow2.jasmine.vmm.agent.domain.Domain;
import org.ow2.jasmine.vmm.agent.domain.DomainBuilder;
import org.ow2.jasmine.vmm.agent.driver.Driver;
import org.ow2.jasmine.vmm.agent.driver.DriverFactory;
import org.ow2.jasmine.vmm.agent.jmx.MBeanObjectNamer;
import org.ow2.jasmine.vmm.agent.main.persistence.ConfigurationGeneratorException;
import org.ow2.jasmine.vmm.agent.main.persistence.RessourceConfigurationGenerator;
import org.ow2.jasmine.vmm.api.DomainMXBean;
import org.ow2.jasmine.vmm.api.VMMAgentMXBean;
import org.ow2.jasmine.vmm.api.VMMException;

/**
 * VirtManagerAgent implements the VMMAgentMXBean interface and is the entry
 * point of the agent. VirtManagerAgent is a singleton. VirtManagerAgent
 * expects:
 * <ul>
 * <li>the vmm.home property to be set to the directory where the VMM agent
 * distribution is installed
 * <li>the etc/agent.properties file to contains agent properties
 * </ul>
 */
public class VirtManagerAgent extends AgentCommon implements VMMAgentMXBean {
    private static Logger logger = Logger.getLogger(VirtManagerAgent.class);

    static VirtManagerAgent instance;

    private JMXConnectorServer cs;

    Domain rootDomain;

    static public VirtManagerAgent getInstance() {
        return VirtManagerAgent.instance;
    }

    /*
     * @see org.ow2.jasmine.vmm.api.VMMAgentMXBean#getRootDomain()
     */
    public DomainMXBean getRootDomain() {
        return this.rootDomain;
    }

    public VirtManagerAgent(final int port, final String managedResourcesFileName) {
        VirtManagerAgent.instance = this;

        ObjectName agentName = MBeanObjectNamer.makeAgentName();
        try {
            AgentCommon.mbs.registerMBean(this, agentName);
        } catch (Exception ex) {
            VirtManagerAgent.logger.fatal("Cannot register MXBean " + agentName, ex);
            System.exit(1);
        }
        VirtManagerAgent.logger.info("Created VMMAgent MBean " + agentName);

        File managedResourcesFile = new File(AgentControl.vmmHome, "etc" + File.separator + managedResourcesFileName);

        this.rootDomain = DomainBuilder.createManagedResourcesFromXML(managedResourcesFile);
        if (this.rootDomain == null) {
            VirtManagerAgent.logger.fatal("Failed to load resource file");
            System.exit(1);
        }

        try {
            LocateRegistry.createRegistry(port);
        } catch (RemoteException ex) {
            VirtManagerAgent.logger.warn("Cannot launch RMI registry on port " + port + ", creating new one");
            try {
                LocateRegistry.createRegistry(port);
            } catch (RemoteException ex2) {
                VirtManagerAgent.logger.fatal("Cannot create RMI registry", ex);
                System.exit(1);
            }
        }
        try {
            JMXServiceURL url = new JMXServiceURL("service:jmx:rmi:///jndi/rmi://localhost:" + port + "/server");
            this.cs = JMXConnectorServerFactory.newJMXConnectorServer(url, null, AgentCommon.mbs);
            VirtManagerAgent.logger.info("Starting RMI connector server with URL \"" + url + "\"");
            this.cs.start();
            VirtManagerAgent.logger.info("VMM agent ready");
        } catch (Exception ex) {
            VirtManagerAgent.logger.fatal("Failed to start JMX connector server", ex);
            System.exit(1);
        }

        synchronized (this) {
            try {
                this.wait();
            } catch (InterruptedException e) {
            }
        }
        try {
            this.cs.stop();
        } catch (IOException e) {
            VirtManagerAgent.logger.error(e.getMessage(), e);
        }
        VirtManagerAgent.logger.info("VMM agent terminated");
        System.exit(0);
    }

    public synchronized void saveConfiguration() {
        VirtManagerAgent.logger.info("Saving XML configuration...");
        RessourceConfigurationGenerator newXML = null;
        try {
            newXML = new RessourceConfigurationGenerator(AgentControl.vmmHome + File.separator + "etc");
        } catch (ConfigurationGeneratorException ex) {
            VirtManagerAgent.logger.error(ex);
            return;
        }
        try {
            newXML.generate(this.rootDomain);
            VirtManagerAgent.logger.info("XML Generation OK");
        } catch (ConfigurationGeneratorException ex) {
            VirtManagerAgent.logger.error(ex);
        }
    }

    /*
     * @see org.ow2.jasmine.vmm.api.VMMAgentMXBean#shutdown()
     */
    public void shutdown() {
        synchronized (this) {
            this.notifyAll();
        }
    }

    public String[] getServerPoolCreationAttributes(final String driverName) throws VMMException {
        Driver driver = DriverFactory.getInstance().newDriver(driverName);
        if (driver == null) {
            throw new VMMException("Unsupported driver " + driverName);
        }
        return driver.getServerPoolCreationAttributes();
    }
}