/**
 * JASMINe VMMapi: JASMINe Virtual Machine Management API
 * Copyright (C) 2010 France Telecom R&D
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: RessourceConfigurationGenerator.java 7452 2011-01-19 22:47:52Z dangtran $
 * --------------------------------------------------------------------------
 */

package org.ow2.jasmine.vmm.agent.main.persistence;

import java.io.File;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.rmi.RemoteException;
import java.text.DateFormat;
import java.text.SimpleDateFormat;

import org.apache.log4j.Logger;
import org.ow2.jasmine.vmm.agent.domain.AbstractHost;
import org.ow2.jasmine.vmm.agent.domain.Domain;
import org.ow2.jasmine.vmm.agent.domain.ServerPool;
import org.ow2.jasmine.vmm.agent.main.AgentControl;
import org.ow2.jasmine.vmm.api.DomainMXBean;
import org.ow2.jasmine.vmm.api.HostMXBean;
import org.ow2.jasmine.vmm.api.ServerPoolMXBean;
import org.ow2.jasmine.vmm.api.VMMException;
import org.xml.sax.ContentHandler;
import org.xml.sax.SAXException;
import org.xml.sax.helpers.AttributesImpl;

public class RessourceConfigurationGenerator extends AbstractConfigurationGenerator {

    private static Logger logger = Logger.getLogger(AgentControl.class);

    protected boolean vmGeneration = false;

    private String xmlFileName;

    public RessourceConfigurationGenerator(final String baseDir) throws ConfigurationGeneratorException {
        this(baseDir, false);
    }

    public RessourceConfigurationGenerator(final String baseDir, final boolean vmGeneration)
        throws ConfigurationGeneratorException {
        super(baseDir);
        this.vmGeneration = vmGeneration;
        this.xmlFileName = null;
    }

    /*
     * (non-Javadoc)
     * @see
     * vmscript.configuration.parsing.ConfigurationGenerator#generate(java.lang
     * .Object)
     */
    public void generate(final Object obj) throws ConfigurationGeneratorException {
        Domain domain = null;
        if (obj instanceof Domain) {
            domain = (Domain) obj;
        } else {
            throw new ConfigurationGeneratorException("Wrong object type.");
        }
        try {
            if (this.xmlFileName == null) {
                this.xmlFileName = AgentControl.getManagedResourcesFileName();
            }
            RessourceConfigurationGenerator.logger.info("Name of the XML : " + this.xmlFileName);
            this.backupFile(this.xmlFileName);
            ContentHandler hd = this.getHandler(this.xmlFileName);
            hd.startDocument();
            this.generateDomain(domain, hd);
            hd.endDocument();
        } catch (Exception e) {
            throw new ConfigurationGeneratorException(e);
        }
    }

    private void backupFile(final String fileName) {
        File inputFile = new File(this.baseDir + fileName);
        File outputFile = new File(this.baseDir + fileName + ".bak");

        try {
            FileReader in = new FileReader(inputFile);
            FileWriter out = new FileWriter(outputFile);
            int c;

            while ((c = in.read()) != -1) {
                out.write(c);
            }

            in.close();
            out.close();
        } catch (IOException ex) {
            RessourceConfigurationGenerator.logger.error("Failed to back up configuration file", ex);
        }
    }

    public void generate(final Object obj, final String xmlName) throws ConfigurationGeneratorException {
        this.xmlFileName = xmlName;
        this.generate(obj);
    }

    /**
     * @param domain
     * @param hd
     * @throws SAXException
     * @throws RemoteException
     */
    protected void generateDomain(final Domain domain, final ContentHandler hd) throws SAXException, RemoteException {
        AttributesImpl atts = new AttributesImpl();
        atts.addAttribute("", "", "name", "CDATA", domain.getName());
        hd.startElement("", "", "domain", atts);
        for (DomainMXBean subDomain : domain.getSubDomains()) {
            this.generateDomainMBean(subDomain, hd);
        }
        for (ServerPoolMXBean serverPool : domain.getServerPools()) {
            this.generateServerMBean(serverPool, hd);
        }
        hd.endElement("", "", "domain");
    }

    /**
     * @param domainMBean
     * @param hd
     * @throws SAXException
     * @throws RemoteException
     */
    protected void generateDomainMBean(final DomainMXBean domainMBean, final ContentHandler hd) throws SAXException,
        RemoteException {
        AttributesImpl atts = new AttributesImpl();
        atts.addAttribute("", "", "name", "CDATA", domainMBean.getName());
        hd.startElement("", "", "domain", atts);
        for (DomainMXBean subDomain : domainMBean.getSubDomains()) {
            this.generateDomainMBean(subDomain, hd);
        }
        for (ServerPoolMXBean serverPoolMBean : domainMBean.getServerPools()) {
            this.generateServerMBean(serverPoolMBean, hd);
        }
        hd.endElement("", "", "domain");
    }

    /**
     * @param ServerPoolMXBean
     * @param hd
     * @throws SAXException
     * @throws RemoteException
     */
    protected void generateServerMBean(final ServerPoolMXBean serverPoolMBean, final ContentHandler hd) throws SAXException,
        RemoteException {
        AttributesImpl att = new AttributesImpl();
        att.addAttribute("", "", "name", "CDATA", serverPoolMBean.getName());
        String driver = ((ServerPool) serverPoolMBean).getAttributes().get("driver");
        att.addAttribute("", "", "driver", "CDATA", driver);
        if (driver.equals("dummy")) {

        } else if (driver.equals("libvirt")) {
            att.addAttribute("", "", "hypervisor", "CDATA", ((ServerPool) serverPoolMBean).getAttributes().get("hypervisor"));
            att.addAttribute("", "", "sharedImageStore", "CDATA",
                ((ServerPool) serverPoolMBean).getAttributes().get("sharedImageStore"));
            att.addAttribute("", "", "sharedDiskStore", "CDATA",
                ((ServerPool) serverPoolMBean).getAttributes().get("sharedDiskStore"));
            if (((ServerPool) serverPoolMBean).getAttributes().containsKey("sshRootPassword")) {
                att.addAttribute("", "", "sshRootPassword", "CDATA",
                    ((ServerPool) serverPoolMBean).getAttributes().get("sshRootPassword"));
            }
            if (((ServerPool) serverPoolMBean).getAttributes().containsKey("sshPrivateKeyFile")) {
                att.addAttribute("", "", "sshPrivateKeyFile", "CDATA",
                    ((ServerPool) serverPoolMBean).getAttributes().get("sshPrivateKeyFile"));
            }
            if (((ServerPool) serverPoolMBean).getAttributes().containsKey("syncPeriodMillis")) {
                att.addAttribute("", "", "syncPeriodMillis", "CDATA",
                    ((ServerPool) serverPoolMBean).getAttributes().get("syncPeriodMillis"));
            }
        } else if (driver.equals("xenapi")) {
            att.addAttribute("", "", "sharedStorageRepository", "CDATA",
                ((ServerPool) serverPoolMBean).getAttributes().get("sharedStorageRepository"));
            att.addAttribute("", "", "master", "CDATA", ((ServerPool) serverPoolMBean).getAttributes().get("master"));
            if (((ServerPool) serverPoolMBean).getAttributes().containsKey("user")) {
                att.addAttribute("", "", "user", "CDATA", ((ServerPool) serverPoolMBean).getAttributes().get("user"));
            }
            if (((ServerPool) serverPoolMBean).getAttributes().containsKey("root")) {
                att.addAttribute("", "", "root", "CDATA", ((ServerPool) serverPoolMBean).getAttributes().get("root"));
            }
        } else if (driver.equals("vmware-vi")) {
            att.addAttribute("", "", "vCenterHostName", "CDATA",
                ((ServerPool) serverPoolMBean).getAttributes().get("vCenterHostName"));
            att.addAttribute("", "", "user", "CDATA", ((ServerPool) serverPoolMBean).getAttributes().get("user"));
            att.addAttribute("", "", "password", "CDATA", ((ServerPool) serverPoolMBean).getAttributes().get("password"));
            att.addAttribute("", "", "datacenter", "CDATA", ((ServerPool) serverPoolMBean).getAttributes().get("datacenter"));
            att.addAttribute("", "", "datastore", "CDATA", ((ServerPool) serverPoolMBean).getAttributes().get("datastore"));
            att.addAttribute("", "", "vmFolderPath", "CDATA", ((ServerPool) serverPoolMBean).getAttributes()
                .get("vmFolderPath"));
            att.addAttribute("", "", "vmTemplateFolderPath", "CDATA",
                ((ServerPool) serverPoolMBean).getAttributes().get("vmTemplateFolderPath"));
            if (((ServerPool) serverPoolMBean).getAttributes().containsKey("network")) {
                att.addAttribute("", "", "network", "CDATA", ((ServerPool) serverPoolMBean).getAttributes().get("network"));
            }
        } else if (driver.equals("hyperv")) {
            att.addAttribute("", "", "vmFolderPath", "CDATA", ((ServerPool) serverPoolMBean).getAttributes()
                .get("vmFolderPath"));
            att.addAttribute("", "", "vmTemplateFolderPath", "CDATA",
                ((ServerPool) serverPoolMBean).getAttributes().get("vmTemplateFolderPath"));
            if (((ServerPool) serverPoolMBean).getAttributes().containsKey("user")) {
                att.addAttribute("", "", "user", "CDATA", ((ServerPool) serverPoolMBean).getAttributes().get("user"));
            }
            if (((ServerPool) serverPoolMBean).getAttributes().containsKey("password")) {
                att.addAttribute("", "", "password", "CDATA", ((ServerPool) serverPoolMBean).getAttributes().get("root"));
            }
            if (((ServerPool) serverPoolMBean).getAttributes().containsKey("lagacyNetworkAdapter")) {
                att.addAttribute("", "", "lagacyNetworkAdapter", "CDATA",
                    ((ServerPool) serverPoolMBean).getAttributes().get("lagacyNetworkAdapter"));
            }
            if (((ServerPool) serverPoolMBean).getAttributes().containsKey("syncPeriodMillis")) {
                att.addAttribute("", "", "syncPeriodMillis", "CDATA",
                    ((ServerPool) serverPoolMBean).getAttributes().get("syncPeriodMillis"));
            }
        }
        hd.startElement("", "", "serverPool", att);
        for (HostMXBean host : ((ServerPool) serverPoolMBean).getManagedHosts()) {
            this.generateHost(host, hd, driver);
        }
        hd.endElement("", "", "serverPool");
    }

    /**
     * @param hostMBean
     * @param hd
     * @throws SAXException
     * @throws RemoteException
     */
    protected void generateHost(final HostMXBean hostMBean, final ContentHandler hd, final String driver) throws SAXException,
        RemoteException {
        AttributesImpl atts = new AttributesImpl();
        AbstractHost host = ((AbstractHost) hostMBean);
        try {
            atts.addAttribute("", "", "name", "CDATA", hostMBean.getHostName());
        } catch (VMMException e) {
            e.printStackTrace();
        }
        if (driver.equals("dummy")) {

        } else if (driver.equals("libvirt")) {
            atts.addAttribute("", "", "url", "CDATA", host.getAttributes().get("url"));
            if (host.getAttributes().containsKey("sshRootPassword")) {
                atts.addAttribute("", "", "sshRootPassword", "CDATA", host.getAttributes().get("sshRootPassword"));
            }
            if (host.getAttributes().containsKey("sshPrivateKeyFile")) {
                atts.addAttribute("", "", "sshPrivateKeyFile", "CDATA", host.getAttributes().get("sshPrivateKeyFile"));
            }
        } else if (driver.equals("xenapi")) {
            if (host.getAttributes().containsKey("user")) {
                atts.addAttribute("", "", "user", "CDATA", host.getAttributes().get("user"));
            }
            if (host.getAttributes().containsKey("root")) {
                atts.addAttribute("", "", "root", "CDATA", host.getAttributes().get("root"));
            }
        } else if (driver.equals("vmware-vi")) {

        } else if (driver.equals("hyperv")) {

            if (host.getAttributes().containsKey("user")) {
                atts.addAttribute("", "", "user", "CDATA", host.getAttributes().get("user"));
            }
            if (host.getAttributes().containsKey("password")) {
                atts.addAttribute("", "", "password", "CDATA", host.getAttributes().get("root"));
            }
        }
        hd.startElement("", "", "host", atts);

        hd.endElement("", "", "host");
    }

    public static class Dates {
        // * Definition du format utilise pour les dates
        static DateFormat dateFormat = new SimpleDateFormat("dd-MM-yyyy_HH:mm:ss");
    }
}
