/**
 * JASMINe VMMapi: JASMINe Virtual Machine Management API
 * Copyright (C) 2009 France Telecom R&D
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: HostMXBean.java 3210 2009-03-31 11:37:38Z chahinem $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.vmm.api;

import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * Management interface of a physical host acting as a container of virtual
 * machines by way of a hypervisor.
 * <p>
 * The <tt>ObjectName</tt> for identifying a HostMXBean follows the following
 * template: <blockquote>
 * org.ow2.jasmine.vmm.api:type=Host,name=&lt;pathname&gt;
 * ,hostname=&lt;hostname&gt; </blockquote>
 * <p>
 * The notifications that can be emitted by a HostMXBean are as follows
 * <ul>
 * <li>NotificationTypes.VM_NEW when a new VM is created
 * <li>NotificationTypes.VM_DEL when a VM is deleted
 * <li>NotificationTypes.PERF_REPORT: periodic performance reports
 * </ul>
 */
public interface HostMXBean extends ManagedResourceMXBean {
    /**
     * Defines the different performance metrics that can be retrieved from a
     * host
     */
    public enum PerfMetric {
        /**
         * Aggregated CPU load per VM
         */
        VM_CPU_LOAD,
        /**
         * Aggregated Network traffic per VM
         */
        VM_NETWORK_TRAFFIC,
        /**
         * Aggregated disk traffic per VM
         */
        VM_DISK_TRAFFIC
    };

    /**
     * Returns the server pool this host belongs to
     * 
     * @return
     */
    ServerPoolMXBean getServerPool();

    /**
     * Returns the DNS name of the host
     * 
     * @return
     */
    String getHostName();

    /**
     * Returns properties of the hypervisor active for this host
     * 
     * @return
     */
    Map<String, String> getHypervisorInfo();

    /**
     * Return the number of CPU cores on the host
     * 
     * @return
     */
    int getNumCPU();

    /**
     * Returns the information about the CPU on this host. The following
     * properties are returned: model, speed and vendor.
     * 
     * @return
     */
    Map<String, String> getCPUInfo();

    /**
     * Returns the VMs resident on this host
     * 
     * @return
     */
    List<VirtualMachineMXBean> getResidentVMs();

    /**
     * Returns the total memory capacity of this host in MBytes
     * 
     * @return
     */
    long getTotalMemoryMB();

    /**
     * Returns the free memory capacity of this host in MBytes
     * 
     * @return
     */
    long getFreeMemoryMB();

    /**
     * Returns the current load on each CPU core of the host
     * 
     * @return
     */
    float[] getLoadPerCPU();

    /**
     * Returns the aggregate CPU load of the host
     * 
     * @return
     */
    float getCPULoad();

    /**
     * Return the CPU load of every VM hosted on this host
     * 
     * @return a map of CPU loads, as percentages, keyed by VM labels
     */
    Map<String, Float> getVMCPULoads();

    /**
     * Creates a new virtual machine on this host. Upon successful creation a
     * notification of type NotificationTypes.VM_NEW will be emitted by the
     * HostMXBean on which the VM has been created.
     * 
     * @param name the name of the virtual machine
     * @param memorySizeMB the initial memory allocation in MBytes
     * @param numVCPU the number of virtual CPUs
     * @param diskSizeMB the disk size in MBytes
     * @param vmImageUUID the uuid of the VM image that will be used to create
     *        the VM
     * @param sync if true the call will return once the VM has been created
     * @return the management interface of the created VM if <tt>sync</tt> is
     *         true, null otherwise
     * @throws InsufficientResourcesException raised if not enough resources are
     *         available to create the VM
     * @throws InvalidVMConfigException raised if the VM configuration is
     *         invalid
     */

    VirtualMachineMXBean createVM(String name, long memorySizeMB, int numVCPU, int diskSizeMB, String vmImageUUID, boolean sync)
        throws InsufficientResourcesException, InvalidVMConfigException, VMMException;

    /**
     * Creates a new virtual machine on this host. Upon successful creation a
     * notification of type NotificationTypes.VM_NEW will be emitted by the
     * HostMXBean on which the VM has been created.
     * 
     * @param vmSpec initial configuration of the virtual machine
     * @param sync if true the call will return once the VM has been created
     * @return the management interface of the created VM if <tt>sync</tt> is
     *         true, null otherwise
     * @throws InsufficientResourcesException raised if not enough resources are
     *         available to create the VM
     * @throws InvalidVMConfigException raised if the VM configuration is
     *         invalid
     */
    VirtualMachineMXBean createVM(VMConfigSpec vmSpec, boolean sync) throws InsufficientResourcesException,
        InvalidVMConfigException, VMMException;

    /**
     * Returns the VM image store associated with this host
     * 
     * @return
     */
    VirtualMachineImageStoreMXBean getVMImageStore();

    /**
     * Configure the reporting of performance metrics as JMX notifications.
     * 
     * @param metricsOfInterest defines the metrics of interest
     * @param periodMillis the period of reporting in milliseconds. If this
     *        value is equal or less than zero, the reporting is disabled
     */
    void configurePerfMonitor(Set<PerfMetric> metricsOfInterest, long periodMillis);
}
